/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-thumbnail.h"
#include "kazehakase.h"

#include <string.h>
#include <glib/gi18n.h>
#include "gtk-utils.h"
#include "egg-pixbuf-thumbnail.h"

/* object class */
typedef struct _KzThumbnailPrivate	KzThumbnailPrivate;
struct _KzThumbnailPrivate
{
	gchar *uri;
	gulong handler_id;
};

#define KZ_THUMBNAIL_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), KZ_TYPE_THUMBNAIL, KzThumbnailPrivate))

static void     dispose      (GObject *object);

G_DEFINE_TYPE(KzThumbnail, kz_thumbnail, GTK_TYPE_EVENT_BOX)

static void
kz_thumbnail_class_init (KzThumbnailClass *klass)
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS(klass);

	object_class->dispose = dispose;

	g_type_class_add_private(object_class, sizeof(KzThumbnailPrivate));
}

static void
kz_thumbnail_init (KzThumbnail *thumbnail)
{
	KzThumbnailPrivate *priv = KZ_THUMBNAIL_GET_PRIVATE(thumbnail);

	priv->uri = NULL;
	priv->handler_id = 0;

	thumbnail->image = gtk_image_new();
	gtk_container_add(GTK_CONTAINER(thumbnail), thumbnail->image);
	gtk_widget_show(thumbnail->image);
}

static void
dispose (GObject *object)
{
	KzThumbnailPrivate *priv = KZ_THUMBNAIL_GET_PRIVATE(object);

	if (priv->uri)
		g_free(priv->uri);
	if (priv->handler_id)
		g_signal_handler_disconnect(kz_app_get(), priv->handler_id);

	priv->uri = NULL;
	priv->handler_id = 0;

	if (G_OBJECT_CLASS (kz_thumbnail_parent_class)->dispose)
		G_OBJECT_CLASS (kz_thumbnail_parent_class)->dispose(object);
}

static void
cb_thumbnail_created (KzApp *app, const gchar *uri, EggPixbufThumbnailSize size,
		      gpointer data)
{
	KzThumbnail *thumbnail = data;
	KzThumbnailPrivate *priv;

	g_return_if_fail(KZ_IS_THUMBNAIL(thumbnail));

	if (size != EGG_PIXBUF_THUMBNAIL_LARGE)
		return;

	priv = KZ_THUMBNAIL_GET_PRIVATE(thumbnail);
	if (!priv->uri)
		return;

	if (strcmp(priv->uri, uri) != 0)
		return;

	kz_thumbnail_set_thumbnail_from_uri(thumbnail, uri);
	gtk_widget_queue_draw(GTK_WIDGET(thumbnail));
}

GtkWidget *
kz_thumbnail_new (void)
{
	KzThumbnail *thumbnail;
	KzThumbnailPrivate *priv;

	thumbnail = g_object_new (KZ_TYPE_THUMBNAIL,
				  "visible-window", FALSE,
				  NULL);

	priv = KZ_THUMBNAIL_GET_PRIVATE(thumbnail);
	priv->handler_id = g_signal_connect(kz_app_get(), "thumbnail-created",
					    G_CALLBACK(cb_thumbnail_created),
					    thumbnail);

	return GTK_WIDGET (thumbnail);
}


void
kz_thumbnail_set_pixbuf (KzThumbnail *thumbnail, GdkPixbuf *pixbuf)
{
	gtk_image_set_from_pixbuf (GTK_IMAGE(thumbnail->image), pixbuf);
}

static void
kz_thumbnail_set_uri (KzThumbnail *thumbnail, const gchar *uri)
{
	KzThumbnailPrivate *priv;

	priv = KZ_THUMBNAIL_GET_PRIVATE(thumbnail);

	if (priv->uri && uri && strcmp(priv->uri, uri) == 0)
		return;

	if (priv->uri)
	    g_free(priv->uri);

	if (uri)
		priv->uri = g_strdup(uri);
	else
		priv->uri = NULL;
}

void
kz_thumbnail_set_thumbnail_from_uri (KzThumbnail *thumbnail,
				     const gchar *uri)
{
	GdkPixbuf *pixbuf = NULL;
	gchar *filename;

	g_return_if_fail(KZ_IS_THUMBNAIL(thumbnail));

	kz_thumbnail_set_uri(thumbnail, uri);

	if (uri && uri[0] != '\0')
	{
		filename = egg_pixbuf_get_thumbnail_filename(uri,
							     EGG_PIXBUF_THUMBNAIL_LARGE);
		pixbuf = egg_pixbuf_get_thumbnail_for_file(filename,
							   EGG_PIXBUF_THUMBNAIL_NORMAL,
							   NULL);
		g_free(filename);
	}

	if (!pixbuf) /* dummy pixbuf */
	{
		gchar *dummy_file_name;
		GdkPixbuf *dummy;

		dummy_file_name = g_build_filename(KZ_GET_SYSTEM_KZ_DATA_DIR,
						   "kz-no-thumbnail.png",
						   NULL);
		dummy = gdk_pixbuf_new_from_file(dummy_file_name, NULL);
		g_free(dummy_file_name);
		pixbuf = egg_pixbuf_get_thumbnail_for_pixbuf(dummy,
							     "no_thumbnail",
							     0,
							     EGG_PIXBUF_THUMBNAIL_NORMAL);
		g_object_unref(dummy);
	}

	gtk_image_set_from_pixbuf (GTK_IMAGE(thumbnail->image), pixbuf);
	g_object_unref(pixbuf);
}

