/*
    gamelauncher.cpp - Kamefu Game(ROM) Launcher

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/
#include "gamelauncher.h"

// KDE includes
#include <kfilemetainfo.h>
#include <kprocess.h>
#include <kconfig.h>
#include <kglobal.h>
#include <kdebug.h>
#include <kfileitem.h>

// Kamefu includes
#include "collectionaccess.h"
#include "rommetadata.h"
#include "rommetadatafactory.h"

namespace Kamefu {

GameLauncher::GameLauncher(QObject *parent, const char *name)
 : QObject(parent, name)
{
}


GameLauncher::~GameLauncher()
{
	kdDebug() << k_funcinfo << endl;
}

bool GameLauncher::launchGame(const QString &romPath)
{
	QString systemName;
	systemName = CollectionAccess::self()->getSystemNameFromRomUrl(romPath);

	if( systemName.isEmpty() )
	{	
		kdDebug() << k_funcinfo << "Guess system name from meta data instead." << endl;

		// Current file isn't in database, use other detection method :)
		// Use RomMetaDataFactory to get ROM meta information (which contain system name)
		KFileItem *guessFileItem = new KFileItem( KFileItem::Unknown, KFileItem::Unknown, KURL(romPath), false);
		RomMetaData *metaDataExtractor = RomMetaDataFactory::self()->createRomMetaData(romPath, guessFileItem->mimetype());
		RomMetaInformation romMetaInfo = metaDataExtractor->extractMetaInformation(guessFileItem);
		
		systemName = romMetaInfo.system();

		delete guessFileItem;
	}
	
	kdDebug() << k_funcinfo << "Retrieved system name: " << systemName << endl;

	if( !systemName.isEmpty() )
	{
		// Read emulator path and arguments from configuration.
		KConfig *config = KGlobal::config();

		// Set configuration group based on system display name.
		config->setGroup( QString("Emulator_%1").arg(systemName) );
		QString emulatorPath = config->readEntry( QString::fromUtf8("EmulatorPath") );
		QString emulatorArgs = config->readEntry( QString::fromUtf8("EmulatorArgs") );

		// Create the emulator process
		KProcess *emulatorProcess = new KProcess;
	
		*emulatorProcess << emulatorPath;

		if( !emulatorArgs.isEmpty() )
		{
			*emulatorProcess << emulatorArgs;
		}
		*emulatorProcess << romPath;
		
		connect(emulatorProcess, SIGNAL(processExited(KProcess* )), this, SLOT(processTerminated(KProcess *)));
		if( !emulatorProcess->start() )
		{
			return false;
		}
		else
		{
			kdDebug() << k_funcinfo << emulatorPath << " " << emulatorArgs << " " << romPath << endl;

			return true;
		}
	}

	return false;
}

void GameLauncher::processTerminated(KProcess *process)
{
	kdDebug() << k_funcinfo << "The emulator exited without problems." << endl;
	process->closeAll();
}

}

#include "gamelauncher.moc"
