/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*
	Author: Micha arok <mishaaq*delete_this*@gmail.com>
	Version: 0.4.3

	Based on Kadu plugin "spy" by Przemysaw Sowa and Wojtek Sulewski
*/

#include "agent.h"
#include "agent_notifications.h"

#include <qfile.h>

#include "debug.h"
#include "chat_manager.h"
#include "message_box.h"
#include "kadu.h"
#include "misc.h"
#include "search.h"
#include "icons_manager.h"
#include "modules/notify/notify.h"


extern "C" int agent_init()
{
	agent = new Agent();
	return 0;
}

extern "C" void agent_close()
{
	delete agent;
	agent = NULL;
}

Agent::Agent() : QObject(NULL, "agent")
{
	kdebugf();

	connect(gadu, SIGNAL(userStatusChangeIgnored(UinType)), this, SLOT(userFound(UinType)));

	menuItemId = kadu->mainMenu()->insertItem(icons_manager->loadIcon("Agent"), tr("Who has me on list"), this, SLOT(resultsRequest()), 0, -1, kadu->mainMenu()->count() - 6);
	icons_manager->registerMenuItem(kadu->mainMenu(), tr("Who has me on list"), "Agent");

	if(config_file.readBoolEntry("Agent", "FirstTime", true))
	{
		QFile listFile;
		listFile.setName(QString(ggPath("spy-unknownslist").ascii()));
		if(listFile.open(IO_ReadOnly))
		{
			if(MessageBox::ask(tr("Agent has founded spy's unknown-users list. Do you want to append this list to agent module?")))
			{
				QTextStream stream(&listFile);

				QString uin_str, date_str, line;
				bool ok;
				while (!stream.atEnd())
				{
					UnknownUser user;
					bool isAlready = false;

					line = stream.readLine();
					uin_str = line.section(',', 0, 0);
					date_str = line.section(',', 1, 1);

					unsigned int uin_int = uin_str.toUInt(&ok, 10);
					if(!ok)
						kdebugm(KDEBUG_PANIC, "Couldn't cast QString to int");

					FOREACH(it, UnknownsList)
					{
						if((*it).uin == uin_int)
						{
							isAlready = true;
							break;
						}
					}
					if(!isAlready)
					{
						user.uin = uin_int;
						user.date = QDate::fromString(date_str, Qt::ISODate);
						user.seq = 0;
						UnknownsList.append(user);
					}
				}
			}
			listFile.close();
		}
		config_file.writeEntry("Agent", "FirstTime", false);
	}
	loadListFromFile();

	notification_manager->registerEvent("Agent/NewFound", QT_TRANSLATE_NOOP("@default", "Agent - new user found"), CallbackRequired);

	kdebugf2();
}

Agent::~Agent()
{
	kdebugf();

	notification_manager->unregisterEvent("Agent/NewFound");

	saveListToFile();

	disconnect(gadu, SIGNAL(userStatusChangeIgnored(UinType)), this, SLOT(userFound(UinType)));
	kadu->mainMenu()->removeItem(menuItemId);

	if(agentWidget != NULL)
	{
		delete agentWidget;
		agentWidget = NULL;
	}

	kdebugf2();
}

void Agent::userFound(UinType uin)
{
	kdebugf();

	FOREACH(it, UnknownsList)
	{
		if((*it).uin == uin)
		{
			(*it).date = QDate::currentDate();
			return;
		}
	}

	UnknownUser newUser;
	newUser.uin = uin;
	newUser.date = QDate::currentDate();

	if(agentWidget != NULL)
	{
		SearchRecord searchUser;
		searchUser.clearData();

		QString uin_str = QString::number(newUser.uin);
		QString date_str = newUser.date.toString(Qt::ISODate);

		searchUser.reqUin(uin_str);
		gadu->searchNextInPubdir(searchUser);
		newUser.seq = searchUser.Seq;

		new QListViewItem(agentWidget->Results, date_str, NULL, uin_str, NULL, NULL, NULL, NULL);
	}
	UnknownsList.append(newUser);

	UserListElement user;
	QString uin_str = QString::number(uin);
	user.addProtocol("Gadu", uin_str);

	UserListElements users(user);

	NewUserFoundNotification *notification = new NewUserFoundNotification(users);
	notification_manager->notify(notification);

	kdebugf2();
}

void Agent::lockupUser(UserListElements users)
{
	kdebugf();

	UserListElement user = *(users.constBegin());
	if(user.usesProtocol("Gadu"))
	{
		SearchDialog *sd = new SearchDialog(kadu, "user_info", user.ID("Gadu").toUInt());
		sd->show();
		sd->firstSearch();
	}

	kdebugf2();
}

void Agent::resultsRequest()
{
	kdebugf();

	if(agentWidget == NULL)
		agentWidget = new AgentWdg;

	agentWidget->show();

	kdebugf2();
}

void Agent::saveListToFile()
{
	kdebugf();

	QFile listFile;
	listFile.setName(QString(ggPath("agent-unknownslist").ascii()));

	listFile.remove();
	if(listFile.open(IO_WriteOnly))
	{
		QTextStream stream(&listFile);
		FOREACH(it, UnknownsList)
		{
			stream << (*it).uin << "," << (*it).date.toString(Qt::ISODate) << endl;
		}
		listFile.close();
	}

	kdebugf2();
}

void Agent::loadListFromFile()
{
	kdebugf();

	QFile listFile;
	listFile.setName(QString(ggPath("agent-unknownslist").ascii()));
	if(listFile.open(IO_ReadOnly))
	{
		QTextStream stream(&listFile);

		QString uin_str, date_str, line;
		bool ok;
		while (!stream.atEnd())
		{
			UnknownUser user;
			
			line = stream.readLine();
			uin_str = line.section(',', 0, 0);
			date_str = line.section(',', 1, 1);

			int uin_int = uin_str.toInt(&ok, 10);
			if(!ok)
				kdebugm(KDEBUG_PANIC, "Couldn't cast QString to int");

			user.uin = uin_int;
			user.date = QDate::fromString(date_str, Qt::ISODate);
			user.seq = 0;

			UnknownsList.append(user);
		}
		listFile.close();
	}

	kdebugf2();
}

void Agent::removeUserFromList(unsigned int uin_int)
{
	FOREACH(it, UnknownsList)
	{
		if((*it).uin == uin_int)
		{
			UnknownsList.remove(it);
			break;
		}
	}
}

AgentWdg::AgentWdg(QWidget * parent, const char * name, WFlags f) : QWidget(NULL, "agentwdg", Qt::WDestructiveClose)
{
	kdebugf();

	vBoxLayout1 = new QVBoxLayout(this, 4, 4, "vblayout1");
	infoLabel = new QLabel(this);
		infoLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
		infoLabel->setText(tr("Warning:\nDetected are users, who has chosen option 'Private status'\nBelow is the list of users, who has you on their lists, but you don't have them on your list"));
		infoLabel->setAlignment(Qt::AlignAuto | Qt::AlignVCenter | Qt::WordBreak);
	vBoxLayout1->addWidget(infoLabel);

	Results = new QListView(this);
		Results->addColumn(tr("Last seen"));
		Results->addColumn(tr("Status"));
		Results->addColumn(tr("Uin"));
		Results->addColumn(tr("Name"));
		Results->addColumn(tr("City"));
		Results->addColumn(tr("Nickname"));
		Results->addColumn(tr("Birth year"));
		Results->setColumnAlignment(2, Qt::AlignHCenter);
		Results->setAllColumnsShowFocus(true);
		Results->setResizeMode(QListView::LastColumn);

	vBoxLayout1->addWidget(Results);

	hBoxLayout1 = new QHBoxLayout(vBoxLayout1, 4, "hblayout1");
	openChat = new QPushButton(icons_manager->loadIconSet("OpenChat"), tr("Open chat"), this);
	addContact = new QPushButton(icons_manager->loadIconSet("AddUser"), tr("Add contact"), this);
	removeFromList = new QPushButton(icons_manager->loadIconSet("DeleteUserlist"), tr("Remove from list"), this);

	hBoxLayout1->addWidget(removeFromList);
	hBoxLayout1->addWidget(addContact);
	hBoxLayout1->addWidget(openChat);

	connect(gadu, SIGNAL(newSearchResults(SearchResults &, int, int)), this, SLOT(pubdirResults(SearchResults &, int, int)));
	connect(removeFromList, SIGNAL(clicked()), this, SLOT(removeUser()));
	connect(openChat, SIGNAL(clicked()), this, SLOT(startChat()));
	connect(addContact, SIGNAL(clicked()), this, SLOT(addUser()));
	connect(Results, SIGNAL(selectionChanged()), this, SLOT(userlistSelectionChanged()));


	removeFromList->setEnabled(false);
	openChat->setEnabled(false);
	addContact->setEnabled(false);

	loadGeometry(this, "Agent", "UsersFoundWidgetGeometry", 0, 0, 150, 70);

	FOREACH(it, agent->UnknownsList)
	{
		SearchRecord searchUser;
		searchUser.clearData();

		QString uin_str = QString::number((*it).uin);
		QString date_str = (*it).date.toString(Qt::ISODate);

		searchUser.reqUin(uin_str);
		gadu->searchNextInPubdir(searchUser);
		(*it).seq = searchUser.Seq;

		new QListViewItem(Results, date_str, NULL, uin_str, NULL, NULL, NULL, NULL);
	}

	this->adjustSize();
	this->setCaption(tr("Who has me on list?"));

	kdebugf2();
};

AgentWdg::~AgentWdg()
{
	kdebugf();

	saveGeometry(this, "Agent", "UsersFoundWidgetGeometry");

	disconnect(gadu, SIGNAL(newSearchResults(SearchResults &, int, int)), this, SLOT(pubdirResults(SearchResults &, int, int)));
	disconnect(removeFromList, SIGNAL(clicked()), this, SLOT(removeUser()));
	disconnect(openChat, SIGNAL(clicked()), this, SLOT(startChat()));
	disconnect(addContact, SIGNAL(clicked()), this, SLOT(addUser()));
	disconnect(Results, SIGNAL(selectionChanged()), this, SLOT(userlistSelectionChanged()));

	delete Results;
	delete openChat;
	delete addContact;
	delete removeFromList;
	delete infoLabel;
	delete hBoxLayout1;
	delete vBoxLayout1;

	agentWidget = NULL;

	kdebugf2();
}

void AgentWdg::userlistSelectionChanged()
{
	if(Results->selectedItem() == NULL)
	{
		removeFromList->setEnabled(false);
		openChat->setEnabled(false);
		addContact->setEnabled(false);
	}
	else
	{
		removeFromList->setEnabled(true);
		openChat->setEnabled(true);
		addContact->setEnabled(true);
	}
}

void AgentWdg::pubdirResults(SearchResults& searchResults, int seq, int fromUin)
{
	kdebugf();

	QListViewItem *qlv = NULL;
	QPixmap pix;

	FOREACH(it, agent->UnknownsList)
	{
		if((*it).seq == seq)
		{
			CONST_FOREACH(searchIterator, searchResults)
			{
				qlv = Results->findItem((*searchIterator).Uin, 2);

				pix = ((*searchIterator).Stat).pixmap((*searchIterator).Stat.status(), false, false);

				if(qlv)
				{
					qlv->setPixmap(1, pix);
					qlv->setText(3, (*searchIterator).First);
					qlv->setText(4, (*searchIterator).City);
					qlv->setText(5, (*searchIterator).Nick);
					qlv->setText(6, (*searchIterator).Born);
				}
			}
		}
	}

	kdebugf2();
}

void AgentWdg::removeUser()
{
	kdebugf();

	QListViewItem *selectedEntry = Results->selectedItem();
	if(Results->childCount() == 1)
		selectedEntry = Results->firstChild();

	bool ok;
	unsigned int uin_int = selectedEntry->text(2).toInt(&ok, 10);
	if(ok)
	{
		agent->removeUserFromList(uin_int);
	}

	delete selectedEntry;

	kdebugf2();
}

void AgentWdg::addUser()
{
	kdebugf();

	QListViewItem *selectedEntry = Results->selectedItem();
	if(Results->childCount() == 1)
		selectedEntry = Results->firstChild();

	QString uin_str = selectedEntry->text(2);
	QString name = selectedEntry->text(3);
	QString nickname = selectedEntry->text(5);

	QString altnick = nickname;

	if (!altnick.length())
	{
		altnick = name;
	}

	if (!altnick.length())
		altnick = uin_str;

	UserListElement e;
	bool ok;
	e.setFirstName(name);
	e.setLastName("");
	e.setNickName(nickname);
	e.setAltNick(altnick);

	UinType uin = uin_str.toUInt(&ok);
	if(!ok)
		uin = 0;

	if(uin)
		e.addProtocol("Gadu", QString::number(uin));

	e.setEmail("");
	
	UserInfo *userinfo = new UserInfo(e, this, "user info");
	connect(userinfo, SIGNAL(updateClicked(UserInfo *)), this, SLOT(userInfoAccepted(UserInfo *)));
	userinfo->show();

	kdebugf2();
}

void AgentWdg::userInfoAccepted(UserInfo *info)
{
	UserListElement new_user = info->user();
	QString uin_str = new_user.ID("Gadu");

	bool ok;
	int uin_int = uin_str.toUInt(&ok);
	if(!ok)
		return;

	agent->removeUserFromList(uin_int);

	QListViewItem *qlv = NULL;
	qlv = Results->findItem(uin_str, 2);
	if(qlv)
		Results->takeItem(qlv);

	disconnect(info, SIGNAL(updateClicked(UserInfo *)), this, SLOT(userInfoAccepted(UserInfo *)));
}

void AgentWdg::startChat()
{
	kdebugf();

	QListViewItem *selectedEntry = Results->selectedItem();
	if(Results->childCount() == 1)
		selectedEntry = Results->firstChild();

	UinType uin = selectedEntry->text(2).toUInt();
	UserListElements users(userlist->byID("Gadu", QString::number(uin)));
	
	chat_manager->openPendingMsgs(users);
	
	kdebugf2();
}

AgentWdg* agentWidget;
Agent* agent;
