#ifndef KADU_GROUPS_MANAGER_H
#define KADU_GROUPS_MANAGER_H

#include <qmap.h>
#include <qobject.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qtimer.h>
#include <qvariant.h>
#include <qwmatrix.h>

#include "configuration_aware_object.h"
#include "usergroup.h"
#include "userlistelement.h"

class KaduTabBar;

/**
	Klasa suca przede wszystkim zarzdzaniu nazwanymi grupami kontaktw,
	ale take zarzdza grupami typu: blokowani, blokujcy, tylko z opisami, tylko dostpni, anonimowi
	\class GroupsManager
	\brief
**/
class GroupsManager : public QObject, ConfigurationAwareObject
{
	Q_OBJECT

	GroupsManager();
	~GroupsManager();
	QMap<QString, UserGroup *> Groups;
	KaduTabBar *GroupBar;
	int lastId;
	QString currentGroup;
	bool showBlocked;
	bool showBlocking;
	bool showOffline;
	bool showWithoutDescription;
	bool showOnlineAndDescription;
	QTimer refreshTimer;

private slots:
	void userDataChanged(UserListElement elem, QString name, QVariant oldValue, QVariant currentValue, bool massively, bool last);
	void userAddedToMainUserlist(UserListElement elem, bool massively, bool last);
	void userRemovedFromMainUserlist(UserListElement elem, bool massively, bool last);

	void userAdded(UserListElement elem, bool massively, bool last);
	void userRemoved(UserListElement elem, bool massively, bool last);

	void tabSelected(int id);
	void iconThemeChanged();

protected:
	virtual void configurationUpdated();

public:
	/**
		\fn static void initModule()
		inicjuje modu zarzdcy grup
	**/
	static void initModule();

	/**
		\fn static void closeModule()
		sprzta po module zarzdcy grup
	**/
	static void closeModule();

	/**
		\fn UserGroup *group(const QString &name) const
		zwraca grup kontaktw o nazwie name
		\param name nazwa grupy
		\return grupa kontaktw
	**/
	UserGroup *group(const QString &name) const;

	/**
		\fn UserGroup *addGroup(const QString &name)
		dodaje now grup kontaktw o nazwie name
		\param name nazwa grupy
		\return grupa kontaktw
	**/
	UserGroup *addGroup(const QString &name);

	/**
		\fn void removeGroup(const QString &name)
		usuwa grup kontaktw o nazwie name
		\param name nazwa grupy
	**/
	void removeGroup(const QString &name);

	/**
		\fn void setTabBar(KaduTabBar *bar)
		ustawia pas zakadek, na ktrym klasa bdzie operowa oraz inicjuje
		wewntrzne dane klasy
		\param bar pas zakadek, w ktrym bd umieszczane zakadki grup
	**/
	void setTabBar(KaduTabBar *bar);

	/**
		\fn QStringList groups() const
		\return lista grup
	**/
	QStringList groups() const;

	/**
		\fn bool groupExists(const QString &name)
		\param name nazwa grupy
		\return informacja czy grupa istnieje
	**/
	bool groupExists(const QString &name);

	/**
		\fn QString currentGroupName() const
		\return nazwa aktualnie wybranej grupy
	**/
	QString currentGroupName() const;

	/**
		\fn void setIconForTab(const QString &name, bool showIcon)
		ustawia lub usuwa ikonk dla grupy o nazwie name
		\param name nazwa grupy
	**/
	void setIconForTab(const QString &name);

public slots:

	/**
		\fn void setActiveGroup(const QString &name)
		ustawia aktywn grup na name
		\param name nazwa grupy
	**/
	void setActiveGroup(const QString& group);

	/**
		\fn void refreshTabBar()
		odwiea pasek zakadek grup
	**/
	void refreshTabBar();

	/**
		\fn void refreshTabBarLater()
		odwiea pasek zakadek grup, ale po zakoczeniu bieacych operacji
	**/
	inline void refreshTabBarLater() { refreshTimer.start(0, true); }

	/**
		\fn void changeDisplayingBlocking()
		wcza lub wycza wywietlanie kontaktw blokujcych
		w gwnej licie kontaktw w zalenoci od poprzedniego stanu
	**/
	void changeDisplayingBlocking();

	/**
		\fn void changeDisplayingBlocked()
		wcza lub wycza wywietlanie kontaktw blokowanych
		w gwnej licie kontaktw w zalenoci od poprzedniego stanu
	**/
	void changeDisplayingBlocked();

	/**
		\fn void changeDisplayingOffline()
		wcza lub wycza wywietlanie kontaktw niedostpnych
		w gwnej licie kontaktw w zalenoci od poprzedniego stanu
	**/
	void changeDisplayingOffline();

	/**
		\fn void changeDisplayingWithoutDescription()
		wcza lub wycza wywietlanie kontaktw o statusach bez opisu
		w gwnej licie kontaktw w zalenoci od poprzedniego stanu
	**/
	void changeDisplayingWithoutDescription();

	/**
		\fn void changeDisplayingOnlineAndDescription()
		wcza lub wycza wywietlanie kontatkw o statusach "dostpny"
		lub "zajty" oraz kontatkw z opisem
	**/
	void changeDisplayingOnlineAndDescription();

};

extern GroupsManager *groups_manager;

/**
	\class UsersWithDescription
	Klasa grupujca kontakty o statusie z opisem,
	automatycznie synchronizujca si z gwn list kontaktw
**/
class UsersWithDescription : public UserGroup
{
	Q_OBJECT
	public:
		UsersWithDescription();
		virtual ~UsersWithDescription();
	private slots:
		void statusChangedSlot(UserListElement elem, QString protocolName,
							const UserStatus &oldStatus, bool massively, bool last);
};
extern UsersWithDescription *usersWithDescription;

/**
	\class OnlineUsers
	Klasa grupujca kontakty o statusie "dostpny" lub "zajty",
	automatycznie synchronizujca si z gwn list kontaktw
**/
class OnlineUsers : public UserGroup
{
	Q_OBJECT
	public:
		OnlineUsers();
		virtual ~OnlineUsers();
	private slots:
		void statusChangedSlot(UserListElement elem, QString protocolName,
							const UserStatus &oldStatus, bool massively, bool last);
};
extern OnlineUsers *onlineUsers;

/**
	\class OnlineAndDescriptionUsers
	Klasa grupujca kontakty o statusie "dostpny",
	lub "zajty", oraz kontakty majca opis. Klasa
	automatycznie synchronizuje si z gwn list kontaktw
**/
class OnlineAndDescriptionUsers : public UserGroup
{
	Q_OBJECT

	public:
		OnlineAndDescriptionUsers();
		virtual ~OnlineAndDescriptionUsers();

	private slots:
		void statusChangedSlot(UserListElement elem, QString protocolName,
					    const UserStatus &oldStatus, bool massively, bool last);
		void userChangedSlot(UserListElement elem, bool massively, bool last);
		void protocolAddedOrRemoved(UserListElement elem, QString protocolName, bool massively, bool last);
};
extern OnlineAndDescriptionUsers *onlineAndDescriptionUsers;

/**
	\class OfflineUsers
	Klasa grupujca kontakty o statusie "niedostpny",
	automatycznie synchronizujca si z gwn list kontaktw
**/
class OfflineUsers : public UserGroup
{
	Q_OBJECT
	public:
		OfflineUsers();
		virtual ~OfflineUsers();
	private slots:
		void statusChangedSlot(UserListElement elem, QString protocolName,
					    const UserStatus &oldStatus, bool massively, bool last);
		void userChangedSlot(UserListElement elem, bool massively, bool last);
		void protocolAddedOrRemoved(UserListElement elem, QString protocolName, bool massively, bool last);
};
extern OfflineUsers *offlineUsers;

/**
	\class BlockedUsers
	Klasa grupujca kontakty, ktre s blokowane,
	automatycznie synchronizujca si z gwn list kontaktw
**/
class BlockedUsers : public UserGroup
{
	Q_OBJECT
	public:
		BlockedUsers();
		virtual ~BlockedUsers();
	private slots:
		void protocolUserDataChangedSlot(QString protocolName, UserListElement elem,
							QString name, QVariant oldValue, QVariant currentValue,
							bool massively, bool last);
};
extern BlockedUsers *blockedUsers;

/**
	\class BlockingUsers
	Klasa grupujca kontakty o statusie blokujcym,
	automatycznie synchronizujca si z gwn list kontaktw
**/
class BlockingUsers : public UserGroup
{
	Q_OBJECT
	public:
		BlockingUsers();
		virtual ~BlockingUsers();
	private slots:
		void statusChangedSlot(UserListElement elem, QString protocolName,
							const UserStatus &oldStatus, bool massively, bool last);
};
extern BlockingUsers *blockingUsers;

/**
	\class AnonymousUsers
	Klasa grupujca kontakty anonimowe,
	automatycznie synchronizujca si z gwn list kontaktw
**/
class AnonymousUsers : public UserGroup
{
	Q_OBJECT
	public:
		AnonymousUsers();
		virtual ~AnonymousUsers();
	private slots:
		void userDataChangedSlot(UserListElement elem,
							QString name, QVariant oldValue, QVariant currentValue,
							bool massively, bool last);
		void userAdded(UserListElement elem, bool massively, bool last);
		void removingUser(UserListElement elem, bool massively, bool last);
};
extern AnonymousUsers *anonymousUsers;

/**
	\class AnonymousUsersWithoutMessages
	Klasa grupujca kontakty anonimowe, nie majce wiadomoci do odebrania
	automatycznie synchronizujca si z gwn list kontaktw
**/
class AnonymousUsersWithoutMessages : public UserGroup
{
	Q_OBJECT
	public:
		AnonymousUsersWithoutMessages();
		virtual ~AnonymousUsersWithoutMessages();
	private slots:
		void messageFromUserAdded(UserListElement elem);
		void messageFromUserDeleted(UserListElement elem);

		void userDataChangedSlot(UserListElement elem,
							QString name, QVariant oldValue, QVariant currentValue,
							bool massively, bool last);
		void userAdded(UserListElement elem, bool massively, bool last);
		void removingUser(UserListElement elem, bool massively, bool last);
};
extern AnonymousUsersWithoutMessages *anonymousUsersWithoutMessages;

#endif
