#ifndef KADU_CHAT_WIDGET_H
#define KADU_CHAT_WIDGET_H

#include <qdatetime.h>
#include <qwidget.h>
#include <qhbox.h>


#include <qvaluelist.h>

#include "chat_messages_view.h"
#include "config_file.h"
#include "configuration_aware_object.h"
#include "custom_input.h"
#include "emoticons.h"
#include "protocol.h"
#include "usergroup.h"
#include "toolbar.h"

class QAccel;
class QHBox;
class QMimeSourceFactory;
class QObject;
class QPushButton;

class ChatWidget;
class ChatManager;
class ChatMessage;
class ColorSelector;
class EmoticonSelector;
class KaduSplitter;
class KaduTextBrowser;
class UserBox;

/** \typedef QValueList<ChatWidget*> ChatList */
typedef QValueList<ChatWidget*> ChatList;

/**
	Klasa tworzca okno rozmowy, rejestrujca przyciski,
	formatujca wiadomo itp.
	\class ChatWidget
	\brief Okno rozmowy
**/

class ChatWidget : public QHBox, ConfigurationAwareObject
{
	Q_OBJECT

private:
	friend class ChatManager;

	QString Caption; /*!< tytu okna */
	Protocol *CurrentProtocol;
	UserGroup *Users; /*!< lista uytkownikw w danym oknie */
	int index;	/*!< nr okna (z chat menadera) */
	QColor actcolor; /*!< zmienna przechowujca aktualny kolor */
	QPixmap pix;
	CustomInput* Edit; /*!< okno do wpisywania wiadomoci */
	QMimeSourceFactory *bodyformat; /*!< zmienna ustawiajca format */
	EmoticonSelector* emoticon_selector; /*!< okienko do wyboru emotikonek */
	ColorSelector* color_selector; /*!< okienko do wyboru koloru */
	bool AutoSend; /*!< okrela czy Return powoduje wysanie wiadomoci */
	bool WaitingForACK;
	UserBox* userbox; /*!< lista kontaktw przydatna gdy jestemy w konferencji */
	QString myLastMessage; /*!< zmienna przechowujca nasz ostatni wiadomo */
	KaduSplitter *vertSplit, *horizSplit; /*!< obiekty oddzielajce kontrolki od siebie */

	QDateTime lastMsgTime; /*!< czas ostatniej wiadomoci */

	ChatMessagesView* body; /*!< historia rozmowy, prosz NIE dawa dostpu na zewntrz do tej zmiennej */

	// TODO: remove
	int activationCount;

	unsigned int NewMessagesCount; /*!< liczba nowych wiadomoci w oknie rozmowy */

private slots:
	/**
		\fn void changeColor()
		Slot zmieniajcy kolor czcionki
	**/
	void changeColor(const QWidget* activating_widget);

	/**
		\fn void connectAcknowledgeSlots()
		Slot podczajcy potwierdzenie dostarczenia wiadomoci
	**/
	void connectAcknowledgeSlots();

	/**
		\fn void disconnectAcknowledgeSlots()
		Slot odczajcy potwierdzenie dostarczenia wiadomoci
	**/
	void disconnectAcknowledgeSlots();

	void selectedUsersNeeded(const UserGroup*& user_group);

	void sendActionAddedToToolbar(ToolButton*, ToolBar*);

	/**
		\fn void colorActionForceRefresh()
		Slot jest wywoywany, po dodaniu do paska narzdzi akcji zmieniajcej
		kolor wiadomoci lub zmianie zestawu ikon. Uywa setActColor() wymuszajc
		odwieenie koloru na przycisku.
	**/
	void colorActionForceRefresh();

	/**
		\fn void setActColor()
		\param force wymuszenie zmiany
		Ustawia poprawny kolor na przycisku akcji obsugujcej kolor wiadomoci
	**/
	void setActColor(bool force);

	void keyPressedSlot(QKeyEvent *e, CustomInput *sender, bool &handled);

protected:
	virtual void keyPressEvent(QKeyEvent* e);
	bool keyPressEventHandled(QKeyEvent *);

	virtual void configurationUpdated();

public:
	/**
		Konstruktor okna rozmowy
		\fn Chat(UserListElements usrs, QWidget* parent = 0, const char* name = 0)
		\param usrs lista kontaktw, z ktrymi prowadzona jest rozmowa
		\param parent rodzic okna
		\param name nazwa obiektu
	**/
	ChatWidget(Protocol *initialProtocol, const UserListElements &usrs, QWidget* parent = 0, const char* name = 0);

	/**
		\fn ~Chat()
		Destruktor okna rozmowy
	**/
	~ChatWidget();

	/**
		Dodaje now wiadomos systemow do okna.

		@param rawContent tre wiadomoci w postaci HTML
		@param backgroundColor kolor ta wiadomoci (format HTML)
		@param fontColor kolor wiadomoci (format HTML)
	 **/
	void appendSystemMessage(const QString &rawContent, const QString &backgroundColor, const QString &fontColor);

	/**
		\fn void newMessage(const QString &protocolName, UserListElements senders, const QString &msg, time_t time)
		Dodaje wiadomo do okna
		\param protocolName nazwa protokou
		\param senders lista uytkownikw
		\param msg wiadomoc
		\param time czas
		**/
	void newMessage(const QString &protocolName, UserListElements senders, const QString &msg, time_t time);

	/**
		\fn const UserGroup *users() const
		Zwraca list numerw rozmowcw.
	**/
	const UserGroup *users() const;

	/**
		\fn void repaintMessages()
		Odwiea zawarto okna uwzgldniajc ewentualne
		zmiany dokonane w ktrej wiadomoci z listy
		uzyskanej za pomoc metody chatMessages(),
		dodanie nowych wiadomoci lub usunicie istniejcych.
	**/
	void repaintMessages();

	/**
		\fn const QString& title() const
		Zwraca aktualny tytu okna
	**/
	const QString& caption() const;

	/**
		\fn CustonInput* edit()
		Zwraca wskanik do okna wpisywania wiadomoci
	**/
	CustomInput* edit();

	/**
		\fn bool autoSend() const
		Zwraca warto okrelajc, czy Return powoduje
		wysanie wiadomoci.
	**/
	bool autoSend() const;

	bool waitingForACK() const;

	virtual void dragEnterEvent(QDragEnterEvent *e);
	virtual void dropEvent(QDropEvent *e);
	virtual void dragMoveEvent(QDragMoveEvent *e);

	virtual bool eventFilter(QObject *watched, QEvent *e);

	Protocol *currentProtocol();

	unsigned int newMessagesCount() const;

	QDateTime getLastMsgTime();
	QPixmap icon();

	void storeGeometry();
	void restoreGeometry();

	unsigned int countMessages() { return body->countMessages(); }

public slots:
	/**
		\fn void refreshTitle()
		Slot ustawiajcy tytu okna zgodnie z konfiguracj
	**/
	void refreshTitle();

	/**
		\fn void addEmoticon(QString)
		Slot dodaje ikonke do pola wpisywania wiadomoci
		\param emot ikonka np. <lol>
	**/
	void addEmoticon(QString);

	void messageNotDeliveredSlot(const QString &message);

	void messageAcceptedSlot();

	/**
		\fn void editTextChanged()
		Slot jest wywoywany, gdy zmienia si zawarto
		pola edycji wiadomoci. Ustawia poprawny stan przycisku
		akcji sendAction w oknie rozmowy.
	**/
	void editTextChanged();

	/**
		\fn void appendMessages(const QValueList<ChatMessage *> &)
		Slot dodaj wiadomoci do okna
		\param messages lista wiadomoci
	**/
	void appendMessages(const QValueList<ChatMessage *> &, bool pending = false);

	/**
		\fn void appendMessage(ChatMessage *)
		Slot dodaj wiadomo do okna
		\param messages lista wiadomoci
	**/
	void appendMessage(ChatMessage *, bool pending = false);

	/**
		\fn void sendMessage()
		Slot wywoywany po naciniciu przycisku
		do wysyania wiadomoci
	**/
	void sendMessage();

	/**
		\fn void cancelMessage()
		Slot wywoywany po nacinieciu przycisku anulowania
		wysyaniu wiadomoci
	**/
	void cancelMessage();

	/**
		\fn void writeMyMessage()
		Slot wpisujcy wiadomo do okna
		\see sendMessage
	**/
	void writeMyMessage();

	/**
		\fn void curPosChanged(int para, int pos)
		Slot wywoywany kiedy pozycja kursosa zmienia si
		\param  para nie obslugiwane
		\param pos nie obslugiwane
	**/
	void curPosChanged(int para, int pos);

	/**
		\fn void specialKeyPressed(int key)
		Slot zostaje wywoany gdy nacinito specjalny klawisz (skrt)
		odpowiadajcy np KEY_BOLD
		\param key warto nacinitego specjalnego klawisza
	**/
	void specialKeyPressed(int key);

	/**
		\fn void colorChanged(const QColor& color)
		Slot zostaje wywowany gdy uytkownik zmieni kolor czcionki
		ktr bdzie pisa
		\param color warto zmienionego koloru
	**/
	void colorChanged(const QColor& color);

	/**
		\fn void colorSelectorAboutToClose()
		Slot zostaje wywoany przy zamykaniu okna wyboru ikonek
	**/
	void colorSelectorAboutToClose();

	/**
		\fn void clearChatWindow()
		Slot czyszczcy okno rozmowy
	**/
	void clearChatWindow();

	/**
		\fn void setAutoSend(bool auto_send)
		Ustaw autowysylanie
	**/
	void setAutoSend(bool auto_send);

	/**
		\fn void emoticonSelectorClicked()
		Slot wywietlajcy okno z wyborem emotikonek
	**/
	void openEmoticonSelector(const QWidget* activating_widget);

	/**
		\fn void insertImage()
		Slot wklejajcy obrazek do okna rozmowy (wybr obrazka)
	**/
	void insertImage();

	void makeActive();
	void markAllMessagesRead();

	void leaveConference();

signals:
	/**
		\fn void messageSendRequested(ChatWidget* chat)
		Sygnal jest emitowany gdy uzytkownik wyda polecenie
		wyslania wiadomosci, np klikajac na guzik "wyslij".
		\param chat wskanik do okna ktre emituje sygna
	**/
	void messageSendRequested(ChatWidget* chat);

	/**
		\fn void messageSent(ChatWidget* chat)
		Sygnal jest emitowany gdy zakonczy sie proces
		wysylania wiadomosci i zwiazanych z tym czynnosci.
		Oczywiscie nie wiemy czy wiadomosc dotarla.
		\param chat wskanik do okna rozmowy,
		 ktre emitowao sygna
	**/
	void messageSent(ChatWidget* chat);

	/**
		\fn void messageSentAndConfirmed(UserListElements receivers, const QString& message)
		This signal is emited when message was sent
		and it was confirmed.
		When confirmations are turned off signal is
		emited immediately after message was send
		just like messageSent() signal.
		\param receivers list of receivers
		\param message the message
	**/
	void messageSentAndConfirmed(UserListElements receivers, const QString& message);

	/**
		\fn void fileDropped(const UserGroup *users, const QString& fileName)
		Sygna jest emitowany, gdy w oknie Chat
		upuszczono plik.
	\param users lista uytkownikw
		\param fileName nazwa pliku
	**/
	void fileDropped(const UserGroup *users, const QString &fileName);

	void messageReceived(ChatWidget *);

	void captionUpdated();
	void closed();

	void keyPressed(QKeyEvent *e, ChatWidget *sender, bool &handled);

};

/**
	@class ChatContainer
	@brief Klasa abstrakcyjna opisujca rodzica klasy ChatWidget.

	Klasa abstrakcyjna z ktrej powinny dziedziczy klasy bdce rodzicami obiektw 
	klasy ChatWidget.
	
	Informuje ktry chat powinien zosta zamknity w przypadku np. ignorowania kontaktu 
	z ktrym prowadzona jest rozmowa
**/

class ChatContainer
{
	public:
		ChatContainer() {};
		virtual ~ChatContainer() {};

		/**
			Metoda informujca, ktry chat powinien zosta zamknity
		 **/
		virtual void closeChatWidget(ChatWidget *chat) = 0;
};

#endif
