# Copyright (C) 2007 Samuel Abels <http://debain.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2, as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
import datetime

def my_weekdatescalendar(cal, *date):
    weeks = cal.monthdatescalendar(*date[:2])
    for weekno, week in enumerate(weeks):
        # Hide all days that are not part of the current week.
        weekdays = [d.timetuple()[:3] for d in week]
        if date[:3] in weekdays:
            return week
    raise Exception('No such week')


def my_monthdatescalendar(cal, *args):
    # Months that have only five weeks are filled with another week from
    # the following month.
    weeks = cal.monthdatescalendar(*args[:2])
    if len(weeks) < 6:
        last_day = weeks[-1][-1]
        offset   = datetime.timedelta(1)
        week     = []
        for i in range(7):
            last_day += offset
            week.append(last_day)
        weeks.append(week)
    return weeks


def previous_month(cal, date):
    year, month, day = date.timetuple()[:3]
    if month == 1:
        year  -= 1
        month  = 12
    else:
        month -= 1
    prev_month_days = [d for d in cal.itermonthdays(year, month)]
    if day not in prev_month_days:
        day = max(prev_month_days)
    return datetime.datetime(year, month, day)


def next_month(cal, date):
    year, month, day = date.timetuple()[:3]
    if month == 12:
        year  += 1
        month  = 1
    else:
        month += 1
    next_month_days = [d for d in cal.itermonthdays(year, month)]
    if day not in next_month_days:
        day = max(next_month_days)
    return datetime.datetime(year, month, day)


def previous_week(cal, date):
    return date - datetime.timedelta(7)


def next_week(cal, date):
    return date + datetime.timedelta(7)


def time_delta(datetime1, datetime2):
    delta = datetime1 - datetime2
    if delta < datetime.timedelta():
        return -delta
    return delta


def event_days(event1, event2):
    return time_delta(event1.start, event1.end).days \
         - time_delta(event2.start, event2.end).days


def event_intersects(event, start, end = None):
    if end is None:
        end = start
    return (event.start >= start and event.start < end) \
        or (event.end > start and event.end <= end) \
        or (event.start < start and event.end > end)


def get_intersection_list(list, start, end):
    intersections = []
    for event in list:
        if event_intersects(event, start, end):
            intersections.append(event)
    return intersections


def count_intersections(list, start, end):
    intersections = 0
    for event in list:
        if event_intersects(event, start, end):
            intersections += 1
    return intersections


def count_parallel_events(list, start, end):
    """
    Given a list of events, this function returns the maximum number of
    parallel events in the given timeframe.
    """
    parallel = 0
    i        = 0
    for i, event1 in enumerate(list):
        if not event_intersects(event1, start, end):
            continue
        parallel = max(parallel, 1)
        for f in range(i + 1, len(list)):
            event2    = list[f]
            new_start = max(event1.start, event2.start)
            new_end   = min(event1.end,   event2.end)
            if event_intersects(event2, start, end) \
                and event_intersects(event2, new_start, new_end):
                n = count_parallel_events(list[f:], new_start, new_end)
                parallel = max(parallel, n + 1)
    return parallel


def color_to_string(color): 
    hexstring = "#%02X%02X%02X" % (color.red   / 256, 
                                   color.blue  / 256, 
                                   color.green / 256)
    return hexstring 


class Calendar(object):
    """
    Base calendar class. This class doesn't do any formatting. It simply
    provides data to subclasses.
    """

    def __init__(self, firstweekday=0):
        self.firstweekday = firstweekday # 0 = Monday, 6 = Sunday

    def getfirstweekday(self):
        return self._firstweekday % 7

    def setfirstweekday(self, firstweekday):
        self._firstweekday = firstweekday

    firstweekday = property(getfirstweekday, setfirstweekday)

    def iterweekdays(self):
        """
        Return a iterator for one week of weekday numbers starting with the
        configured first one.
        """
        for i in xrange(self.firstweekday, self.firstweekday + 7):
            yield i%7

    def itermonthdates(self, year, month):
        """
        Return an iterator for one month. The iterator will yield datetime.date
        values and will always iterate through complete weeks, so it will yield
        dates outside the specified month.
        """
        date = datetime.date(year, month, 1)
        # Go back to the beginning of the week
        days = (date.weekday() - self.firstweekday) % 7
        date -= datetime.timedelta(days=days)
        oneday = datetime.timedelta(days=1)
        while True:
            yield date
            date += oneday
            if date.month != month and date.weekday() == self.firstweekday:
                break

    def itermonthdays2(self, year, month):
        """
        Like itermonthdates(), but will yield (day number, weekday number)
        tuples. For days outside the specified month the day number is 0.
        """
        for date in self.itermonthdates(year, month):
            if date.month != month:
                yield (0, date.weekday())
            else:
                yield (date.day, date.weekday())

    def itermonthdays(self, year, month):
        """
        Like itermonthdates(), but will yield day numbers tuples. For days
        outside the specified month the day number is 0.
        """
        for date in self.itermonthdates(year, month):
            if date.month != month:
                yield 0
            else:
                yield date.day

    def monthdatescalendar(self, year, month):
        """
        Return a matrix (list of lists) representing a month's calendar.
        Each row represents a week; week entries are datetime.date values.
        """
        dates = list(self.itermonthdates(year, month))
        return [ dates[i:i+7] for i in xrange(0, len(dates), 7) ]

    def monthdays2calendar(self, year, month):
        """
        Return a matrix representing a month's calendar.
        Each row represents a week; week entries are
        (day number, weekday number) tuples. Day numbers outside this month
        are zero.
        """
        days = list(self.itermonthdays2(year, month))
        return [ days[i:i+7] for i in xrange(0, len(days), 7) ]

    def monthdayscalendar(self, year, month):
        """
        Return a matrix representing a month's calendar.
        Each row represents a week; days outside this month are zero.
        """
        days = list(self.itermonthdays(year, month))
        return [ days[i:i+7] for i in xrange(0, len(days), 7) ]

    def yeardatescalendar(self, year, width=3):
        """
        Return the data for the specified year ready for formatting. The return
        value is a list of month rows. Each month row contains upto width months.
        Each month contains between 4 and 6 weeks and each week contains 1-7
        days. Days are datetime.date objects.
        """
        months = [
            self.monthdatescalendar(year, i)
            for i in xrange(January, January+12)
        ]
        return [months[i:i+width] for i in xrange(0, len(months), width) ]

    def yeardays2calendar(self, year, width=3):
        """
        Return the data for the specified year ready for formatting (similar to
        yeardatescalendar()). Entries in the week lists are
        (day number, weekday number) tuples. Day numbers outside this month are
        zero.
        """
        months = [
            self.monthdays2calendar(year, i)
            for i in xrange(January, January+12)
        ]
        return [months[i:i+width] for i in xrange(0, len(months), width) ]

    def yeardayscalendar(self, year, width=3):
        """
        Return the data for the specified year ready for formatting (similar to
        yeardatescalendar()). Entries in the week lists are day numbers.
        Day numbers outside this month are zero.
        """
        months = [
            self.monthdayscalendar(year, i)
            for i in xrange(January, January+12)
        ]
        return [months[i:i+width] for i in xrange(0, len(months), width) ]
