#ifndef HEAP_DUMP_H__
#define HEAP_DUMP_H__

#include <jni.h>
#include <jvmpi.h>
#include <cls.h>

/** Handle the heap and monitor dumps. */

/** or these values together to get the four (4) combinations possible.. */
#define STATIC_VARIABLE 0
#define VARIABLE 1
#define ARRAY 2

/** link from object to owner of object. */
typedef struct object_link object_link;


struct object_link {
    jobjectID  obj;    /** The object in question. */
    jobjectID  parent; /** The parent/owner of obj. */
    jobjectID  clz;    /** The class of the parent. 
			   Needed to find right variable. */
    object_link* next; /** The next owner of of this obj. */ 
    int        type;   /** Type of link. Or value of VARIABLE and ARRAY */    
    jint       variable; /** index in field array for class. */
    jint       index;    /** array index. */
};

/** down-link from an object to the instances in the object. */
typedef struct down_link down_link;

struct down_link {
    unsigned char type;
    field* fld;
    jint   pos;        /** position in array */
    union {
	unsigned char u1;
	jshort u2;
	jint   u4;
	jlong  u8;
	jobjectID o;
	unsigned char* txt;   // text data for primitive arrays
    } value;
    down_link* next;
};

typedef struct monitor_info monitor_info;

struct monitor_info {
    char* name;
    void* id;
    jmpthread* owner;
    jint entry_count;
    jint num_waiting_to_enter;
    jmpthread** waiting_to_enter;
    jint num_waiting_for_notify;
    jmpthread** waiting_for_notify;
};

/** Free a monitor info structure. */
void free_monitor_info (monitor_info* mi);

/** Handle the heap dump level 0. */
void heap_dump_0 (int dump_level,
		  char *begin,
		  char *end,
		  jint num_traces,
		  JVMPI_CallTrace *traces);

/** Handle the heap dump level 1. */
void heap_dump_1 (char* dumpdir, 
		  int dump_level,
		  char* begin,
		  char* end,
		  jint num_traces,
		  JVMPI_CallTrace* traces);

/** Handle the heap dump level 2. */
void heap_dump_2 (char* dumpdir, 
		  int dump_level,
		  char* begin,
		  char* end,
		  jint num_traces,
		  JVMPI_CallTrace* traces);


void object_dump (jint data_len, char* data); 

/** Get the owners of the given object at the last heap dump.
 * @param obj the object to get the owners for.
 * @return an linked list of object_link containing the owners of obj.
 */
object_link* get_owners (jobjectID obj);

/** Check if it is possible to get object owners...
 */
int is_get_owners_possible ();

/** Throw away heap dump information due to GC. */
void remove_owners_information ();

/** get the down_link for the last requested object. */
down_link* get_last_down_link ();

/** Free the down_link for the last requested object. */
void free_last_down_link ();


/** handle the monitor dump */
hashtab* monitor_dump (char* begin,
		       char* end,
		       jint num_traces,
		       JVMPI_CallTrace* traces,
		       jint* threads_status); 

#endif /* HEAP_DUMP_H__ */

/* Emacs Local Variables: */
/* Emacs mode:C */
/* Emacs c-indentation-style:"gnu" */
/* Emacs c-hanging-braces-alist:((brace-list-open)(brace-entry-open)(defun-open after)(substatement-open after)(block-close . c-snug-do-while)(extern-lang-open after)) */
/* Emacs c-cleanup-list:(brace-else-brace brace-elseif-brace space-before-funcall) */
/* Emacs c-basic-offset:4 */
/* Emacs End: */
