/*
    Copyright (C) 2008 Fons Adriaensen <fons@kokkinizita.net>
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <stdlib.h>
#include <stdio.h>
#include <clthreads.h>
#include <sys/mman.h>
#include <signal.h>
#include "styles.h"
#include "mainwin.h"


#define NOPTS 9


XrmOptionDescRec options [NOPTS] =
{ 
    {"-help",    ".help",            XrmoptionNoArg,   "true"    },
    {"-name",    ".name",            XrmoptionSepArg,  0         },
    {"-type",    ".type",            XrmoptionSepArg,  "k20"     },
    {"-rate",    ".rate",            XrmoptionSepArg,  "25"      },
    {"-A",       ".ambisonic",       XrmoptionNoArg,   "true"    },
    {"-C",       ".correlation",     XrmoptionNoArg,   "true"    },
    {"-V",       ".vertical",        XrmoptionNoArg,   "true"    },
    {"-H",       ".horizontal",      XrmoptionNoArg,   "true"    },
    {"-O",       ".orientation",     XrmoptionSepArg,  0         }
};


static Jclient  *jclient = 0;
static Mainwin  *mainwin = 0;


static void help (void)
{
    fprintf (stderr, "\nJkmeter-%s\n", VERSION);
    fprintf (stderr, "(C) 2008 Fons Adriaensen  <fons@kokkinizita.net>\n");
    fprintf (stderr, "Syntax: jkmeter <options>\n");
    fprintf (stderr, "Options:\n");
    fprintf (stderr, "  -h            Display this text\n");
    fprintf (stderr, "  -name <name>  Name to use as jack client\n");
    fprintf (stderr, "  -type <type>  Meter type, k20 or k14 (k20)\n");
    fprintf (stderr, "  -rate <rate>  Update rate (30)\n");
    fprintf (stderr, "  -A            Ambisonic (4ch) mode\n");   
    fprintf (stderr, "  -C            Add correlation meter\n");   
    fprintf (stderr, "  -V            Vertical layout\n");   
    fprintf (stderr, "  -H            Horizontal layout\n");   
    exit (1);
}


static void sigint_handler (int)
{
    signal (SIGINT, SIG_IGN);
    mainwin->stop ();
}


int main (int ac, char *av [])
{
    X_resman      xresman;
    X_display     *display;
    X_handler     *handler;
    X_rootwin     *rootwin;
    int           ev;
    int           chan;
    int           corr;

    xresman.init (&ac, av, "jkmeter", options, NOPTS);
    if (xresman.getb (".help", 0)) help ();
            
    display = new X_display (xresman.get (".display", 0));
    if (display->dpy () == 0)
    {
	fprintf (stderr, "Can't open display.\n");
        delete display;
	exit (1);
    }

    chan = (xresman.getb (".ambisonic", 0)) ? 4 : 2;
    corr = (xresman.getb (".correlation", 0));
    Kmeter::load_images (display, SHARED "/jkmeter");
    Cmeter::load_images (display, SHARED "/jkmeter");
    styles_init (display, &xresman);
    rootwin = new X_rootwin (display);
    jclient = new Jclient (xresman.rname (), chan, corr);
    mainwin = new Mainwin (rootwin, &xresman, jclient);
    handler = new X_handler (display, mainwin, EV_X11);
    handler->next_event ();
    mainwin->x_map ();
    XFlush (display->dpy ());

    if (mlockall (MCL_CURRENT | MCL_FUTURE)) fprintf (stderr, "Warning: memory lock failed.\n");
    signal (SIGINT, sigint_handler); 

    do
    {
	ev = mainwin->process ();
	if (ev == EV_X11)
	{
	    rootwin->handle_event ();
	    handler->next_event ();
	}
    }
    while (ev != EV_EXIT);	

    delete jclient;
    delete handler;
    delete rootwin;
    delete display;
   
    return 0;
}



