/*
 * $Id: xmpp-servers.c,v 1.40 2008/03/01 17:57:21 errtu Exp $
 *
 * Copyright (C) 2007 Colin DIDIER
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <sys/types.h>
#include <sys/socket.h>

#include "module.h"
#include "network.h"
#include "recode.h"
#include "settings.h"
#include "signals.h"

#include "xmpp-servers.h"
#include "xmpp-channels.h"
#include "xmpp-protocol.h"
#include "xmpp-rosters.h"
#include "xmpp-tools.h"

static int
isnickflag_func(SERVER_REC *server, char flag)
{
	return FALSE;
}

static int
ischannel_func(SERVER_REC *server, const char *data)
{
	return FALSE;
}

static const char *
get_nick_flags(SERVER_REC *server)
{
	return "";
}

static void
send_message(SERVER_REC *server, const char *target, const char *msg,
    int target_type)
{
	char *recoded;

	g_return_if_fail(server != NULL);
	g_return_if_fail(target != NULL);
	g_return_if_fail(msg != NULL);

	if (!IS_XMPP_SERVER(server))
		return;
	
	/* ugly from irssi: recode the sent message back */
	recoded = recode_in(server, msg, target);

	if (target_type == SEND_TARGET_CHANNEL)
		xmpp_channel_send_message(XMPP_SERVER(server), target, recoded);
	else
		xmpp_send_message(XMPP_SERVER(server), target, recoded);

	g_free(recoded);
}

static void
server_cleanup(XMPP_SERVER_REC *server)
{
	g_return_if_fail(IS_XMPP_SERVER(server));

	/* close the connection */
	if (lm_connection_get_state(server->lmconn) !=
	    LM_CONNECTION_STATE_CLOSED) {
		lm_connection_close(server->lmconn, NULL);
	}
	lm_connection_unref(server->lmconn);

	lookup_servers = g_slist_remove(lookup_servers, server);
	servers = g_slist_remove(servers, server);

	g_free(server->nickname);
	g_free(server->jid);
	g_free(server->user);
	g_free(server->host);
	g_free(server->resource);
	g_free(server->ping_id);
}


SERVER_REC *
xmpp_server_init_connect(SERVER_CONNECT_REC *conn)
{
	XMPP_SERVER_REC *server;
	char *str;

	g_return_val_if_fail(IS_XMPP_SERVER_CONNECT(conn), NULL);
	if (conn->address == NULL || conn->address[0] == '\0')
		return NULL;
	if (conn->nick == NULL || conn->nick[0] == '\0')
		return NULL;

	server = g_new0(XMPP_SERVER_REC, 1);
	server->chat_type = XMPP_PROTOCOL;

	/* extract user informations */
	str = conn->nick;

	server->user = xmpp_extract_user(str);
	server->host = g_strdup(conn->address);
	server->jid = xmpp_have_host(str) ? xmpp_strip_resource(str) :
	    g_strconcat(server->user, "@", server->host, NULL);
	server->resource = xmpp_extract_resource(str);
	if (server->resource == NULL)
		server->resource = g_strdup("irssi-xmpp");

	g_free(str);

	/* init xmpp's properties */
	server->priority = settings_get_int("xmpp_priority");
	if (xmpp_priority_out_of_bound(server->priority))
		server->priority = 0;
	server->default_priority = TRUE;

	server->ping_id = NULL;
	server->features = 0;
	server->resources = NULL;
	server->roster = NULL;
	server->hmessage = NULL;
	server->hpresence = NULL;
	server->hiq = NULL;

	/* fill connrec record */
	server->connrec = (XMPP_SERVER_CONNECT_REC *)conn;
	server_connect_ref(conn);

	/* don't use irssi's sockets */
	server->connrec->no_connect = TRUE;
	server->connect_pid = -1;

	if (server->connrec->port <= 0)
		server->connrec->port = (server->connrec->use_ssl) ?
		    LM_CONNECTION_DEFAULT_PORT_SSL : LM_CONNECTION_DEFAULT_PORT;

	server->connrec->nick =
	    g_strdup(settings_get_bool("xmpp_set_nick_as_username") ?
	    server->user : server->jid);
	server->nickname = g_strdup(server->connrec->nick);

	/* init loudmouth connection structure */
	server->lmconn = lm_connection_new(NULL);
	lm_connection_set_server(server->lmconn, server->connrec->address);
	lm_connection_set_port(server->lmconn, server->connrec->port);
	lm_connection_set_jid(server->lmconn, server->jid);
	lm_connection_set_keep_alive_rate(server->lmconn, 30);

	server_connect_init((SERVER_REC *)server);
	server->connect_tag = 1;

	return (SERVER_REC *)server;
}

static LmSSLResponse
lm_ssl_cb(LmSSL *ssl, LmSSLStatus status, gpointer user_data)
{
	XMPP_SERVER_REC *server = XMPP_SERVER(user_data);

	switch (status) {
	case LM_SSL_STATUS_NO_CERT_FOUND:
		signal_emit("xmpp server status", 2, server,
			"SSL: No certificate found!");
		break;
	case LM_SSL_STATUS_UNTRUSTED_CERT:
		signal_emit("xmpp server status", 2, server,
			"SSL: Certificate is not trusted!");
		break;
	case LM_SSL_STATUS_CERT_EXPIRED:
		signal_emit("xmpp server status", 2, server,
			"SSL: Certificate has expired!");
		break;
	case LM_SSL_STATUS_CERT_NOT_ACTIVATED:
		signal_emit("xmpp server status", 2, server,
			"SSL: Certificate has not been activated!");
		break;
	case LM_SSL_STATUS_CERT_HOSTNAME_MISMATCH:
		signal_emit("xmpp server status", 2, server,
			"SSL: Certificate hostname does not match expected hostname!");
		break;
	case LM_SSL_STATUS_CERT_FINGERPRINT_MISMATCH: 
		signal_emit("xmpp server status", 2, server,
			"SSL: Certificate fingerprint does not match expected fingerprint!");
		break;
	case LM_SSL_STATUS_GENERIC_ERROR:
		/*signal_emit("xmpp server status", 2, server,
			"SSL: Generic SSL error!");*/
		break;
	}

	return LM_SSL_RESPONSE_CONTINUE;
}

static void
lm_close_cb(LmConnection *connection, LmDisconnectReason reason,
    gpointer user_data)
{
	XMPP_SERVER_REC *server;

	server = XMPP_SERVER(user_data);
	if (server == NULL || !server->connected)
		return;

	/* normal disconnection */
	if (reason == LM_DISCONNECT_REASON_OK)
		return;

	/* connection lost */
	server->connection_lost = TRUE;
	server_disconnect(SERVER(server));
}

static void
lm_auth_cb(LmConnection *connection, gboolean success,
    gpointer user_data)
{
	XMPP_SERVER_REC *server;
	LmMessage *msg;
	LmMessageNode *query;
	char *priority;
	
	server = XMPP_SERVER(user_data);
	if (server == NULL)
		return;

	if (!success)
		goto err;

	signal_emit("xmpp server status", 2, server,
	    "Authenticated successfully.");

	/* fetch the roster */
	signal_emit("xmpp server status", 2, server, "Requesting the roster.");
	msg = lm_message_new_with_sub_type(NULL, LM_MESSAGE_TYPE_IQ,
	    LM_MESSAGE_SUB_TYPE_GET);
	query = lm_message_node_add_child(msg->node, "query", NULL);
	lm_message_node_set_attribute(query, "xmlns", XMLNS_ROSTER);
	lm_send(server, msg, NULL);
	lm_message_unref(msg);

	/* discover server's features */
	msg = lm_message_new_with_sub_type(server->host, LM_MESSAGE_TYPE_IQ,
	    LM_MESSAGE_SUB_TYPE_GET);
	query = lm_message_node_add_child(msg->node, "query", NULL);
	lm_message_node_set_attribute(query, "xmlns", XMLNS_DISCO_INFO);
	lm_send(server, msg, NULL);
	lm_message_unref(msg);

	/* set presence available */
	signal_emit("xmpp server status", 2, server,
	    "Sending available presence message.");
	msg = lm_message_new_with_sub_type(NULL, LM_MESSAGE_TYPE_PRESENCE,
	    LM_MESSAGE_SUB_TYPE_AVAILABLE);

	priority = g_strdup_printf("%d", server->priority);
	lm_message_node_add_child(msg->node, "priority", priority);
	g_free(priority);

	lm_send(server, msg, NULL);
	lm_message_unref(msg);
	
	server->show = XMPP_PRESENCE_AVAILABLE;

	lookup_servers = g_slist_remove(lookup_servers, server);
	server_connect_finished(SERVER(server));
	signal_emit("event connected", 1, server);
	return;

err:
	server_connect_failed(SERVER(server), "Authentication failed");
}

static void
lm_open_cb(LmConnection *connection, gboolean success,
    gpointer user_data)
{
	XMPP_SERVER_REC *server;
	IPADDR ip;
	char *host;
	GError *error = NULL;

	server = XMPP_SERVER(user_data);
	if (server == NULL)
		return;

	if (!success)
		goto err;

	/* get the server address */
	host = lm_connection_get_local_host(server->lmconn);
	if (host != NULL) {
		net_host2ip(host, &ip);
		signal_emit("server connecting", 2, server, &ip);
		g_free(host);
	} else
		signal_emit("server connecting", 1, server);

	if (!lm_connection_authenticate(connection, server->user,
	    server->connrec->password, server->resource,
	    (LmResultFunction)lm_auth_cb, server, NULL, &error))
		goto err;

	return;

err:
	server->connection_lost = TRUE;
	server_connect_failed(SERVER(server),
	    (error != NULL) ? error->message : "Connection failed");
	if (error != NULL)
		g_error_free(error);
}


static gboolean
set_proxy(XMPP_SERVER_REC *server, GError **error)
{
	LmProxy *proxy;
	LmProxyType type;
	const char *str;
	char *recoded;

	g_return_val_if_fail(IS_XMPP_SERVER(server), FALSE);

	str = settings_get_str("xmpp_proxy_type");
	if (str != NULL && g_ascii_strcasecmp(str, XMPP_PROXY_HTTP) == 0)
		type = LM_PROXY_TYPE_HTTP;
	else {
		if (error != NULL) {
			*error = g_new(GError, 1);
			(*error)->message = g_strdup("Invalid proxy type");
		}
		return FALSE;
	}

	str = settings_get_str("xmpp_proxy_address");
	if (str == NULL || *str == '\0') {
		if (error != NULL) {
			*error = g_new(GError, 1);
			(*error)->message =
			    g_strdup("Proxy address not specified");
		}
		return FALSE;
	}

	int port = settings_get_int("xmpp_proxy_port");
	if (port <= 0) {
		if (error != NULL) {
			*error = g_new(GError, 1);
			(*error)->message =
			    g_strdup("Invalid proxy port range");
		}
		return FALSE;
	}

	proxy = lm_proxy_new_with_server(type, str, port);
	
	str = settings_get_str("xmpp_proxy_user");
	if (str != NULL && *str != '\0') {
		recoded = xmpp_recode_out(str);
		lm_proxy_set_username(proxy, recoded);
		g_free(recoded);
	}

	str = settings_get_str("xmpp_proxy_password");
	if (str != NULL && *str != '\0') {
		recoded = xmpp_recode_out(str);
		lm_proxy_set_password(proxy, recoded);
		g_free(recoded);
	}

	lm_connection_set_proxy(server->lmconn, proxy);
	lm_proxy_unref(proxy);

	return TRUE;
}

void
xmpp_server_connect(XMPP_SERVER_REC *server)
{
	LmSSL *ssl;
	GError *error = NULL;

	if (!IS_XMPP_SERVER(server))
		return;

	/* SSL */
	if (server->connrec->use_ssl) {
		if (!lm_ssl_is_supported()) {
			error = g_new(GError, 1);
			error->message =
			    g_strdup("SSL is not supported in this build");
			goto err;
		}

		ssl = lm_ssl_new(NULL, (LmSSLFunction)lm_ssl_cb,
		    server, NULL);
		lm_connection_set_ssl(server->lmconn, ssl);
		lm_ssl_unref(ssl);
	} 

	/* Proxy */
	if (settings_get_bool("xmpp_use_proxy"))
		if (!set_proxy(server, &error))
			goto err;

	lm_connection_set_disconnect_function(server->lmconn,
	    (LmDisconnectFunction)lm_close_cb, (gpointer)server,
	    NULL);

	lookup_servers = g_slist_append(lookup_servers, server);
	signal_emit("server looking", 1, server);

	if (!lm_connection_open(server->lmconn, 
	    (LmResultFunction)lm_open_cb, (gpointer)server, NULL,
	    &error))
		goto err;

	return;

err:
	if (IS_SERVER(server)) {
		server->connection_lost = TRUE;
		server_connect_failed(SERVER(server),
		    (error != NULL) ? error->message : NULL);
	}
	if (error != NULL)
		g_error_free(error);
}

static void
sig_connected(XMPP_SERVER_REC *server)
{
	if (!IS_XMPP_SERVER(server))
		return;

	server->channels_join = (void (*)(SERVER_REC *, const char *, int))
	    xmpp_channels_join;
	server->isnickflag = isnickflag_func;
	server->ischannel = ischannel_func;
	server->get_nick_flags = get_nick_flags;
	server->send_message = send_message;
	
	server->connected = TRUE;
	server->connect_tag = -1;
}

static void
sig_server_disconnected(XMPP_SERVER_REC *server)
{
	if (!IS_XMPP_SERVER(server))
		return;

	server_cleanup(server);
}

static void
sig_server_connect_failed(XMPP_SERVER_REC *server, char *msg)
{
	if (!IS_XMPP_SERVER(server))
		return;

	server_cleanup(server);
}

static void
sig_server_quit(XMPP_SERVER_REC *server, char *reason)
{
	LmMessage *msg;
	char *status_recoded;

	if (!IS_XMPP_SERVER(server))
		return;
	
	msg = lm_message_new_with_sub_type(NULL, LM_MESSAGE_TYPE_PRESENCE,
	    LM_MESSAGE_SUB_TYPE_UNAVAILABLE);

	status_recoded = xmpp_recode_out((reason != NULL) ?
	    reason : settings_get_str("quit_message"));
	lm_message_node_add_child(msg->node, "status", status_recoded);
	g_free(status_recoded);

	lm_send(server, msg, NULL);
	lm_message_unref(msg);
}

void
xmpp_servers_init(void)
{
	signal_add_first("server connected", (SIGNAL_FUNC)sig_connected);
	signal_add_last("server disconnected",
	    (SIGNAL_FUNC)sig_server_disconnected);
	signal_add_last("server connect failed",
	    (SIGNAL_FUNC)sig_server_connect_failed);
	signal_add("server quit", (SIGNAL_FUNC)sig_server_quit);

	settings_add_bool("xmpp_lookandfeel", "xmpp_set_nick_as_username",
	    FALSE);
	settings_add_bool("xmpp_proxy", "xmpp_use_proxy", FALSE);
	settings_add_str("xmpp_proxy", "xmpp_proxy_type", "http");
	settings_add_str("xmpp_proxy", "xmpp_proxy_address", NULL);
	settings_add_int("xmpp_proxy", "xmpp_proxy_port", 8080);
	settings_add_str("xmpp_proxy", "xmpp_proxy_user", NULL);
	settings_add_str("xmpp_proxy", "xmpp_proxy_password", NULL);
}

void
xmpp_servers_deinit(void)
{
	GSList *tmp, *next;

	/* disconnect all servers before unloading the module */
	for (tmp = lookup_servers; tmp != NULL; tmp = next) {
		next = tmp->next;
		if (IS_XMPP_SERVER(tmp->data))
			server_connect_failed(SERVER(tmp->data), NULL);
	}
	for (tmp = servers; tmp != NULL; tmp = next) {
		next = tmp->next;
		if (IS_XMPP_SERVER(tmp->data))
			server_disconnect(SERVER(tmp->data));
	}

	signal_remove("server connected", (SIGNAL_FUNC)sig_connected);
	signal_remove("server disconnected",
	    (SIGNAL_FUNC)sig_server_disconnected);
	signal_remove("server connect failed",
	    (SIGNAL_FUNC)sig_server_connect_failed);
	signal_remove("server quit", (SIGNAL_FUNC)sig_server_quit);
}
