/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#ifndef IHISTOGRAMMAKER_H
#define IHISTOGRAMMAKER_H


#include <vtkProcessObject.h>
#include "idatastretch.h"
#include "iparallelworker.h"


#include "iposition.h"

class iHistogram;

class vtkDataSet;
class vtkFloatArray;


class iHistogramMaker : public vtkProcessObject, public iDataStretchUser, protected iParallelWorker
{
	
	IPOINTER_AS_PART(ViewModule);

public:
	
	vtkTypeMacro(iHistogramMaker,vtkProcessObject);
	static iHistogramMaker* New(iViewModule *vm = 0);
	
	inline int GetCurrentNumberOfBins(){ return mNumBins; }
	inline int GetInputComponent(){ return mComponent; }
	virtual int GetEstimatedNumberOfBins();

	virtual const iHistogram* GetHistogram(int nBins = 0);
	virtual const iHistogram* GetHistogram(int nBins, float vMin, float vMax);
	void GetHistogramRange(float &min, float &max);
	void GetHistogramRange(float &valMin, vtkIdType &cellMin, const iPosition* &posMin, float &valMax, vtkIdType &cellMax, const iPosition* &posMax);
	
	void SetFullResolution(bool s);
	inline bool GetFullResolution() const { return mFullResolution; }
	inline bool SupportsFullResolution() const { return mSupportsFullResolution; }

	virtual void SetInput(vtkDataSet *input, int c);
	virtual void SetInputComponent(int c);
	virtual vtkDataSet* GetInput();

	void AttachHistogram(bool s);

	void SetDataRank(unsigned int r);
	inline unsigned int GetDataRank() const { return mDataRank; }

	void Update();

protected:
	
	iHistogramMaker(iViewModule *vm);
	virtual ~iHistogramMaker();

	virtual int ExecuteStep(int step, iParallel::ProcessorInfo &p);

	virtual void OnStretchChanged();

	bool ComputeRange();
	float GetValue(vtkIdType l) const;

	//
	//  Inherit these functions
	//
	virtual bool IsDataPresent();
	virtual vtkIdType GetNumberOfCells();
	virtual void FindPositionForCell(vtkIdType cell, iPosition &pos);
	virtual bool PrepareForPass();
	virtual bool IsValidIndex(vtkIdType l) const;
	virtual float GetScalarValue(vtkIdType l) const;
	virtual float GetVectorValue(vtkIdType l) const;
	virtual float GetTensorValue(vtkIdType l) const;
	virtual double GetValueWeight(vtkIdType l) const;

protected: 
	
	int mComponent;

private:

	const int mLowResBins, mLowResSize;
	bool mAttached, mFullResolution, mSupportsFullResolution;
	iHistogram *mHisto;
	int mNumProcs, mNumBins;
	unsigned int mDataRank;

	//
	//  MTime for the last calculation of the histogram - need to keep track of it
	//  separately so that other changes to this class do not hide the change in the input data.
	//
	vtkTimeStamp mComputeTime, mComputeRangeTime;

	//
	//  Persisting helper variables
	//
	struct Range
	{
		float ValMax;
		float ValMin;
		vtkIdType CellMax;
		vtkIdType CellMin;
	} *mRanges;
	double **mValues;
	//
	//  Work variables
	//
	float *wDataPtr;
	vtkIdType wNumCom;
};


//
//  Inline to allow a smart compiler to take the switch out of inner loops
//
inline float iHistogramMaker::GetValue(vtkIdType l) const
{
	switch(mDataRank)
	{
	case 0U:
		{
			return this->GetScalarValue(l);
		}
	case 1U:
		{
			return this->GetVectorValue(l);
		}
	case 2U:
		{
			return this->GetTensorValue(l);
		}
	default:
		{
			return 0.0f;
		}
	}
}

#endif

