/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


//
//  A base class for all GUI shells around the VTK-based core
//

#ifndef ISHELL_H
#define ISHELL_H


#include <vtkObjectBase.h>


#include "iarray.h"
#include "istring.h"

#include <vtkSetGet.h>

class iControlModule;
class iFile;

class vtkTimerLog;


class iShell : public vtkObjectBase
{

public:

	struct Option
	{
		iString Name;
		iString Help;
		bool NeedsValue;
		iString Value;
		Option() { NeedsValue = false; }
	};

	vtkTypeMacro(iShell,vtkObjectBase);

	//
	//  A helper function that creates, runs, and deletes a shell in one call
	//
	static void RunApplication(const iString &type, int argc, char **argv);

	//
	//  Create a shell with this function
	//
	static iShell* New(const iString &type = "", int argc = 0, char **argv = 0);

	//
	//  This function does the shell's work after all the components are created. 
	//
	void Run();

	//
	//  Exit from the current shell 
	//
	virtual void Exit() = 0;

	//
	//  Destroy the shell with this function
	//
	void Delete();

	//
	//  Access to components
	//
	inline const iString& Type() const { return mType; }
	inline iControlModule* GetControlModule() const { return mControlModule; }
	inline bool IsRunning() const { return mRunning; }

	const iString& GetEnvironment(int key) const;
	inline const iString& GetStateFileName() const { return mStateFileName; }

	bool LoadShellStateFromFile(iFile &F);
	bool SaveShellStateToFile(iFile &F) const;

	void OnInitAtom();
	void OnLoadStateFileAtom(int prog);
	
protected:

	iShell(const iString &type, int argc, char **argv);
	virtual ~iShell();  

	//
	//  This functions parses command-line options.
	//
	void ParseCommandLineOptions(int argc, char **argv);  
	void AddCommandLineOption(const iString &root, bool hasValue, const iString &help);
	bool AnalyseOneCommandLineOption(const struct Option &o);
	
	//
	//  Because of cyclic dependence of different objects on each other, the constructor
	//  and destructor functions of a child shell should be empty, and all constructor/destructor 
	//  work must be done in overwrites of the following two functions:
	//
	virtual void ConstructorBody() = 0;  
	virtual void DestructorBody() = 0;  

	//
	// These functions must be overwritten in a child class
	//
	virtual void Start() = 0;
	virtual void RunBody() = 0; // body of Run()
	virtual void PrepareForConstruction() = 0;

	virtual void PrepareToLoadStateFile() = 0;
	virtual bool LoadShellStateFromFileBody(iFile &F) = 0;
	virtual bool SaveShellStateToFileBody(iFile &F) const  = 0;

	virtual void DefineSpecificCommandLineOptions() = 0;
	virtual bool AnalyseOneExtendedCommandLineOption(const struct Option &o) = 0;

	//
	//  These functions can be used to animate the initialization/load state file progress.
	//  Do nothing by default.
	//
	virtual void OnInitAtomBody(bool timed);
	virtual void OnLoadStateFileAtomBody(int prog);
	
	//
	//  Useful helpers
	//
	bool HasStateFile() const;

	iControlModule* ControlModule() const { return mControlModule; }

private:

	iControlModule *mControlModule;

	bool mRunning;
	const iString mType;
	static const iString mOptionPrefix;
	iArray<Option> mOptions;
	int mNumProcs, mFirstOption;

	//
	//  Initialization
	//
	struct InitStep
	{
		int Current;
		int Total;
	};

	int mCurInitStep;
	InitStep mInitSteps[4];
	vtkTimerLog *mInitTimer;

	//
	//  Environment
	//
	iString mEnvBaseDir;
	iString mEnvDataDir;
	iString mEnvImageDir;
	iString mEnvScriptDir;
	iString mEnvPaletteDir;

	iString mStateFileName, mScriptFileName;

#ifdef I_DEBUG
	void InitHelper();
#endif
};

#endif  // ISHELL_H
