/*
 * userland + libusb specific functions
 * $Id: ifp_os_libusb.c,v 1.2 2005/08/25 04:10:54 jim-campbell Exp $
 *
 * Copyright (C) 2004 Geoff Oakham <oakhamg@users.sourceforge.net>
 */

#include <string.h>
#include <time.h>
#include <usb.h>
#include <iconv.h>
#include <locale.h>
#include <langinfo.h>

#include "ifp.h"
#include "ifp_os.h"
#include "prim.h"

static int local_iconv(char const * i_type, char const * o_type,
	char * ob, int max_o, const char * ib, int max_i)
{
	int i=0, e=0;
	char const * ibb = (char const *)ib;
	char * obb = (char *)ob;
	size_t i_n = max_i;
	size_t o_n = max_o;
	iconv_t ICONV;
	size_t r;

	if (i_type == NULL) {
		ifp_err("itype is NULL");
		i = -EINVAL;
		goto err;
	}

	if (o_type == NULL) {
		ifp_err("otype is NULL");
		i = -EINVAL;
		goto err;
	}

	ICONV = iconv_open(o_type, i_type);
	if (ICONV == (iconv_t)-1) {
		i = -errno;
		if(i==-EINVAL) {
			ifp_err_i(i, "conversion not supported by system");
		} else {
			ifp_err_i(i, "couldn't open conversion handle");
		}
		goto err;
	}

	r = iconv(ICONV, &ibb, &i_n, &obb, &o_n);
	if (r == (size_t)-1) {
		i = -errno;
		ifp_err_i(i, "problem converting, i_n is %d, o_n is %d, r = %d", i_n, o_n, r);
		goto err2;
	}

err2:
	e = iconv_close(ICONV);
	if (e) {
		e = -errno;
		ifp_err_i(e, "couldn't close conversion");
		i = i?i:e;
	}
err:
	return i;
}

int ifp_locale_to_utf16(char * ob, int max_o, const char * ib, int max_i)
{
	int i;

	i = local_iconv(nl_langinfo(CODESET), "UTF-16LE", ob, max_o, ib, max_i);
	ifp_err_jump(i, err, "conversion failed");

err:
	return i;
}

static int utf16_sizeof(uint16_t * data, int max) {
	int i=0, j;
	for (j=0; j!=max && i==0; j++) {
		if (data[j] == 0) {
			return j;
		}
	}
	return max;
}

int ifp_utf16_to_locale(char * ob, int max_o, const char * ib, int max_i)
{
	int i;
	int n;

	n = utf16_sizeof((uint16_t *)ib, max_i/sizeof(uint16_t));

	i = local_iconv("UTF-16LE", nl_langinfo(CODESET), ob, max_o, ib, (n+1)*sizeof(uint16_t));
	//i = local_iconv("UTF-16LE", nl_langinfo(CODESET), ob, max_o, ib, max_i);
	ifp_err_jump(i, err, "conversion failed");

err:
	return i;
}

int ifp_os_sleep(int ms) {
    struct timespec ts;

    ts.tv_sec = ms / 1000;
    ts.tv_nsec = (ms % 1000)*1000000;
    return nanosleep(&ts, NULL);
}

#define MAX_TRIES 8

int ifp_os_control_send(struct ifp_device * dev, int command, int arg1, int arg2,
	int * r1, int * r2)
{
	struct usb_dev_handle * usbdev = dev->device;
	uint8_t ctl[8];
	int c_size;
	int i;
	int max_try = MAX_TRIES; IFP_BUG_ON(r1 == NULL); c_size = r2 ? 8 : 4;

	//usb PIPE/stall error detection and correction.
	//
	// AFAIK, only Linux is affected by this bug, so I think it's important
	// that the code get really noisy if this happens on another platform.
	// (Hence the ifdefs.)
	do {
		i = usb_control_msg(usbdev, IFP_REQ_TYPE, command, arg1, arg2, ctl, c_size,
		    IFP_TIMEOUT);
		max_try--;
		if (i == -EPIPE) {
#ifdef linux
			if (command != IFP_FILE_DOWNLOAD || max_try != (MAX_TRIES - 1)) {
#endif
				ifp_err("found an EPIPE error not previously documented. "
					"command=%02x try=%d", command, MAX_TRIES - max_try);
#ifdef linux
			}

			if (command == IFP_FILE_DOWNLOAD) {
				dev->download_pipe_errors++;
			}
#endif
			ifp_os_sleep(100); //I'm still not convienced this helps anything
		}
	} while(i == -EPIPE && max_try > 0);
	if (i < 0) {
		ifp_err_i(i, "error %s ifp control code the command %0x (%d, %d)."
			" ctl[%d]  Returned %d.",
			"sending", command, arg1, arg2, c_size, i);
		return i;
	} else if (i != c_size) {
		if (command == IFP_FILE_DOWNLOAD) { dev->alt_readcount++; }
		ifp_err_i(i, "warning: unexpected error value.. I expected %d.",
			c_size);
	} else {
		if (command == IFP_FILE_DOWNLOAD) { dev->alt_readcount++; }
		i = 0;
	}

	if (r2) {
		*r2 = ifp_os_le32_to_cpup(ctl + 4);
	}
	*r1 = ifp_os_le32_to_cpup(ctl);

	return i;
}

int ifp_os_push(struct ifp_device * dev, void * p, int n)
{
	int i = 0;

	i = usb_bulk_write(dev->device, dev->bulk_to,
		p, n, IFP_TIMEOUT);

	if (i < 0) {
		ifp_err_i(i, "usb_bulk_msg failed");
		return i;
	} else if (i != n) {
		ifp_wrn(" usb_bulk_msg wrote %d bytes instead of %d.", i, n);
	}

	return 0;
}

int ifp_os_pop(struct ifp_device * dev, void * p, int n)
{
	int i = 0;
	memset(p, 0, n);

	i = usb_bulk_read(dev->device, dev->bulk_from,
		p, n, IFP_TIMEOUT);

	if (i < 0) {
		ifp_err_i(i, "usb_bulk_msg failed");
		return i;
	} else if (i != n) {
		ifp_wrn("read in %d bytes instead of %d",i,n);
	}

	return 0;
}

int ifp_os_init(struct ifp_device * dev, void * device_handle)
{
	int i = 0;
	struct usb_dev_handle * ldev = device_handle;
	struct usb_device * usbdev = usb_device(ldev);
	struct usb_endpoint_descriptor * endpoint;
	int address0, address1;

	//FIXME: move this to a more appropriate place.
	setlocale(LC_ALL, "");
	
	dev->model = usbdev->descriptor.idProduct;
	dev->device = device_handle;

	endpoint = usbdev->config[0].interface[0].altsetting[0].endpoint;
	address0 = endpoint[0].bEndpointAddress;
	address1 = endpoint[1].bEndpointAddress;
	if (address0 & 0x0080) {
		IFP_BUG_ON(address1 & 0x0080);
		dev->bulk_from = address0;
		dev->bulk_to = address1;
	} else {
		IFP_BUG_ON((address1 & 0x0080)==0);
		dev->bulk_from = address1;
		dev->bulk_to = address0;
	}

	return i;
}
int ifp_os_finalize(struct ifp_device * dev)
{

	//release happens somewhere else
	dev->device = NULL;
	return 0;
}

void * ifp_find_device(void) {
	const int iRiver_Vendor = 0x4102;

	struct usb_bus *busses;
	struct usb_bus *bus;
	struct usb_device *tmp_dev;

	if(usb_find_busses() < 0) {
		ifp_err("Could not find any USB busses.");
		return NULL;
	}
	if(usb_find_devices() < 0) {
		ifp_err("USB devices not found(nor hubs!).\n"
		"(On some systems you may need to run this program as root.)");
		return NULL;
	}

	busses = usb_get_busses();

	for (bus = busses; bus; bus = bus->next) {
		for (tmp_dev = bus->devices; tmp_dev; tmp_dev = tmp_dev->next) {
			// reject UMS firmware.
			if (tmp_dev->descriptor.idVendor == iRiver_Vendor) {
				if ((tmp_dev->descriptor.idProduct & 0xFF00)
					== 0x1100)
				{
					ifp_info("ignoring device with UMS firmware.");
				} else if ((tmp_dev->descriptor.idProduct
					& 0xFF00) != 0x1100)
				{
					return usb_open(tmp_dev);
				}
			}
		}
	}

	return NULL;
}

int ifp_release_device(void * dh) {
#if 0
	//Does this actually help anything?
    usb_reset(dh);      // Don't use for Mac OS X.
#endif

    usb_close(dh);
    return 0;
}


