/** 
* @file hyantes.c
* @brief front end part
* @author serge guelton
* @date 2008-02-13
*/
/*
 * This file is part of hyantes.
 *
 * hyantes is free software; you can redistribute it and/or modify
 * it under the terms of the CeCILL-C License
 *
 * You should have received a copy of the CeCILL-C License
 * along with this program.  If not, see <http://www.cecill.info/licences>.
 */

/* project header */
#include "hs_config.h"
#include "planartree.h"
#include "read_input.h"
#include "hyantes.h"
#include "computing.h"
#include "interpolation.h"
#include "module.h"

/* system header */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <sys/time.h>
#include <time.h>
#include <float.h>
#include <string.h>
#include <values.h>
#include <assert.h>


/* globals */


/** 
* @brief frees allocated memory in case of error in  hs_smoothing
* 
* @return NULL
*/
#define smoothing_error()\
  if( ptrees != NULL ) free(ptrees);\
  if( pReference != NULL ) free(pReference);\
  if( out != NULL ) free(out);\
  return NULL;

/** 
* @brief list all available smoothing methods
*  a smoothing method always take a radius and a distance (in kilometers) as input
*  and returns a new distance (possibly zero)
* 
* @param sz pointer to the number of  smoothing methods
* 
* @return array of string constant of size *sz. Memory is still owned by hyantes
*/
const char **hs_list_smoothing(size_t * sz)
{
    return list_smoothing(sz);
}

/** 
* @brief observer of the execution of the computation
* @return number of executed tasks from the beginning of the computation
* It should be 0 before the beginning
* And resoLat x resoLon, in the end
*/
unsigned long hs_status()
{
    return get_status();
}

/** 
* @brief perform the smoothing of target area inside visu, using potentials from pFileReference
* the smoothing is performed using function_name smoothing method, with a radius of function_param
* the resolution of the output matrix will be resoLat x resoLon
* 
* @param resoLat number of latitude points computed
* @param resoLon  number of longitude points computed
* @param function_name name of a smoothing method listed by hs_list_smoothing
* @param function_param parameter (in kilometers) of the smoothing method
* @param visu visualization window 
* @param pFileReference file containg the data in the format 
*    latitude longitude potential
*    latitude longitude potential
*    ...
*    latitude longitude potential
*  where latitude and longitude are given in degrees
* 
* @return an allocated array of size resoLat x resoLon containing triplets (lat, lon, pot) 
*   or
*       NULL if an error occured
*/
hs_potential_t *hs_smoothing
   (int _resoLat,
   int _resoLon,
   const char *function_name,
   double function_param, hs_coord_t visu, FILE * pFileReference)
{

#define TIME_VAL_INIT { (time_t)0,(suseconds_t)0 }

    struct timeval start = TIME_VAL_INIT;
    struct timeval end = TIME_VAL_INIT;
    long t_lissage;

    size_t ref_count = 0;
    Stock *pReference = NULL;
    double stockTotal = 0.;
    size_t resoLat=(_resoLat>0)?(size_t)_resoLat:0;
    size_t resoLon=(_resoLon>0)?(size_t)_resoLon:0;

    /* qtree et ptree */
    struct qt tree = {
        NULL,                   /*fhg */
        NULL,                   /*fhd */
        NULL,                   /*fbg */
        NULL,                   /*fbd */
        0.f,                    /*valeur */
        {                       /*coords */
               FLT_MAX,
               FLT_MAX,
               -FLT_MAX,
           -FLT_MAX},
        1                       /*size */
    };
    planartree *ptrees = NULL;
    hs_potential_t *out = NULL;


    /* prepare smoothing function */
    set_func_inter(function_name, function_param);

    /* convert visu to Radian */
    visu.mLat *= M_PI / 180;
    visu.mLon *= M_PI / 180;
    visu.MLat *= M_PI / 180;
    visu.MLon *= M_PI / 180;

    /* read data */
    fprintf(stdout, "# reading input data file ...\n");
    pReference = read_input(pFileReference,
       &tree.coords, &ref_count, &stockTotal);

    if(ref_count == 0 || pReference == NULL)
    {
        free_quadtree();
        smoothing_error();
    }

    fprintf(stdout, "# creating  quadtree ...\n");
    if(build_quadtree(pReference, ref_count, &tree) == 0)
    {
        free_quadtree();
        smoothing_error();
    }


    ptrees = create_planartree(&tree);
    free_quadtree();

    if(ptrees == NULL)
    {
        fprintf(stderr,
           "[hs_smoothing::create_planartree] error while creating planartree\n");
        smoothing_error();
    }

    if(gettimeofday(&start, NULL) == -1)
    {
        perror("[hs_smoothing::time] ");
        smoothing_error();
    }

    /* main computation */
    fprintf(stdout, "# computing map ...\n");
    out = do_smoothing(ptrees, visu, resoLat, resoLon);
    if(out == NULL)
    {
        fprintf(stderr,
           "[hs_smoothing::do_smoothing] error while computing map");
        smoothing_error();
    }
    if(gettimeofday(&end, NULL) == -1)
    {
        perror("[hs_smoothing::time] ");
        smoothing_error();
    }

    /* fix latitude and longitude back to degree */
    {
        size_t i;
        for(i = 0; i < resoLat * resoLon; ++i)
        {
            out[i].lat *= 180 / M_PI;
            out[i].lon *= 180 / M_PI;
        }
    }

    t_lissage =
       (end.tv_sec - start.tv_sec) * 1000000 + (end.tv_usec -
       start.tv_usec);
    fprintf(stdout, "# smoothing duration: %ld micro seconds\n",
       t_lissage);

    fprintf(stdout, "# cleaning memory ...\n");
    free(ptrees);
    free(pReference);

    /* the end */
    return out;


}
