#include "hrn-cluster-node.h"

enum {
    VISIBILITY_CHANGED,
    CHILD_CHANGED,
    CHILD_ADDED,
    CHILD_REMOVED,
    LAST_SIGNAL,
};

G_DEFINE_TYPE (HrnClusterNode, hrn_cluster_node, G_TYPE_OBJECT);
static guint32 signals[LAST_SIGNAL] = {0,};

static void
free_audio_cluster (AudioCluster *cluster)
{
    g_hash_table_destroy (cluster->name_to_artist);
    g_hash_table_destroy (cluster->name_to_album);
    g_slice_free (AudioCluster, cluster);
}

static void
free_image_cluster (ImageCluster *cluster)
{
    g_hash_table_destroy (cluster->year_to_cluster);
    g_hash_table_destroy (cluster->month_to_cluster);
    g_slice_free (ImageCluster, cluster);
}

static void
free_artist_cluster (ArtistCluster *cluster)
{
    g_free (cluster->thumbnail_uri);

    g_slice_free (ArtistCluster, cluster);
}

static void
free_album_cluster (AlbumCluster *cluster)
{
    int i;

    g_free (cluster->thumbnail_uri);

    for (i = 0; i < cluster->artists->len; i++) {
        g_free (cluster->artists->pdata[i]);
    }
    g_ptr_array_free (cluster->artists, TRUE);

    g_slice_free (AlbumCluster, cluster);
}

static void
free_track_cluster (TrackCluster *cluster)
{
    g_slice_free (TrackCluster, cluster);
}

static void
free_year_cluster (YearCluster *cluster)
{
    g_free (cluster->thumbnail_uri);
    g_free (cluster->orient);
    g_slice_free (YearCluster, cluster);
}

static void
free_month_cluster (MonthCluster *cluster)
{
    g_free (cluster->thumbnail_uri);
    g_free (cluster->orient);
    g_slice_free (MonthCluster, cluster);
}

static void
free_picture_cluster (PictureCluster *cluster)
{
    g_slice_free (PictureCluster, cluster);
}

static void
free_video_cluster (VideoCluster *cluster)
{
    g_slice_free (VideoCluster, cluster);
}

static void
hrn_cluster_node_finalize (GObject *object)
{
    HrnClusterNode *self = (HrnClusterNode *) object;

    if (self->children) {
        GSequenceIter *iter = g_sequence_get_begin_iter (self->children);

        while (!g_sequence_iter_is_end (iter)) {
            HrnClusterNode *node = g_sequence_get (iter);

            g_object_unref (node);
            iter = g_sequence_iter_next (iter);
        }

        g_sequence_free (self->children);
        self->children = NULL;
    }

    switch (self->type) {
    case HRN_CLUSTER_NODE_TYPE_AUDIO_ROOT:
        free_audio_cluster ((AudioCluster *) self->data);
        break;

    case HRN_CLUSTER_NODE_TYPE_IMAGE_ROOT:
        free_image_cluster ((ImageCluster *) self->data);
        break;

    case HRN_CLUSTER_NODE_TYPE_VIDEO_ROOT:
        /* Nothing to free */
        break;

    case HRN_CLUSTER_NODE_TYPE_ARTIST:
        free_artist_cluster ((ArtistCluster *) self->data);
        break;

    case HRN_CLUSTER_NODE_TYPE_ALBUM:
        free_album_cluster ((AlbumCluster *) self->data);
        break;

    case HRN_CLUSTER_NODE_TYPE_TRACK:
        free_track_cluster ((TrackCluster *) self->data);
        break;

    case HRN_CLUSTER_NODE_TYPE_YEAR:
        free_year_cluster ((YearCluster *) self->data);
        break;

    case HRN_CLUSTER_NODE_TYPE_MONTH:
        free_month_cluster ((MonthCluster *) self->data);
        break;

    case HRN_CLUSTER_NODE_TYPE_IMAGE:
        free_picture_cluster ((PictureCluster *) self->data);
        break;

    case HRN_CLUSTER_NODE_TYPE_VIDEO:
        free_video_cluster ((VideoCluster *) self->data);
        break;

    default:
        break;
    }

    g_free (self->name);
    g_free (self->canonical_name);

    G_OBJECT_CLASS (hrn_cluster_node_parent_class)->finalize (object);
}

static void
hrn_cluster_node_dispose (GObject *object)
{
    G_OBJECT_CLASS (hrn_cluster_node_parent_class)->dispose (object);
}

static void
hrn_cluster_node_class_init (HrnClusterNodeClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;

    o_class->dispose = hrn_cluster_node_dispose;
    o_class->finalize = hrn_cluster_node_finalize;

    signals[VISIBILITY_CHANGED] = g_signal_new ("visibility-changed",
                                                G_TYPE_FROM_CLASS (klass),
                                                G_SIGNAL_NO_RECURSE |
                                                G_SIGNAL_RUN_FIRST,
                                                0, NULL, NULL,
                                                g_cclosure_marshal_VOID__BOOLEAN,
                                                G_TYPE_NONE, 1,
                                                G_TYPE_BOOLEAN);
    signals[CHILD_ADDED] = g_signal_new ("child-added",
                                         G_TYPE_FROM_CLASS (klass),
                                         G_SIGNAL_NO_RECURSE |
                                         G_SIGNAL_RUN_FIRST, 0, NULL, NULL,
                                         g_cclosure_marshal_VOID__OBJECT,
                                         G_TYPE_NONE, 1,
                                         HRN_TYPE_CLUSTER_NODE);
    signals[CHILD_REMOVED] = g_signal_new ("child-removed",
                                           G_TYPE_FROM_CLASS (klass),
                                           G_SIGNAL_NO_RECURSE |
                                           G_SIGNAL_RUN_FIRST, 0, NULL, NULL,
                                           g_cclosure_marshal_VOID__OBJECT,
                                           G_TYPE_NONE, 1,
                                           HRN_TYPE_CLUSTER_NODE);
    signals[CHILD_CHANGED] = g_signal_new ("child-changed",
                                           G_TYPE_FROM_CLASS (klass),
                                           G_SIGNAL_RUN_FIRST |
                                           G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                           g_cclosure_marshal_VOID__OBJECT,
                                           G_TYPE_NONE, 1,
                                           HRN_TYPE_CLUSTER_NODE);
}

static void
hrn_cluster_node_init (HrnClusterNode *self)
{
    self->hidden = FALSE; /* Shown by default */
    self->visible_children = 0;
}

HrnClusterNode *
hrn_cluster_node_new (HrnClusterNodeType type,
                      GCompareDataFunc   sorter)
{
    HrnClusterNode *node = g_object_new (HRN_TYPE_CLUSTER_NODE, NULL);

    node->type = type;

    /* If there is no sorter, then there are no children */
    if (sorter) {
        node->children = g_sequence_new (NULL);
        node->sorter = sorter;
    }

    return node;
}

static guint
count_visible_children (HrnClusterNode *parent)
{
    GSequenceIter *iter;
    int i = 0;

    if (parent->children == NULL) {
        return 0;
    }

    iter = g_sequence_get_begin_iter (parent->children);
    while (g_sequence_iter_is_end (iter) == FALSE) {
        HrnClusterNode *child_node = g_sequence_get (iter);

        if (child_node->hidden == FALSE) {
            i++;
        }

        iter = g_sequence_iter_next (iter);
    }

    return i;
}

static void
child_visibility_changed (HrnClusterNode *child,
                          gboolean        hidden,
                          HrnClusterNode *parent)
{
    parent->visible_children = count_visible_children (parent);

    if (parent->visible_children <= 0) {
        hrn_cluster_node_set_hidden (parent, TRUE);
    } else {
        hrn_cluster_node_set_hidden (parent, FALSE);
    }
}

void
hrn_cluster_node_add_child (HrnClusterNode *parent,
                            HrnClusterNode *child)
{
    child->iter = g_sequence_insert_sorted (parent->children, child,
                                            parent->sorter, NULL);
    if (child->parent != NULL) {
        g_warning ("Child %s is already owned\n", child->name);
    }

    child->parent = parent;

    /* If the parent is hidden, then we want the child hidden too */
    if (parent->hidden == FALSE) {
        parent->visible_children++;
    } else {
        child->hidden = TRUE;
    }

    child->visibility_id = g_signal_connect (child, "visibility-changed",
                                             G_CALLBACK (child_visibility_changed),
                                             parent);

    g_signal_emit (parent, signals[CHILD_ADDED], 0, child);
}

void
hrn_cluster_node_remove_child (HrnClusterNode *parent,
                               HrnClusterNode *child)
{
    g_signal_handler_disconnect (child, child->visibility_id);
    child->visibility_id = 0;

    if (child->hidden == FALSE) {
        parent->visible_children--;
    }
    g_sequence_remove (child->iter);

    g_signal_emit (parent, signals[CHILD_REMOVED], 0, child);
}

void
hrn_cluster_node_set_hidden (HrnClusterNode *node,
                             gboolean        hidden)
{
    g_return_if_fail (node != NULL);

    if (node->hidden == hidden) {
        return;
    }

    node->hidden = hidden;
    g_signal_emit (node, signals[VISIBILITY_CHANGED], 0, hidden);
}

/* FIXME? hrn_cluster_node_flatten returns the tree in reverse order
   this is useful for the needs we have, but is it sane? */
void
hrn_cluster_node_flatten_reversed (HrnClusterNode *node,
                                   GList         **flat)
{
    if (node->children) {
        GSequenceIter *iter = g_sequence_get_begin_iter (node->children);

        while (!g_sequence_iter_is_end (iter)) {
            HrnClusterNode *child = (HrnClusterNode *) g_sequence_get (iter);

            hrn_cluster_node_flatten_reversed (child, flat);

            iter = g_sequence_iter_next (iter);
        }

    } else {
        *flat = g_list_prepend (*flat, node);
    }
}

void
hrn_cluster_node_set_children_hidden (HrnClusterNode *node,
                                      gboolean        hidden)
{
    GList *children = NULL, *c;

    hrn_cluster_node_flatten_reversed (node, &children);
    for (c = children; c; c = c->next) {
        HrnClusterNode *child_node = c->data;

        hrn_cluster_node_set_hidden (child_node, hidden);
    }
    g_list_free (children);
}

HrnClusterNode *
hrn_cluster_node_get_first_leaf (HrnClusterNode *parent)
{
    if (parent == NULL || parent->children == NULL) {
        return parent;
    } else {
        GSequenceIter *iter;
        HrnClusterNode *child;

        iter = g_sequence_get_begin_iter (parent->children);
        child = g_sequence_get (iter);

        return hrn_cluster_node_get_first_leaf (child);
    }
}

HrnClusterNode *
hrn_cluster_node_get_last_leaf (HrnClusterNode *parent)
{
    if (parent == NULL || parent->children == NULL) {
        return parent;
    } else {
        GSequenceIter *iter;
        HrnClusterNode *child;

        iter = g_sequence_get_end_iter (parent->children);
        iter = g_sequence_iter_prev (iter);
        child = g_sequence_get (iter);

        return hrn_cluster_node_get_last_leaf (child);
    }
}

HrnClusterNode *
hrn_cluster_node_get_next_leaf (HrnClusterNode *node)
{
    GSequenceIter *iter;

    if (node == NULL) {
        return NULL;
    }

    iter = g_sequence_iter_next (node->iter);
    if (g_sequence_iter_is_end (iter)) {
        HrnClusterNode *parent = node->parent;
        HrnClusterNode *next_child;

        if (parent == NULL) {
            return NULL;
        }

        next_child = hrn_cluster_node_get_next_leaf (parent);
        return next_child;
    } else {
        HrnClusterNode *child = g_sequence_get (iter);

        return hrn_cluster_node_get_first_leaf (child);
    }
}

HrnClusterNode *
hrn_cluster_node_get_previous_leaf (HrnClusterNode *node)
{
    GSequenceIter *iter;

    if (node == NULL) {
        return NULL;
    }

    iter = g_sequence_iter_prev (node->iter);
    if (g_sequence_iter_is_begin (node->iter)) {
        HrnClusterNode *parent = node->parent;
        HrnClusterNode *prev_child;

        if (parent == NULL) {
            return NULL;
        }

        prev_child = hrn_cluster_node_get_previous_leaf (parent);
        return prev_child;
    } else {
        HrnClusterNode *child = g_sequence_get (iter);

        return hrn_cluster_node_get_last_leaf (child);
    }
}
