/* 
 * File: xmms2ctrl.c
 *
 * GKrellM 2 Plugin to control XMMS2
 * Copyright (c) 2005 Johannes Heimansberg
 *
 * requires GKrellM 2.0.0 or better and XMMS2 DR1 or better
 *
 * Released under the GNU General Public License v2
 */

#include <xmmsclient/xmmsclient.h>
#include <xmmsclient/xmmsclient-glib.h>
#include "xmms2ctrl.h"

guint connection_error = 0;

guint xmms2ctrl_get_error(void)
{
	return connection_error;
}

void xmms2ctrl_set_error(guint error)
{
	connection_error = error;
}

static void n_check_for_error(xmmsc_result_t *res, void *error)
{
	guint e = GPOINTER_TO_INT(error);
	if (xmmsc_result_iserror(res)) {
		fprintf(stderr, "ERROR: %s\n", xmmsc_result_get_error(res));
		connection_error = e;
	} else
		connection_error = 0;
	xmmsc_result_unref(res);
}

static void n_player_status_play(xmmsc_result_t *res, void *connection)
{
	xmmsc_connection_t *conn = (xmmsc_connection_t *)connection;
	guint               status = -1;
	xmmsc_result_t     *res2;

	xmmsc_result_get_uint(res, &status);
	if (status == XMMS_PLAYBACK_STATUS_PLAY) {
		res2 = xmmsc_playback_tickle(conn);
		xmmsc_result_notifier_set(res2, n_check_for_error, 
		                          (gpointer)ERROR_CONNECTION);
		xmmsc_result_unref(res2);
	} else {
		res2 = xmmsc_playback_start(conn);
		xmmsc_result_notifier_set(res2, n_check_for_error,
	                          (gpointer)ERROR_CONNECTION);
		xmmsc_result_unref(res2);
	}
	xmmsc_result_unref(res);
}

void xmms2ctrl_play(xmmsc_connection_t *conn)
{
	xmmsc_result_t *res;

	res = xmmsc_playback_status(conn);
	xmmsc_result_notifier_set(res, n_player_status_play, conn);
	xmmsc_result_unref(res);
}

void xmms2ctrl_pause(xmmsc_connection_t *conn)
{
	xmmsc_result_t *res;

	res = xmmsc_playback_pause(conn);
	xmmsc_result_notifier_set(res, n_check_for_error,
	                          (gpointer)ERROR_CONNECTION);
	xmmsc_result_unref(res);
}

static void n_player_status_start_pb(xmmsc_result_t *res, void *connection)
{
	xmmsc_connection_t *conn = (xmmsc_connection_t *)connection;
	guint               status;
	
	xmmsc_result_get_uint(res, &status);
	if (status == XMMS_PLAYBACK_STATUS_PLAY)
		xmms2ctrl_pause(conn);
	else
		xmms2ctrl_play(conn);
	xmmsc_result_unref(res);
}

static void n_player_status(xmmsc_result_t *res, void *connection)
{
	xmmsc_connection_t *conn = (xmmsc_connection_t *)connection;
	guint              status = -1;
	
	xmmsc_result_get_uint(res, &status);
	if (status == XMMS_PLAYBACK_STATUS_PLAY)
		xmms2ctrl_pause(conn);
	else if (status == XMMS_PLAYBACK_STATUS_PAUSE)
		xmms2ctrl_play(conn);
	xmmsc_result_unref(res);
}

void xmms2ctrl_toggle_pause(xmmsc_connection_t *conn, gboolean start_playback)
{
	xmmsc_result_t *res;

	res = xmmsc_playback_status(conn);
	if (!start_playback)
		xmmsc_result_notifier_set(res, n_player_status, conn);
	else
		xmmsc_result_notifier_set(res, n_player_status_start_pb, conn);
	xmmsc_result_unref(res);
}

void xmms2ctrl_next(xmmsc_connection_t *conn)
{
	xmms2ctrl_do_reljump(conn, 1);
}

void xmms2ctrl_prev(xmmsc_connection_t *conn)
{
	xmms2ctrl_do_reljump(conn, -1);
}

void xmms2ctrl_stop(xmmsc_connection_t *conn)
{
	xmmsc_result_t *res;

	res = xmmsc_playback_stop(conn);
	xmmsc_result_notifier_set(res, n_check_for_error,
	                          (gpointer)ERROR_CONNECTION);
	xmmsc_result_unref(res);
}

static void n_tickle_on_success(xmmsc_result_t *res, void *connection)
{
	xmmsc_result_t     *res2;
	xmmsc_connection_t *conn = (xmmsc_connection_t *)connection;

	if (xmmsc_result_iserror(res)) {
		fprintf(stderr, "Couldn't advance in playlist: %s\n", 
		        xmmsc_result_get_error(res));
		connection_error = ERROR_ADVANCE;
	} else {
		res2 = xmmsc_playback_tickle(conn);
		xmmsc_result_notifier_set(res2, n_check_for_error, 
	                              (gpointer)ERROR_CONNECTION);
		xmmsc_result_unref(res2);
	}
	xmmsc_result_unref(res);
}

void xmms2ctrl_do_reljump(xmmsc_connection_t *conn, int where)
{
	xmmsc_result_t *res;

	res = xmmsc_playlist_set_next_rel(conn, where);
	xmmsc_result_notifier_set(res, n_tickle_on_success, conn);
	xmmsc_result_unref(res);
}

void xmms2ctrl_volume_set(xmmsc_connection_t *conn, gint volume)
{
	xmmsc_result_t *res;

	res = xmmsc_playback_volume_set (conn, "left", volume);
	xmmsc_result_unref(res);
	res = xmmsc_playback_volume_set (conn, "right", volume);
	xmmsc_result_unref(res);
}
