/*
 * Copyright (C) 1994,95,96 Eric M. Ludlam
 * Copyright (C) 1997 Free Software Foundation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 *
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 *
 * Please send bug reports, etc. to zappo@gnu.org.
 *
 * Purpose:
 *  To read a .etc.hosts file, which will contain the type of daemon
 * on the remote machine.
 *
 * Format is:
 * number.id machine.name daemon_type
 *
 * where:
 *  number.id    is not used, and can be the numeric address. "128.119.28.53"
 *  machine.name is required, and follows at least one space.
 *                  and is of the form "choochoo.dmc.com"
 *  daemon_type is the string GTALKD NTALKD or OTALKD,
 *                  where GTALKD has not been written,
 *                        NTALKD is BSD talk, and
 *                        OTALKD is the old version of talk found on suns.
 *
 * Also now reads an command init file ~/.etalkrc.  This has the same format 
 * as commands that are tped in.  Will most likely contain associations,
 * name preferences, and the like.
 *
 * $Log: gt_rc.c,v $
 * Revision 1.14  1997/12/14 19:16:42  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.13  1997/11/05 03:45:04  zappo
 * Changed name of SYSTEM_RC to SYSTEM_HOST_RC when adding system wide
 * rc file for etalk commands.
 *
 * Revision 1.12  1997/03/12 01:34:39  zappo
 * RC reader now removes blank lines and comments before it hits the interpreter.
 *
 * Revision 1.11  1996/03/02 03:19:14  zappo
 * Fixed some warnings
 *
 * Revision 1.10  1996/02/01  02:30:46  zappo
 * Updated for prototype changes
 *
 * Revision 1.9  1996/01/06  02:29:17  zappo
 * Modified call to gethome to include my username
 *
 * Revision 1.8  1995/12/10  00:14:09  zappo
 * Fixed some error messages, removed old DEBUG_2 stuff, and added a script
 * reader for etalk commands
 *
 * Revision 1.7  1995/09/22  14:02:49  zappo
 * Errors now call disp_message, and calls now take Ctxt
 *
 * Revision 1.6  1995/04/10  21:43:37  zappo
 * Modified parameters to strtok's for better compatibility with other
 * implementations.
 *
 * Revision 1.5  1995/03/25  03:36:44  zappo
 * Update copyright.
 *
 * Revision 1.4  1995/02/28  03:36:21  zappo
 * Added a new read command which will open, and load one specified file.
 *
 * Revision 1.3  1995/02/17  00:03:20  zappo
 * Local RC file can now take the environment variable TALK_HOSTS
 *
 * Revision 1.2  1995/01/28  16:06:35  zappo
 * Added sitecnfg.h to includes.
 *
 * Revision 1.1  1994/08/29  23:29:17  zappo
 * Initial revision
 *
 * ::Header:: gtalkc.h 
 */
#include "gtalklib.h"
#include "gtalkc.h"
#include "sitecnfg.h"

static char *names[] = { "OTALKD", "NTALKD", "GTALKD" };

/*
 * Function: parse_file
 *
 * Read in file stream, and just load in those host entities into the
 * host's table.
 * 
 * Parameters: file   - the file to parse from
 *             rctype - the type of RC file read from.
 * History:
 * eml 5/2/94
 */
static void parse_file(file, rctype)
     FILE           *file;
     enum RCFileType rctype;
{
  char  buffer[100];		/* buffer to read into               */
  char *s;			/* search pointer                    */
  char *hostname;		/* pointer to the hostname in buffer */
  char *daemon;			/* pointer to daemone type in buffer */
  int   i;			/* index                             */
  int   linenum = 0;		/* currently read line number        */
  struct HostObject *host;	/* new host to be created            */
  
  while(fgets(buffer, 100, file))
    {
      linenum++;
      s = buffer;		/* initialize parts.                 */
      hostname = NULL;
      daemon = NULL;

      if(*s == '#')
	{
	  continue;	/* # means comment.                  */
	}
      /* Pass over the first string (the number part)
       */
      
      if(!strtok(buffer, " \t\n"))
	{
	  continue;	/* EOL */
	}
      
      /* Save this place, and read over it, and place a NULL
       */
      hostname = strtok(NULL, " \t\n");
      daemon = strtok(NULL, " \t\n");

      /*
       * if we don't have hostname or daemon types, then there is no reason
       * to load that host in.
       */
      if(hostname && daemon)
	{
	  /*
	   * If the host already exists, then that host is returned by
	   * this function, therefore, changes in daemon type will hold.
	   */
	  host = HOST_gen_host( hostname );

	  if(host)
	    {
	      i = 0;
	      while(i < (sizeof(names) / (sizeof(char *))))
		{
		  if(!strcmp(daemon, names[i]))
		    {
		      host->type = i;
		      host->rc   = rctype;
		      break;
		    }
		  i++;
		}
	      if(i ==  (sizeof(names) / (sizeof(char *))))
		{
		  fprintf(stderr, "Line %d: Daemon type %s is unknown.\n", 
			  linenum, daemon);
		}
	    }
	  else
	    {
	      fprintf(stderr, "Line %d: Host %s is unknown.\n",
		      linenum, hostname);
	    }
	}
      else if(verbose)
	{
	  if(hostname)
	    {
	      fprintf(stderr, "Line %d: Incomplete line encountered.\n", 
		      linenum);
	    }
	}
    }
}


/*
 * Function: RC_load_file
 *
 *   Loads in file specified by the parameter, and reads it into the
 * list of known hosts.
 *
 * Returns:     int  - 
 * Parameters:  filename - Name of the file to read
 *
 * History:
 * zappo   2/27/95    Created
 */
int RC_load_file(Ctxt, filename)
     struct TalkContext *Ctxt;
     char *filename;
{
  FILE *stream;			/* open file stream    */

  if(verbose)
    printf("Reading hosts from %s.\n", filename);

  stream = fopen(filename, "ra");
  if(stream)
    {
      parse_file(stream, RC_system);
      fclose(stream);
      return Success;
    }
  else
    {
      char buff[100];
      sprintf(buff, "Could not open file \"%s\"", filename);
      DISP_message(Ctxt, buff);
      return Fail;
    }
}

/*
 * Function: RC_load_all_hosts
 *
 * Read in every host in the available hosts RC files, and define them.
 * These machines should be small in number, and represent those machines
 * which are different only.  Normal machines need no such special treatment.
 * 
 * Parameters: None
 *
 * History:
 * eml 5/2/94
 */
void RC_load_all_hosts(Ctxt)
     struct TalkContext *Ctxt;
{
  FILE *stream;			/* open file stream    */
#ifdef LOCAL_RC
  char *fileenv;
  char  lfile[100];
#endif

#ifdef SYSTEM_RC
  if(verbose)
    printf("Reading hosts from system file.\n");

  stream = fopen(SYSTEM_HOST_RC, "ra");
  if(stream)
    {
      parse_file(stream, RC_system);
      fclose(stream);
    }
#endif /* system_rc */

#ifdef LOCAL_RC
  if(verbose)
    printf("Reading hosts from user file.\n");

  fileenv = getenv (LOCAL_RC_ENV);

  if(!fileenv)
    {
      char *home;
      home = GTL_gethome(Ctxt->myname, TRUE);
      if(home)
	{
	  strcpy(lfile, home);
	  if(lfile[strlen(lfile)] != '/')
	    {
	      strcat(lfile, "/");
	    }
	  strcat(lfile, LOCAL_RC);

	  stream = fopen(lfile, "ra");
	}
    }
  else
    {
      stream = fopen(fileenv, "ra");
    }
  if(stream)
    {
      parse_file(stream, RC_local);
      fclose(stream);
    }
  else if(verbose)
    {
      fprintf(stderr, "Failed to open host RC file %s\n",
	      fileenv?fileenv:lfile);
    }
#endif /* local_rc */
    return;
}


/*
 * Function: RC_read_script
 *
 *   Reads from a script file of etalk commands.  Things like "show X"
 * when in interface mode are, by definition, not wise, but worry
 * about this later.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt  - Context
 *              fname - Name of
 * History:
 * zappo   11/19/95   Created
 */
int RC_read_script(Ctxt, fname)
     struct TalkContext *Ctxt;
     char *fname;
{
  char buffer[200];
  FILE *f;
  char *ret;
  
  if(!fname)
    {
      /* Assume the init-script */
      char *env;
      env = getenv(SCRIPT_RC_ENV);
      if(env)
	fname = env;
      else
	{
	  char *path;
	  fname = buffer;
	  path = GTL_gethome(Ctxt->myname, TRUE);
	  strcpy(fname, path);
	  strcat(fname, "/");
	  strcat(fname, SCRIPT_RC);
	}
    }

  /* Don't use fname after this point */
  f = fopen(fname, "r");

  if(!f)
    {
      return Fail;
    }

  do {
    ret = fgets(buffer, sizeof(buffer), f);

    if(ret)
      {
	if((buffer[0] != '#') && (buffer[0] != '\n'))
	  {
	    GTC_interpret_string(Ctxt, buffer);
	  }
      }
  } while(ret);
  
  fclose(f);

  return Success;
}
