/* GStreamer Element
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

/**
 * SECTION:element-rbswap
 *
 * <refsect2>
 * <para>
 * Swaps red (Cr) and blue (Cb) in video frames.
 * This should rarely be needed, but may be required in unusual cases,
 * e.g. if people have blue faces.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * It is akin to transcode's rgbswap [Thomas Oestreich]
 * </listitem>
 * <listitem>
 * Similar basic operation also available in e.g. mplayer's swapuv, etc
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-tc.h"

#include <string.h>


#define GST_TYPE_RB_SWAP \
  (gst_rb_swap_get_type())
#define GST_RB_SWAP(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_RB_SWAP,GstRbSwap))
#define GST_RB_SWAP_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_RB_SWAP,GstRbSwapClass))
#define GST_IS_RB_SWAP(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_RB_SWAP))
#define GST_IS_RB_SWAP_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_RB_SWAP))


typedef struct _GstRbSwap GstRbSwap;
typedef struct _GstRbSwapClass GstRbSwapClass;

struct _GstRbSwap
{
  GstVideoFilter videofilter;

  gint width, height;

  /* properties */
  guint delay;
};


struct _GstRbSwapClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (rb_swap_debug);
#define GST_CAT_DEFAULT rb_swap_debug


/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_DELAY
      /* FILL ME */
};

static GstStaticPadTemplate gst_rb_swap_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstStaticPadTemplate gst_rb_swap_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );


static GstCaps *gst_rb_swap_transform_caps (GstBaseTransform * trans,
    GstPadDirection direction, GstCaps * caps);
static GstFlowReturn gst_rb_swap_transform_ip (GstBaseTransform * btrans,
    GstBuffer * in);

static void gst_rb_swap_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_rb_swap_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstRbSwap, gst_rb_swap, GstVideoFilter, GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_GET_UNIT_SIZE_BOILERPLATE (gst_rb_swap);

static void
gst_rb_swap_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Rbswap",
      "Filter/Effect/Video", "Swap red and blue components",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_rb_swap_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_rb_swap_src_template));
}

static void
gst_rb_swap_class_init (GstRbSwapClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (rb_swap_debug, "rbswap", 0, "rbswap");

  gobject_class->set_property = gst_rb_swap_set_property;
  gobject_class->get_property = gst_rb_swap_get_property;

  trans_class->get_unit_size = GST_DEBUG_FUNCPTR (gst_rb_swap_get_unit_size);
  trans_class->transform_caps = GST_DEBUG_FUNCPTR (gst_rb_swap_transform_caps);
  /* dummy seems needed */
  trans_class->transform_ip = GST_DEBUG_FUNCPTR (gst_rb_swap_transform_ip);
}

static void
gst_rb_swap_init (GstRbSwap * filter, GstRbSwapClass * g_class)
{

}

static GstCaps *
gst_rb_swap_transform_caps (GstBaseTransform * trans,
    GstPadDirection direction, GstCaps * caps)
{
  GstCaps *ret;
  GstStructure *structure;
  guint32 fourcc;

  GST_DEBUG_OBJECT (trans, "receiving caps: %" GST_PTR_FORMAT, caps);

  ret = gst_caps_copy (caps);

  /* this function is always called with a simple caps */
  g_return_val_if_fail (GST_CAPS_IS_SIMPLE (ret), NULL);

  structure = gst_caps_get_structure (ret, 0);

  gst_structure_get_fourcc (structure, "format", &fourcc);
  switch (fourcc) {
    case GST_MAKE_FOURCC ('I', '4', '2', '0'):
    case GST_MAKE_FOURCC ('I', 'Y', 'U', 'V'):
      GST_DEBUG ("I420");
      fourcc = GST_MAKE_FOURCC ('Y', 'V', '1', '2');
      break;
    case GST_MAKE_FOURCC ('Y', 'V', '1', '2'):
      GST_DEBUG ("YV12");
      fourcc = GST_MAKE_FOURCC ('I', '4', '2', '0');
      break;
    default:
      g_return_val_if_reached (NULL);
      break;
  }

  gst_structure_set (structure, "format", GST_TYPE_FOURCC, fourcc, NULL);

  GST_DEBUG_OBJECT (trans, "returning caps: %" GST_PTR_FORMAT, ret);

  return ret;
}

static GstFlowReturn
gst_rb_swap_transform_ip (GstBaseTransform * btrans, GstBuffer * in)
{
  return GST_FLOW_OK;
}

static void
gst_rb_swap_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstRbSwap *src;

  g_return_if_fail (GST_IS_RB_SWAP (object));
  src = GST_RB_SWAP (object);

  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_rb_swap_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstRbSwap *src;

  g_return_if_fail (GST_IS_RB_SWAP (object));
  src = GST_RB_SWAP (object);

  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
