/* GStreamer Element
 * Copyright (C) 2008 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * avidemux filter:
 * Copyright (C) Mean - 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-stabilize
 *
 * <refsect2>
 * <para>
 * Light-weight denoising/smoothing using previous and next frames.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * avidemux stabilize filter [Mean]
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-ad.h"


#define GST_TYPE_STABILIZE \
  (gst_stabilize_get_type())
#define GST_STABILIZE(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_STABILIZE,GstStabilize))
#define GST_STABILIZE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_STABILIZE,GstStabilizeClass))
#define GST_IS_STABILIZE(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_STABILIZE))
#define GST_IS_STABILIZE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_STABILIZE))

typedef struct _GstStabilize GstStabilize;
typedef struct _GstStabilizeClass GstStabilizeClass;


struct _GstStabilize
{
  GstVideoFilter videofilter;

  gint width, height;

  /* past/future buffers */
  GstBuffer *queue[3];
  guint count;

  /* stabilize parameters */
  guint luma_threshold;
};

struct _GstStabilizeClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (smooth_debug);
#define GST_CAT_DEFAULT smooth_debug

/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_LUMA_THRESHOLD
      /* FILL ME */
};

#define DEFAULT_LUMA_THRESHOLD     30

static GstStaticPadTemplate gst_stabilize_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstStaticPadTemplate gst_stabilize_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static gboolean gst_stabilize_hook_caps (GstStabilize * filter,
    GstCaps * incaps, GstCaps * outcaps);
static gboolean gst_stabilize_sink_event (GstBaseTransform * btrans,
    GstEvent * event);
static GstFlowReturn gst_stabilize_transform (GstBaseTransform * btrans,
    GstBuffer * in, GstBuffer * out);
static gboolean gst_stabilize_start (GstBaseTransform * btrans);
static gboolean gst_stabilize_stop (GstBaseTransform * btrans);

static void gst_stabilize_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_stabilize_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstStabilize, gst_stabilize, GstVideoFilter,
    GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE_FULL (GstStabilize, gst_stabilize,
    gst_stabilize_hook_caps);

GST_VIDEO_FILTER_GET_UNIT_SIZE_BOILERPLATE (gst_stabilize);

static void
gst_stabilize_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Stabilize",
      "Filter/Effect/Video", "Stabilizing",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n" "Mean");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_stabilize_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_stabilize_src_template));
}

static void
gst_stabilize_class_init (GstStabilizeClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (smooth_debug, "smooth", 0, "smooth");

  gobject_class->set_property = gst_stabilize_set_property;
  gobject_class->get_property = gst_stabilize_get_property;

  g_object_class_install_property (gobject_class, PROP_LUMA_THRESHOLD,
      g_param_spec_uint ("luma-threshold", "Luma Threshold", "Luma Threshold",
          1, 255, DEFAULT_LUMA_THRESHOLD,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_stabilize_set_caps);
  trans_class->get_unit_size = GST_DEBUG_FUNCPTR (gst_stabilize_get_unit_size);
  trans_class->transform = GST_DEBUG_FUNCPTR (gst_stabilize_transform);
  trans_class->event = GST_DEBUG_FUNCPTR (gst_stabilize_sink_event);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_stabilize_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_stabilize_stop);
}

static void
gst_stabilize_init (GstStabilize * filter, GstStabilizeClass * g_class)
{
  filter->luma_threshold = DEFAULT_LUMA_THRESHOLD;
}

// TODO consider this as a library function
/* empties queue and resets frame count */
static void
gst_stabilize_flush (GstStabilize * filter, gboolean send)
{
  GstBuffer *buf;
  GstBaseTransform *btrans;
  gint i;

  btrans = GST_BASE_TRANSFORM (filter);

  filter->count = 0;
  for (i = 2; i >= 0; i--) {
    buf = filter->queue[i];
    if (buf) {
      filter->queue[i] = NULL;
      if (send) {
        gst_buffer_set_caps (buf, GST_PAD_CAPS (btrans->srcpad));
        gst_pad_push (btrans->srcpad, buf);
      } else
        gst_buffer_unref (buf);
    }
  }
}

static gboolean
gst_stabilize_hook_caps (GstStabilize * filter, GstCaps * incaps,
    GstCaps * outcaps)
{
  gst_stabilize_flush (filter, TRUE);

  return TRUE;
}

static gboolean
gst_stabilize_sink_event (GstBaseTransform * btrans, GstEvent * event)
{
  GstStabilize *filter;

  filter = GST_STABILIZE (btrans);

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_EOS:
      gst_stabilize_flush (filter, TRUE);
      break;
    default:
      break;
  }

  return GST_BASE_TRANSFORM_CLASS (parent_class)->event (btrans, event);
}


/* see below for specific (loop) context */
#define PONDERATE(x) G_STMT_START { \
      if (distMatrix[*incur][x] < threshold) { \
        c += x; \
        coeff++; \
      } \
    } G_STMT_END

static void
gst_stabilize (guint8 * src, guint8 * dst, guint8 * src_prev, guint8 * src_next,
    gint width, gint height, gint threshold)
{
  guint8 *inprev, *innext, *incur, *nl, *pl, *nc, *pc, *zout;
  guint x, y;

  inprev = src_prev + width + 1;
  innext = src_next + width + 1;
  incur = src + width + 1;
  zout = dst + width + 1;

  for (y = height - 1; y > 1; y--) {
    nl = incur + width;
    pl = incur - width;
    nc = incur + 1;
    pc = incur - 1;

    for (x = width - 1; x > 1; x--) {
      guint16 c, coeff;

      c = *incur * 4;
      coeff = 4;
      PONDERATE (*innext);
      PONDERATE (*inprev);
      PONDERATE (*pc);
      PONDERATE (*nc);
      PONDERATE (*nl);
      PONDERATE (*pl);

      *zout = (c * fixMul[coeff]) >> 16;
      zout++;
      incur++;
      innext++;
      inprev++;
      nl++;
      pl++;
      nc++;
      pc++;
    }

    /* proceed to next line */
    zout += 2;
    incur += 2;
    innext += 2;
    inprev += 2;
  }
}

static GstFlowReturn
gst_stabilize_transform (GstBaseTransform * btrans, GstBuffer * in,
    GstBuffer * out)
{
  GstStabilize *filter;
  guint8 *src, *dest;

  gst_object_sync_values (G_OBJECT (btrans), GST_BUFFER_TIMESTAMP (in));

  filter = GST_STABILIZE (btrans);

  filter->queue[filter->count] = gst_buffer_ref (in);
  filter->count++;

  if (filter->count < 3)
    return GST_BASE_TRANSFORM_FLOW_DROPPED;

  src = (guint8 *) GST_BUFFER_DATA (filter->queue[1]);
  dest = (guint8 *) GST_BUFFER_DATA (out);

  oil_memcpy (dest, src, GST_VIDEO_I420_SIZE (filter->width, filter->height));

  /* only luma is processed */
  gst_stabilize (src, dest, GST_BUFFER_DATA (filter->queue[0]),
      GST_BUFFER_DATA (filter->queue[1]),
      GST_VIDEO_I420_Y_ROWSTRIDE (filter->width), filter->height,
      filter->luma_threshold);

  gst_buffer_unref (filter->queue[0]);
  filter->queue[0] = filter->queue[1];
  filter->queue[1] = filter->queue[2];
  filter->queue[2] = NULL;
  filter->count--;

  return GST_FLOW_OK;
}

static gboolean
gst_stabilize_start (GstBaseTransform * btrans)
{

  return TRUE;
}

static gboolean
gst_stabilize_stop (GstBaseTransform * btrans)
{
  GstStabilize *filter = GST_STABILIZE (btrans);

  gst_stabilize_flush (filter, FALSE);

  return TRUE;
}

static void
gst_stabilize_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstStabilize *src;

  g_return_if_fail (GST_IS_STABILIZE (object));
  src = GST_STABILIZE (object);

  switch (prop_id) {
    case PROP_LUMA_THRESHOLD:
      src->luma_threshold = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_stabilize_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstStabilize *src;

  g_return_if_fail (GST_IS_STABILIZE (object));
  src = GST_STABILIZE (object);

  switch (prop_id) {
    case PROP_LUMA_THRESHOLD:
      g_value_set_uint (value, src->luma_threshold);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
