/* GStreamer video utils
 * Copyright (C) <2006> Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

/**
 * SECTION:video-utils
 * @short_description: Useful boilerplate for video filters
 */

#ifndef __GST_VIDEO_UTILS_H__
#define __GST_VIDEO_UTILS_H__


G_BEGIN_DECLS

/**
 * GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE_FULL:
 * @type:             the name of the type struct
 * @type_as_function: the prefix for the functions
 * @hook_function:    a function with #GstBaseTransform set_caps signature
 *                    that can extract custom additional info or accept/reject caps
 *
 * Define a typical set_caps function for a video filter,
 * allowing for additional custom verification and initialization.
 */
#define GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE_FULL(type, type_as_function, hook_function)  \
static gboolean \
type_as_function ## _set_caps (GstBaseTransform * btrans, GstCaps * incaps, \
    GstCaps * outcaps) \
{ \
  type *filter = (type *) (btrans); \
  GstStructure *structure;  \
  gboolean ret = FALSE; \
\
  structure = gst_caps_get_structure (incaps, 0); \
\
  if (gst_structure_get_int (structure, "width", &filter->width) && \
      gst_structure_get_int (structure, "height", &filter->height)) { \
    ret = hook_function (filter, incaps, outcaps); \
  } \
\
  return ret;\
}


#define __GST_NO_HOOK(filter, incaps, outcaps)  TRUE

/**
 * GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE:
 * @type:             the name of the type struct
 * @type_as_function: the prefix for the functions
 *
 * Define a typical set_caps function for a video filter.
 */
#define GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE(type, type_as_function)  \
  GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE_FULL(type, type_as_function, \
    __GST_NO_HOOK)


/**
 * GST_VIDEO_FILTER_GET_UNIT_SIZE_BOILERPLATE:
 * @type_as_function: the prefix for the functions
 *
 * Define a typical get_unit_size function for a video filter that provides
 * proper unit size for x-raw-rgb and some standard x-raw-yuv
 * colorspaces (YUY2, YUYV, YVYU, IYUV, I420, YV12).
 */
#define GST_VIDEO_FILTER_GET_UNIT_SIZE_BOILERPLATE(type_as_function)  \
static gboolean \
type_as_function ## _get_unit_size (GstBaseTransform * btrans, GstCaps * caps, \
    guint * size) \
{ \
  GstStructure *structure; \
  gboolean ret = FALSE; \
  gint width, height; \
 \
  structure = gst_caps_get_structure (caps, 0); \
 \
  if (gst_structure_get_int (structure, "width", &width) && \
      gst_structure_get_int (structure, "height", &height)) { \
    if (gst_structure_has_name (structure, "video/x-raw-rgb")) { \
      gint bpp; \
 \
      if (gst_structure_get_int (structure, "bpp", &bpp)) { \
        ret = TRUE; \
        *size = width * height * bpp / 8; \
      } \
    } \
    else { \
      guint32 fourcc; \
 \
      if (gst_structure_get_fourcc (structure, "format", &fourcc)) { \
        ret = TRUE; \
        switch (fourcc) { \
          case GST_MAKE_FOURCC ('Y', 'U', 'Y', '2'): \
          case GST_MAKE_FOURCC ('Y', 'U', 'Y', 'V'): \
          case GST_MAKE_FOURCC ('Y', 'V', 'Y', 'U'): \
            *size = width * height * 2; \
            break; \
          case GST_MAKE_FOURCC ('I', 'Y', 'U', 'V'): \
          case GST_MAKE_FOURCC ('I', '4', '2', '0'): \
          case GST_MAKE_FOURCC ('Y', 'V', '1', '2'): \
          default: \
            *size = GST_VIDEO_I420_SIZE (width, height); \
        } \
      } \
    } \
    GST_DEBUG_OBJECT (btrans, "our frame size is %d bytes (%dx%d)", *size, \
        width, height); \
  } \
 \
  return ret; \
}

G_END_DECLS

#endif /* __GST_VIDEO_UTILS_H__ */
