//*****************************************************************************
//                                ConvertType.hpp                             *
//                               -----------------                            *
//  Started     : 21/09/2004                                                  *
//  Last Update : 19/02/2010                                                  *
//  Copyright   : (C) 2004 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "utility/ConvertType.hpp"

//*****************************************************************************
// Allocate storage for static data members.

int  ConvertType::m_iFltRes = CNVTYPE_DFLT_FLTRES;

//*****************************************************************************
// Constructor.

ConvertType::ConvertType( void )
{
}

//*****************************************************************************
// Destructor.

ConvertType::~ConvertType( )
{
}

//*****************************************************************************
// Set the resolution of conversions from floats to strings ie. the number of
// digits after the decimal point.
//
// Argument List :
//   iFltRes - The desired resolution
//
// Return Values :
//   Success
//   Failure

bool  ConvertType::bSetFltRes( int iFltRes )
{
  if( iFltRes<CNVTYPE_MIN_FLTRES || iFltRes>CNVTYPE_MAX_FLTRES )
    return( FALSE );

  m_iFltRes = iFltRes;

  return( TRUE );
}

//*****************************************************************************
// Extract the numeric part of a string.
// Eg. 10.0E-2uVolt becomes 10.0E-8 and 0.0E-2uF becomes 0.0E-8.
//
// Argument List :
//   rosNum - The string containing a numeric value
//
// Return Values :
//   Success - The numeric part of the string
//   Failure - An empty string

wxString & ConvertType::rosGetNum( const wxString & rosNum )
{
  static  wxString  osNum;
  wxString  osInt;   // The integer part of the number
  wxString  osFrac;  // The part of the string after a decimal point
  wxString  osExp;   // The exponent part of the string
  wxChar    oc1;
  size_t    sz1;
  long      li1;

  // Clear the local static variable
  osNum.Empty( );

  // Basic argument test
  if( rosNum.IsEmpty( ) ) return( osNum );

  // Extract the first non-space character
  for( sz1=0; sz1<rosNum.Length( ); sz1++ )
    if( (oc1=rosNum.GetChar( sz1 )) != wxT(' ') )
      break;
  if( !wxIsdigit( oc1 ) && oc1!=wxT('+') && oc1!=wxT('-') && oc1!=wxT('.') )
    return( osNum );

  // Extract the first part of the number
  osInt.Empty( );
  if( oc1 != wxT('.') ) osInt << oc1;
  for( ++sz1; sz1<rosNum.Length( ); sz1++ )
  {
    oc1 = rosNum.GetChar( sz1 );
    if( ! wxIsdigit( oc1 ) ) break;
    osInt << oc1;
  }

  // Extract the part of the string after a decimal point
  osFrac.Empty( );
  if( oc1 == wxT('.') )
  {
    for( ++sz1; sz1<rosNum.Length( ); sz1++ )
    {
      oc1 = rosNum.GetChar( sz1 );
      if( ! wxIsdigit( oc1 ) ) break;
      osFrac << oc1;
    }
  }

  // Extract the part of the string after an exponent
  osExp.Empty( );
  if( oc1==wxT('e') || oc1==wxT('E') )
  {
    for( ++sz1; sz1<rosNum.Length( ); sz1++ )
    {
      oc1 = rosNum.GetChar( sz1 );

      if( osExp.IsEmpty( ) )
        if( oc1==wxT('+') || oc1==wxT('-') )
          { osExp << oc1; continue; }

      if( ! wxIsdigit( oc1 ) ) break;

      osExp << oc1;
    }
  }

  // Check for a units specifier
  if( ! osExp.ToLong( &li1 ) ) li1 = 0;
  switch( oc1 )
  {
    case wxT('T') : li1 += 12; break;
    case wxT('G') : li1 +=  9; break;
    case wxT('M') : li1 +=  6; break;
    case wxT('K') :
    case wxT('k') : li1 +=  3; break;
    case wxT('m') : li1 -=  3; break;
    case wxT('u') : li1 -=  6; break;
    case wxT('n') : li1 -=  9; break;
    case wxT('p') : li1 -= 12; break;
    case wxT('f') : li1 -= 15; break;
    default       :            break;
  }
  if( li1 != 0 ) osExp = wxString::Format( wxT("%i"), (int) li1 );

  // Construct the number string
  osNum << osInt;
  if( ! osFrac.IsEmpty( ) ) osNum << wxT('.') << osFrac;
  if( ! osExp .IsEmpty( ) ) osNum << wxT('E') << osExp;

  return( osNum );
}

//*****************************************************************************
// Convert a string into a double float, round it and then convert it to a
// long integer.
//
// Some examples of strings that can be converted to a float:
//   * 12
//   * 12k  ("k" could alternatively be T, Tera, G, Giga, M, Meg, Mega, K)
//   * 12kV ("V" could alternatively be Ohm, F, H, A, Sec and is ignored)
//
// Argument List :
//   rosNum - The string value
//   pliNum - The equivalent long integer
//
// Return Values :
//   Success - TRUE  (part or all of the string was converted)
//   Failure - FALSE (no part of the string could be converted)

bool  ConvertType::bStrToLong( const wxString & rosNum, long * pliNum )
{
  double  df1;

  if( ! bStrToDFlt( rosNum, &df1 ) ) return( FALSE );
  *pliNum = lround( df1 );

  return( TRUE );
}

//*****************************************************************************
// Convert a string into a double floating point number.
//
// Some examples of strings that can be converted to a float :
//   * 12.3
//   * 12.3m  ("m" could also be T, Tera, G, Giga, M, Meg, Mega, K, k, u, n, p, f)
//   * 12.3mF ("F" could also be Ohm, H, V, A, Sec and is ignored)
//   * 12.3E-3
//
// Argument List :
//   rosNum - The string to be converted
//   pdfNum - The equivalent double float
//
// Return Values :
//   Success - TRUE  (part or all of the string was converted)
//   Failure - FALSE (no part of the string could be converted)

bool  ConvertType::bStrToDFlt( const wxString & rosNum, double * pdfNum )
{
  wxString  osNum;

  osNum = rosGetNum( rosNum );
  if( ! osNum.ToDouble( pdfNum ) ) return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Convert a double floating point number into a fixed format string.
// The format used by this function is as follows :
//    1.234E-05
//   -1.234E+05
//
// Argument List :
//   dfNum  - The double float value
//   rosNum - The string to hold the result
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  ConvertType::bDFltToStr( double dfNum, wxString & rosNum )
{
  wxString  osFmt;

  osFmt = wxString::Format( wxT("%% %i.%iE"), m_iFltRes+2, m_iFltRes );

  if( rosNum.Printf( osFmt, dfNum ) < 0 ) return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Convert a double floating point value to a string using engineering format.
// (Eg. convert 100000 to 100k.)
//
// Argument List :
//   dfNum  - The double float value
//   rosNum - The string to hold the result
//
// Return Values :
//   Success - The string containing the engineering format
//   Failure - An empty string

bool  ConvertType::bDFltToStrEng( double dfNum, wxString & rosNum )
{
  wxString  osFmt;
  float     fMan;  // Mantissa
  int       iExp;  // Exponent
  int       i1;

  // Setup the format string based on the desired resolution
  osFmt = wxString::Format( wxT("%%%i.%if"), m_iFltRes+2, m_iFltRes );

  // Parse the number to be converted
  if( ! bParseFlt( dfNum, &fMan, &iExp ) ) return( FALSE );

  // Round the mantissa based on the desired resolution
  fMan *= pow( 10.0, (float) m_iFltRes );
  fMan = roundf( fMan );
  fMan /= pow( 10.0, (float) m_iFltRes );

  // Scale the mantissa so that the exponent is a multiple of 3
  while( iExp % 3 )
  {
    iExp--;
    fMan *= 10.0;
  }

  // Determine the appropriate scaling factor to be appended as units
  switch( iExp )
  {
    case  12 : osFmt << wxT("Tera");  break;
    case   9 : osFmt << wxT("Giga");  break;
    case   6 : osFmt << wxT("Meg");   break;
    case   3 : osFmt << wxT("K");     break;
    case   0 : osFmt << wxT("");      break;
    case  -3 : osFmt << wxT("m");     break;
    case  -6 : osFmt << wxT("u");     break;
    case  -9 : osFmt << wxT("n");     break;
    case -12 : osFmt << wxT("p");     break;
    case -15 : osFmt << wxT("f");     break;
    default  : osFmt << wxT("E%02i"); break;
  }

  // Create the string representation of the number
  i1 = rosNum.Printf( osFmt, fMan, iExp );

  return( i1<0 ? FALSE : TRUE );
}

//*****************************************************************************
// Parse a double float value into it's mantissa and exponent.
//
// Argument List :
//   dfNum - The double float value to be parsed
//   pfMan - A pointer to float   to hold the mantissa (1.000 to 9.999)
//   piExp - A pointer to integer to hold the exponent
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  ConvertType::bParseDFlt( double dfNum, float * pfMan, int * piExp )
{
  if( dfNum<-FLT_MAX || dfNum>FLT_MAX ) return( FALSE );

  if( dfNum != 0.0 )
  {
    *piExp = (int) floor( log10( fabs( dfNum ) ) );
    *pfMan = dfNum / pow( 10.0, *piExp );

    if( fabs(*pfMan) <   1.0 ) { (*pfMan) *= 10.0; (*piExp)--; }
    if( fabs(*pfMan) >= 10.0 ) { (*pfMan) /= 10.0; (*piExp)++; }
  }
  else
  {
    *pfMan = 0.0;
    *piExp = 0;
  }

  return( TRUE );
}

//*****************************************************************************
// Convert a string into an integer.
//
// Argument List :
//   rosNum - The string to be converted
//   piNum  - A pointer to an integer to hold the conversion
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  ConvertType::bStrToInt( const wxString & rosNum, int * piNum )
{
  bool  bRtn;
  long  li1 = (long) *piNum;

  bRtn = bStrToLong( rosNum, &li1 );
  *piNum = (int) li1;

  return( bRtn );
}

//*****************************************************************************
// Convert a string into a float.
//
// Argument List :
//   rosNum - The string to be converted
//   piNum  - A pointer to a float to hold the conversion
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  ConvertType::bStrToFlt( const wxString & rosNum, float * pfNum )
{
  bool    bRtn;
  double  df1 = (double) *pfNum;

  bRtn = bStrToDFlt( rosNum, &df1 );
  *pfNum = (float) df1;

  return( bRtn );
}

//*****************************************************************************
// Convert a float to a string.
//
// Argument List :
//   fNum   - The float to be converted
//   rosNum - A reference to a string to hold the conversion
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  ConvertType::bFltToStr( float fNum, wxString & rosNum )
{
  double  df1 = (double) fNum;

  return( bDFltToStr( df1, rosNum ) );
}

//*****************************************************************************
// Convert a floating point value to a string using engineering format.
// (Eg. convert 100000 to 100k.)
//
// Argument List :
//   fNum   - The float to be converted
//   rosNum - A reference to a string to hold the conversion
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  ConvertType::bFltToStrEng( float fNum, wxString & rosNum )
{
  double  df1 = (double) fNum;

  return( bDFltToStrEng( df1, rosNum ) );
}

//*****************************************************************************
// Parse a float value into it's mantissa and exponent.
//
// Argument List :
//   fNum  - The float value to be parsed
//   pfMan - A pointer to float   to hold the mantissa (1.000 to 9.999)
//   piExp - A pointer to integer to hold the exponent
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  ConvertType::bParseFlt( float fNum, float * pfMan, int * piExp )
{
  double  df1 = (double) fNum;

  return( bParseDFlt( df1, pfMan, piExp ) );
}

//*****************************************************************************
//                                                                            *
//                                 Test Utility                               *
//                                                                            *
//*****************************************************************************

#ifdef TEST_CONVERTTYPE

// Function prototypes

void  Usage( char * psAppName );

//*****************************************************************************

int  main( int argc, char * argv[ ] )
{
  wxString  osNum, os1;
  double    dfNum;
  long      liNum;
  float     fNum;
  int       iNum;
  float     fMan;
  int       iExp;

  // Validate the argument count passed to the application
  if( argc > 2 )           { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }

  // Process the command line arguments
  os1 = wxConvLibc.cMB2WC( argv[ 1 ] );
  if( argc > 1 )
  {
    if( os1 == wxT("-h") ) { Usage( argv[ 0 ] ); exit( EXIT_SUCCESS ); }
    else                   { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }
  }

  // Display the utility banner
  cout << "\n  ConvertType Class Test Utility"
       << "\n    Version 1.05 (19/02/2010)\n\n";

  cout << "ConvertType::rosGetNum    ( ) : ";
  osNum = wxT("-10.0E-2uF");
  os1 = ConvertType::rosGetNum( osNum );
  cout << osNum.mb_str( ) << " -> " << os1.mb_str( ) << "\n";

  cout << "ConvertType::rosGetNum    ( ) : ";
  osNum = wxT("0.0E2uVolt");
  os1 = ConvertType::rosGetNum( osNum );
  cout << osNum.mb_str( ) << " -> " << os1.mb_str( ) << "\n";

  cout << "ConvertType::bStrToLong   ( ) : ";
  osNum = wxT("123.5V");
  if( ConvertType::bStrToLong( osNum, &liNum ) )         cout << "Success";
  else                                                   cout << "Failure";
  cout << " (" << osNum.mb_str( ) << " -> " << liNum << ")\n";

  cout << "ConvertType::bStrToDFlt   ( ) : ";
  osNum = wxT("123.4V");
  if( ConvertType::bStrToDFlt( osNum, &dfNum ) )         cout << "Success";
  else                                                   cout << "Failure";
  cout << " (" << osNum.mb_str( ) << " -> " << dfNum << ")\n";

  cout << "ConvertType::bDFltToStr   ( ) : ";
  dfNum = 1.23E+04;
  if( ConvertType::bDFltToStr( dfNum, osNum ) )          cout << "Success";
  else                                                   cout << "Failure";
  cout << " (" << dfNum << " -> " << osNum.mb_str( ) << ")\n";

  cout << "ConvertType::bDFltToStrEng( ) : ";
  dfNum = 1.23E+04;
  if( ConvertType::bDFltToStrEng( dfNum, osNum ) )       cout << "Success";
  else                                                   cout << "Failure";
  cout << " (" << dfNum << " -> " << osNum.mb_str( ) << ")\n";

  cout << "ConvertType::bParseDFlt   ( ) : ";
  dfNum = 12.3E-04;
  if( ConvertType::bParseDFlt( dfNum, &fMan, &iExp ) )   cout << "Success";
  else                                                   cout << "Failure";
  cout << " (" << dfNum << " -> " << fMan << "E" << iExp << ")\n";

  cout << "ConvertType::bStrToInt    ( ) : ";
  osNum = wxT("123.45E-5");
  if( ConvertType::bStrToInt( osNum, &iNum ) )           cout << "Success";
  else                                                   cout << "Failure";
  cout << " (" << osNum.mb_str( ) << " -> " << iNum << ")\n";

  cout << "ConvertType::bStrToFlt    ( ) : ";
  osNum = wxT("123.45E-5");
  if( ConvertType::bStrToFlt( osNum, &fNum ) )           cout << "Success";
  else                                                   cout << "Failure";
  cout << " (" << osNum.mb_str( ) << " -> " << fNum << ")\n";

  cout << "ConvertType::bFltToStr    ( ) : ";
  fNum = 1.23E+04;
  if( ConvertType::bFltToStr( fNum, osNum ) )            cout << "Success";
  else                                                   cout << "Failure";
  cout << " (" << fNum << " -> " << osNum.mb_str( ) << ")\n";

  cout << "ConvertType::bFltToStrEng ( ) : ";
  fNum = 1.23E+04;
  if( ConvertType::bFltToStrEng( fNum, osNum ) )         cout << "Success";
  else                                                   cout << "Failure";
  cout << " (" << fNum << " -> " << osNum.mb_str( ) << ")\n";

  cout << "ConvertType::bParseFlt    ( ) : ";
  fNum = 12.3E-04;
  if( ConvertType::bParseFlt( fNum, &fMan, &iExp ) )     cout << "Success";
  else                                                   cout << "Failure";
  cout << " (" << fNum << " -> " << fMan << "E" << iExp << ")\n";

  cout << '\n';

  exit( EXIT_SUCCESS );
}

//*****************************************************************************

void  Usage( char * psAppName )
{
  cout << "\nUsage   : " << psAppName << " [-OPTIONS]"
       << "\nOptions :"
       << "\n  -h : Print usage (this message)\n";
}

#endif // TEST_CONVERTTYPE

//*****************************************************************************
