//*****************************************************************************
//                               PrcGnuCap.cpp                                *
//                              ---------------                               *
//  Started     : 01/09/2003                                                  *
//  Last Update : 07/01/2008                                                  *
//  Copyright   : (C) 2003 by M.S.Waters                                      *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "process/PrcGnuCap.hpp"

//*****************************************************************************
// Constructor.

PrcGnuCap::PrcGnuCap( void ) : PrcSimrBase( )
{
  // Set the simulation engine type
  m_eSimrType = eSIMR_GNUCAP;

  // Set the simulator binary file name if it can be found
  bSetBinary( wxT("gnucap") );

  m_osInput.Empty( );
}

//*****************************************************************************
// Destructor.

PrcGnuCap::~PrcGnuCap( )
{
}

//*****************************************************************************
// Parse a GENERATOR command and put the data in a simulation object.
//
// Argument List:
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bParseCmdGEN( wxString & rosCmd, Simulation & roSim )
{
  // ??? Not yet implemented

  return( FALSE );
}

//*****************************************************************************
// Parse a OP command and put the data in a simulation object.
// Eg.s: .OP 27.0 BASIC
//       .OP 27.0 60.0 BY 1.0 BASIC
//       .OP 27.0 60.0 TI 1.1 BASIC
//       .OP 27.0 60.0 DE 10 BASIC
//
// Argument List:
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bParseCmdOP( wxString & rosCmd, Simulation & roSim )
{
  wxStringTokenizer  ostk1;
  wxString           os1;
  bool               bRtnValue=TRUE;
  int                i1;
  double             df1;

  // Basic argument checks
  if( rosCmd.Upper( ).AfterLast( wxT(' ') ).IsSameAs( wxT("BASIC") ) )
       ostk1.SetString( rosCmd.BeforeLast( wxT(' ') ) );
  else ostk1.SetString( rosCmd );
  i1 = ostk1.CountTokens( );
  if( i1!=2 && i1!=5 )                        return( FALSE );

  // Check command type
  os1 = ostk1.GetNextToken( );
  if( os1.Upper( ) != wxT(".OP") )            return( FALSE );

  // Extract the start temperature
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) ) bRtnValue = FALSE;
  if( ! roSim.bSetSwpStart( (float) df1 ) )   bRtnValue = FALSE;
  if( ! ostk1.HasMoreTokens( ) )
  { // There are no more fields
    if( ! roSim.bSetSwpStop( (float) df1 ) )  bRtnValue = FALSE;
    if( ! roSim.bSetSwpStep( 1.0 ) )          bRtnValue = FALSE;
    roSim.bSetSwpScale( 0 );
    return( TRUE );
  }

  // Extract the stop temperature
  if( ostk1.CountTokens( ) != 3 )             bRtnValue = FALSE;
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) ) bRtnValue = FALSE;
  if( ! roSim.bSetSwpStop( (float) df1 ) )    bRtnValue = FALSE;

  // Extract the sweep type: linear or log
  os1 = ostk1.GetNextToken( );
  os1.MakeUpper( );
  if(      os1 == wxT("BY") ) roSim.bSetSwpScale( 0 );
  else if( os1 == wxT("TI") ) roSim.bSetSwpScale( 1 );
  else if( os1 == wxT("DE") ) roSim.bSetSwpScale( 2 );
  else                                        bRtnValue = FALSE;

  // Extract the step size
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) ) bRtnValue = FALSE;
  if( ! roSim.bSetSwpStep( (float) df1 ) )    bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Parse a DC command and put the data in a simulation object.
//
// Eg.s: .DC Vin 0.00 TE 27.00 BASIC
//       .DC Vin 0.00 100.00m BY 10.00m TE 27.00 BASIC
//       .DC Vin 0.00 100.00m TI 1.01 TE 27.00 BASIC
//       .DC Vin 0.00 100.00m DE 100 TE 27.00 BASIC
//
// Argument List:
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bParseCmdDC( wxString & rosCmd, Simulation & roSim )
{
  wxStringTokenizer  ostk1;
  wxString           os1;
  bool               bRtnValue=TRUE;
  int                i1;
  double             df1, df2;
  long               li1;

  // Basic argument checks
  if( rosCmd.Upper( ).AfterLast( wxT(' ') ).IsSameAs( wxT("BASIC") ) )
       ostk1.SetString( rosCmd.BeforeLast( wxT(' ') ) );
  else ostk1.SetString( rosCmd );
  i1 = ostk1.CountTokens( );
  if( i1!=5 && i1!=8 )                            return( FALSE );

  // Check command type
  os1 = ostk1.GetNextToken( );
  if( os1.Upper( ) != wxT(".DC") )                return( FALSE );

  // Extract the sweep source label
  os1 = ostk1.GetNextToken( );
  os1 = roSim.rosGetCpnt( os1 );
  if( os1.IsEmpty( ) )                            bRtnValue = FALSE;
  if( roSim.bSetSigSrc( os1 ) )                   bRtnValue = FALSE;

  // Extract the start voltage
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) )     bRtnValue = FALSE;
  if( ! roSim.bSetSwpStart( (float) df1 ) )       bRtnValue = FALSE;

  // Is the next field a number?
  os1 = ostk1.GetNextToken( );
  if( ConvertType::bStrToFlt( os1, &df2 ) )
  {
    // Extract the stop voltage
    if( ! ConvertType::bStrToFlt( os1, &df1 ) )   bRtnValue = FALSE;
    if( ! roSim.bSetSwpStop( (float) df1 ) )      bRtnValue = FALSE;

    // Extract the sweep type: linear, log or steps per decade
    os1 = ostk1.GetNextToken( );
    os1.MakeUpper( );
    if(      os1 == wxT("BY") ) roSim.bSetSwpScale( 0 );
    else if( os1 == wxT("TI") ) roSim.bSetSwpScale( 1 );
    else if( os1 == wxT("DE") ) roSim.bSetSwpScale( 2 );
    else                                          bRtnValue = FALSE;

    // Extract the step size/count
    os1 = ostk1.GetNextToken( );
    if( roSim.iGetSwpScale( ) < 2 )
    {
      if( ! ConvertType::bStrToFlt( os1, &df1 ) ) bRtnValue = FALSE;
      if( ! roSim.bSetSwpStep( (float) df1 ) )    bRtnValue = FALSE;
    }
    else
    {
      if( ! ConvertType::bStrToInt( os1, &li1 ) ) bRtnValue = FALSE;
      if( ! roSim.bSetSwpStep( (float) li1 ) )    bRtnValue = FALSE;
    }

    os1 = ostk1.GetNextToken( );
  }
  else
  {
    if( ! roSim.bSetSwpStop( (float) df1 ) )       bRtnValue = FALSE;
    if( ! roSim.bSetSwpStep( 1.0 ) )               bRtnValue = FALSE;
    roSim.bSetSwpScale( 0 );
  }

  // Extract the analysis temperature
  if( os1.Upper( ) == wxT("TE") )
  {
    os1 = ostk1.GetNextToken( );
    if( ! ConvertType::bStrToFlt( os1, &df1 ) )   bRtnValue = FALSE;
    if( ! roSim.bSetTempC( (float) df1 ) )        bRtnValue = FALSE;
  }

  return( bRtnValue );
}

//*****************************************************************************
// Parse a AC command and put the data in a simulation object.
// Eg.s: .AC 0.00 TE 27.00 BASIC
//       .AC 0.00 100.00K BY 10.00K TE 27.00 BASIC
//       .AC 0.00 100.00K TI 1.01 TE 30.00 BASIC
//       .AC 0.00 100.00K DE 1 TE 30.00 BASIC
//       .AC 0.00 100.00K OC 10 TE 30.00 BASIC
//
// Argument List:
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bParseCmdAC( wxString & rosCmd, Simulation & roSim )
{
  wxStringTokenizer  ostk1;
  wxString           os1;
  bool               bRtnValue=TRUE;
  double             df1, df2;
  long               li1;
  int                i1;

  // Basic argument checks
  if( rosCmd.Upper( ).AfterLast( wxT(' ') ).IsSameAs( wxT("BASIC") ) )
       ostk1.SetString( rosCmd.BeforeLast( wxT(' ') ) );
  else ostk1.SetString( rosCmd );
  i1 = ostk1.CountTokens( );
  if( i1!=4 && i1!=7 )                          return( FALSE );

  // Check command type
  os1 = ostk1.GetNextToken( );
  if( os1.Upper( ) != wxT(".AC") )              return( FALSE );

  // Extract the start frequency
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) )   bRtnValue = FALSE;
  if( ! roSim.bSetSwpStart( (float) df1 ) )     bRtnValue = FALSE;

  // Is the next field a number?
  os1 = ostk1.GetNextToken( );
  if( ConvertType::bStrToFlt( os1, &df2 )  )
  {
    // Extract the stop frequency
    if( ! ConvertType::bStrToFlt( os1, &df1 ) )   bRtnValue = FALSE;
    if( ! roSim.bSetSwpStop( (float) df1 ) )      bRtnValue = FALSE;

    // Extract the sweep type: linear, log or steps per decade
    os1 = ostk1.GetNextToken( );
    os1.MakeUpper( );
    if(      os1 == wxT("BY") ) roSim.bSetSwpScale( 0 );
    else if( os1 == wxT("TI") ) roSim.bSetSwpScale( 1 );
    else if( os1 == wxT("DE") ) roSim.bSetSwpScale( 2 );
    else if( os1 == wxT("OC") ) roSim.bSetSwpScale( 3 );
    else                                          bRtnValue = FALSE;

    // Extract the step size/count
    os1 = ostk1.GetNextToken( );
    if( roSim.iGetSwpScale( ) < 2 )
    {
      if( ! ConvertType::bStrToFlt( os1, &df1 ) ) bRtnValue = FALSE;
      if( ! roSim.bSetSwpStep( (float) df1 ) )    bRtnValue = FALSE;
    }
    else
    {
      if( ! ConvertType::bStrToInt( os1, &li1 ) ) bRtnValue = FALSE;
      if( ! roSim.bSetSwpStep( (float) li1 ) )    bRtnValue = FALSE;
    }

    os1 = ostk1.GetNextToken( );
  }
  else
  {
    if( ! roSim.bSetSwpStop( (float) df1 ) )       bRtnValue = FALSE;
    if( ! roSim.bSetSwpStep( 1.0 ) )               bRtnValue = FALSE;
    roSim.bSetSwpScale( 0 );
  }

  // Extract the analysis temperature
  if( os1.Upper( ) == wxT("TE") )
  {
    os1 = ostk1.GetNextToken( );
    if( ! ConvertType::bStrToFlt( os1, &df1 ) )    bRtnValue = FALSE;
    if( ! roSim.bSetTempC( (float) df1 ) )         bRtnValue = FALSE;
  }

  return( bRtnValue );
}

//*****************************************************************************
// Parse a TRANSIENT command and put the data in a simulation object.
// Eg.s: .TR 0.00 100.00m 10.00m TE 27.00 COLD BASIC
//       .TR 0.00 100.00m 10.00m TE 27.00 UIC BASIC
//       .TR 0.00 100.00m 10.00m TE 27.00 BASIC
//
// Argument List:
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bParseCmdTR( wxString & rosCmd, Simulation & roSim )
{
  wxStringTokenizer  ostk1;
  wxString           os1;
  bool               bRtnValue=TRUE;
  double             df1;
  int                i1;

  // Basic argument checks
  if( rosCmd.Upper( ).AfterLast( wxT(' ') ).IsSameAs( wxT("BASIC") ) )
       ostk1.SetString( rosCmd.BeforeLast( wxT(' ') ) );
  else ostk1.SetString( rosCmd );
  i1 = ostk1.CountTokens( );
  if( i1!=6 && i1!=7 )                          return( FALSE );

  // Check command type
  os1 = ostk1.GetNextToken( );
  if( os1.Upper( ) != wxT(".TR") )              return( FALSE );

  // Extract the start time
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) )   bRtnValue = FALSE;
  if( ! roSim.bSetSwpStart( (float) df1 ) )     bRtnValue = FALSE;

  // Extract the stop time
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) )   bRtnValue = FALSE;
  if( ! roSim.bSetSwpStop( (float) df1 ) )      bRtnValue = FALSE;

  // Extract the step increment
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) )   bRtnValue = FALSE;
  if( ! roSim.bSetSwpStep( (float) df1 ) )      bRtnValue = FALSE;

  // Extract the analysis temperature
  os1 = ostk1.GetNextToken( );
  if( os1.Upper( ) == wxT("TE") )
  {
    os1 = ostk1.GetNextToken( );
    if( ! ConvertType::bStrToFlt( os1, &df1 ) ) bRtnValue = FALSE;
    if( ! roSim.bSetTempC( (float) df1 ) )      bRtnValue = FALSE;
  }

  // Extract the initial conditions
  if( ostk1.HasMoreTokens( ) )
  {
    os1 = ostk1.GetNextToken( );
    os1.MakeUpper( );
    if(      os1 == wxT("COLD") ) roSim.bSetSwpScale( 0 );
    else if( os1 == wxT("UIC") )  roSim.bSetSwpScale( 1 );
    else                                        bRtnValue = FALSE;
  }
  else                            roSim.bSetSwpScale( 2 );

  return( bRtnValue );
}

//*****************************************************************************
// Parse a FOURIER command and put the data in a simulation object.
// Eg.s: .FO 1.00K 100.00K 10.00K TE 27.00 COLD BASIC
//       .FO 1.00K 100.00K 10.00K TE 27.00 BASIC
//
// Argument List:
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bParseCmdFO( wxString & rosCmd, Simulation & roSim )
{
  wxStringTokenizer  ostk1;
  wxString           os1;
  bool               bRtnValue=TRUE;
  int                i1;
  double             df1;

  // Basic argument checks
  if( rosCmd.Upper( ).AfterLast( wxT(' ') ).IsSameAs( wxT("BASIC") ) )
       ostk1.SetString( rosCmd.BeforeLast( wxT(' ') ) );
  else ostk1.SetString( rosCmd );
  i1 = ostk1.CountTokens( );
  if( i1!=6 && i1!=7 )                          return( FALSE );

  // Check command type
  os1 = ostk1.GetNextToken( );
  if( os1.Upper( ) != wxT(".FO") )              return( FALSE );

  // Extract the start frequency
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) )   bRtnValue = FALSE;
  if( ! roSim.bSetSwpStart( (float) df1 ) )     bRtnValue = FALSE;

  // Extract the stop frequency
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) )   bRtnValue = FALSE;
  if( ! roSim.bSetSwpStop( (float) df1 ) )      bRtnValue = FALSE;

  // Extract the step increment
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) )   bRtnValue = FALSE;
  if( ! roSim.bSetSwpStep( (float) df1 ) )      bRtnValue = FALSE;

  // Extract the analysis temperature
  os1 = ostk1.GetNextToken( );
  if( os1.Upper( ) == wxT("TE") )
  {
    os1 = ostk1.GetNextToken( );
    if( ! ConvertType::bStrToFlt( os1, &df1 ) ) bRtnValue = FALSE;
    if( ! roSim.bSetTempC( (float) df1 ) )      bRtnValue = FALSE;
  }

  // Extract the initial conditions
  if( ostk1.HasMoreTokens( ) )
  {
    os1 = ostk1.GetNextToken( );
    os1.MakeUpper( );
    if( os1 == wxT("COLD") ) roSim.bSetSwpScale( 0 );
    else                                        bRtnValue = FALSE;
  }
  else roSim.bSetSwpScale( 1 );

  return( bRtnValue );
}

//*****************************************************************************
// Parse a PRINT command and put the data in a simulation object.
//
// Argument List:
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bParseCmdPR( wxString & rosCmd, Simulation & roSim )
{
  wxStringTokenizer  ostk1;
  wxString           os1;
  bool               bRtnValue=TRUE;
  size_t             sz1, sz2;

  // Basic argument checks
  ostk1.SetString( rosCmd );
  if( ostk1.CountTokens( ) < 3 )   return( FALSE );

  // Check command type
  os1 = ostk1.GetNextToken( ).Left( 3 );
  if( os1.Upper( ) != wxT(".PR") ) return( FALSE );

  // Extract the analysis type
  os1 = ostk1.GetNextToken( ).Left( 2 );
  os1.MakeUpper( );
  if(      os1 == wxT("OP") ) roSim.bSetAnaType( eANA_OP );
  else if( os1 == wxT("DC") ) roSim.bSetAnaType( eANA_DC );
  else if( os1 == wxT("AC") ) roSim.bSetAnaType( eANA_AC );
  else if( os1 == wxT("TR") ) roSim.bSetAnaType( eANA_TR );
  else if( os1 == wxT("FO") ) roSim.bSetAnaType( eANA_FO );
  else return( FALSE );

  // Extract the parameters to derive, any complex parts and the test component
  // and/or test node labels
  while( ostk1.HasMoreTokens( ) )
  {
    // Extract the next field
    os1 = ostk1.GetNextToken( );
    if( os1.Length( ) < 4 ) { bRtnValue = FALSE; continue; }
    // Extract the parameter specifier
    switch( wxToupper( os1.GetChar( 0 ) ) )
    {
      case wxT('V'): roSim.bSetOutPar( ePAR_VLT, TRUE ); break;
      case wxT('I'): roSim.bSetOutPar( ePAR_CUR, TRUE ); break;
      case wxT('P'): roSim.bSetOutPar( ePAR_PWR, TRUE ); break;
      case wxT('R'): roSim.bSetOutPar( ePAR_RES, TRUE ); break;
      default: bRtnValue = FALSE; break;
    }
    // Extract the complex part
    if( roSim.eGetAnaType( ) == eANA_AC )
    {
      switch( wxToupper( os1.GetChar( 1 ) ) )
      {
        case wxT('M'): roSim.bSetOutCpx( eCPX_MAG,   TRUE ); break;
        case wxT('P'): roSim.bSetOutCpx( eCPX_PHASE, TRUE ); break;
        case wxT('R'): roSim.bSetOutCpx( eCPX_REAL,  TRUE ); break;
        case wxT('I'): roSim.bSetOutCpx( eCPX_IMAG,  TRUE ); break;
        default: bRtnValue = FALSE; break;
      }
      if( wxToupper( os1.GetChar( 1 )==wxT('M') ) )
        if( os1.Mid( 2, 2 ).Upper( ) == wxT("DB") )
          roSim.bSetOutCpx( eCPX_MAGDB, TRUE );
    }
    // Extract the component or node label
    sz1 = os1.Index( wxT('(') );
    sz2 = os1.Index( wxT(')') );
    if( (int)sz1==wxNOT_FOUND || (int)sz2==wxNOT_FOUND || (sz1+1)>=sz2 )
       { bRtnValue = FALSE; continue; }
    os1 = os1.Mid( sz1+1, sz2-sz1-1 );
    if(      roSim.roasGetNodeLbls( ).Index( os1 ) != wxNOT_FOUND )
      roSim.bAddTstNode( os1 );
    else if( roSim.roasGetCpntLbls( ).Index( os1 ) != wxNOT_FOUND )
      roSim.bAddTstCpnt( os1 );
    else bRtnValue = FALSE;
  }

  return( bRtnValue );
}

//*****************************************************************************
// Create the simulator command sequence for an operating point analysis.
//
// Argument List:
//   roSim - Simulation object

void  PrcGnuCap::MakeCmdOP( Simulation & roSim )
{
  wxString  osCmdOP;
  wxString  osCmdPR;
  wxString  osPar, os1;
  uint      uiPar;

  m_osInput = wxT("Temp");

  // Create the OP command
  osCmdOP = wxT(".OP ");
  osCmdOP << roSim.rosGetSwpStart( );
  if( roSim.fGetSwpStop( ) != roSim.fGetSwpStart( ) )
  {
    osCmdOP << wxT(' ') << roSim.rosGetSwpStop( );
    if( roSim.fGetSwpStep( ) != 0.0 )
    {
      switch( roSim.iGetSwpScale( ) )
      {
        case 0 : osCmdOP << wxT(" BY ") << roSim.rosGetSwpStep( );     break;
        case 1 : osCmdOP << wxT(" TI ") << roSim.rosGetSwpStep( );     break;
        case 2 : osCmdOP << wxT(" DE ") << (int) roSim.fGetSwpStep( ); break;
      }
    }
  }
  osCmdOP << wxT(" BASIC");

  // Create the PRINT command
  osCmdPR = wxT(".PR OP");
  for( uiPar=ePAR_FST; uiPar<=ePAR_LST; uiPar++ )
  {
    if( ! roSim.bGetOutPar( (eParType) uiPar ) ) continue;

    switch( uiPar )
    {
      case ePAR_VLT : osPar = wxT("V("); break;
      case ePAR_CUR : osPar = wxT("I("); break;
      case ePAR_PWR : osPar = wxT("P("); break;
      case ePAR_RES : osPar = wxT("R("); break;
      default: continue;
    }

    // Add any components and/or nodes to derivation list for this parameter
    os1 = rosMakeArgsPR( roSim, osPar );
    if( ! os1.IsEmpty( ) ) osCmdPR << os1;
  }

  // Add simulation commands to Simulation object
  roSim.bAddSimCmd( osCmdPR );
  roSim.bAddSimCmd( osCmdOP );
}

//*****************************************************************************
// Create the simulator command sequence for a DC analysis.
//
// Argument List:
//   roSim - Simulation object

void  PrcGnuCap::MakeCmdDC( Simulation & roSim )
{
  wxString  osCmdPR;
  wxString  osCmdDC;
  wxString  osPar, os1;
  uint      uiPar;

  m_osInput = roSim.rosGetSigSrc( ).BeforeFirst( wxT(' ') );

  // Create the DC command
  osCmdDC = wxT(".DC ");
  osCmdDC << m_osInput << wxT(' ') << roSim.rosGetSwpStart( );
  if( roSim.fGetSwpStop( ) != roSim.fGetSwpStart( ) )
  {
    osCmdDC << wxT(' ') << roSim.rosGetSwpStop( );
    if( roSim.fGetSwpStep( ) != 0.0 )
    {
      switch( roSim.iGetSwpScale( ) )
      {
        case 0 : osCmdDC << wxT(" BY ") << roSim.rosGetSwpStep( );     break;
        case 1 : osCmdDC << wxT(" TI ") << roSim.rosGetSwpStep( );     break;
        case 2 : osCmdDC << wxT(" DE ") << (int) roSim.fGetSwpStep( ); break;
      }
    }
  }
  osCmdDC << wxT(" TE ") << roSim.rosGetTempC( ) << wxT(" BASIC");

  // Create the PRINT command
  osCmdPR = wxT(".PR DC");
  for( uiPar=ePAR_FST; uiPar<=ePAR_LST; uiPar++ )
  {
    if( ! roSim.bGetOutPar( (eParType) uiPar ) ) continue;

    switch( uiPar )
    {
      case ePAR_VLT : osPar = wxT("V("); break;
      case ePAR_CUR : osPar = wxT("I("); break;
      case ePAR_PWR : osPar = wxT("P("); break;
      case ePAR_RES : osPar = wxT("R("); break;
      default: continue;
    }

    // Add any components and/or nodes to derivation list for this parameter
    os1 = rosMakeArgsPR( roSim, osPar );
    if( ! os1.IsEmpty( ) ) osCmdPR << os1;
  }

  // Add simulation commands to Simulation object
  roSim.bAddSimCmd( osCmdPR );
  roSim.bAddSimCmd( osCmdDC );
}

//*****************************************************************************
// Create the simulator command sequence for a AC analysis.
//
// Argument List:
//   roSim - Simulation object

void  PrcGnuCap::MakeCmdAC( Simulation & roSim )
{
  wxString  osCmdOP;
  wxString  osCmdAC;
  wxString  osCmdPR;
  wxString  osPar, osCpx, os1;
  uint      uiPar, uiCpx;

  m_osInput.Empty( );

  // Create the OP command
  osCmdOP = wxT(".OP ");
  osCmdOP << roSim.rosGetTempC( );

  // Create the AC command
  osCmdAC = wxT(".AC ");
  osCmdAC << roSim.rosGetSwpStart( );
  if( roSim.fGetSwpStop( ) != roSim.fGetSwpStart( ) )
  {
    osCmdAC << wxT(' ') << roSim.rosGetSwpStop( );
    if( roSim.fGetSwpStep( ) != 0.0 )
    {
      switch( roSim.iGetSwpScale( ) )
      {
        case 0 : osCmdAC << wxT(" BY ") << roSim.rosGetSwpStep( );     break;
        case 1 : osCmdAC << wxT(" TI ") << roSim.rosGetSwpStep( );     break;
        case 2 : osCmdAC << wxT(" DE ") << (int) roSim.fGetSwpStep( ); break;
        case 3 : osCmdAC << wxT(" OC ") << (int) roSim.fGetSwpStep( ); break;
      }
    }
  }
  osCmdAC << wxT(" TE ") << roSim.rosGetTempC( ) << wxT(" BASIC");

  // Create the PRINT commands
  osCmdPR = wxT(".PR AC");
  for( uiPar=ePAR_FST; uiPar<=ePAR_LST; uiPar++ )
  {
    if( ! roSim.bGetOutPar( (eParType) uiPar ) ) continue;

    // Add the parameter prefix
    switch( uiPar )
    {
      case ePAR_VLT : osPar = wxT('V'); break;
      case ePAR_CUR : osPar = wxT('I'); break;
      case ePAR_PWR : osPar = wxT('P'); break;
      case ePAR_RES : osPar = wxT('R'); break;
      default: continue;
    }

    // Add the complex part prefix
    for( uiCpx=eCPX_FST; uiCpx<eCPX_LST; uiCpx++ )
    {
      if( ! roSim.bGetOutCpx( (eCpxType) uiCpx ) ) continue;

      osCpx = osPar;

      switch( uiCpx )
      {
        case eCPX_MAG   : osCpx << wxT('M'); break;
        case eCPX_PHASE : osCpx << wxT('P'); break;
        case eCPX_REAL  : osCpx << wxT('R'); break;
        case eCPX_IMAG  : osCpx << wxT('I'); break;
        default: continue;
      }
      if( uiCpx != eCPX_PHASE )
        if( roSim.bGetOutCpx( eCPX_MAGDB ) )
          osCpx << wxT("DB");
      osCpx << wxT('(');

      // Add any components and/or nodes to derivation list for this parameter
      os1 = rosMakeArgsPR( roSim, osCpx );
      if( ! os1.IsEmpty( ) ) osCmdPR << os1;
    }
  }

  // Add simulation commands to Simulation object
  roSim.bAddSimCmd( osCmdPR );
  roSim.bAddSimCmd( osCmdOP );
  roSim.bAddSimCmd( osCmdAC );
}

//*****************************************************************************
// Create the simulator command sequence for a transient analysis.
//
// Argument List:
//   roSim - Simulation object

void  PrcGnuCap::MakeCmdTR( Simulation & roSim )
{
  wxString  osCmdPR;
  wxString  osCmdTR;
  wxString  osPar, os1;
  uint      uiPar;

  m_osInput.Empty( );

  // Create the TR command
  osCmdTR = wxT(".TR ");
  osCmdTR << roSim.rosGetSwpStart( );
  if( roSim.fGetSwpStop( ) != roSim.fGetSwpStart( ) )
  {
    osCmdTR << wxT(' ') << roSim.rosGetSwpStop( )
            << wxT(' ') << roSim.rosGetSwpStep( );
  }
  osCmdTR << wxT(" TE ") << roSim.rosGetTempC( );
  switch( roSim.iGetSwpScale( ) )
  {
    case eINITC_COLD: osCmdTR << wxT(" COLD"); break;
    case eINITC_UICS: osCmdTR << wxT(" UIC");  break;
    case eINITC_WARM: break;
    default : break;
  }
  osCmdTR << wxT(" BASIC");

  // Create the PRINT commands
  osCmdPR = wxT(".PR TR");
  for( uiPar=ePAR_FST; uiPar<=ePAR_LST; uiPar++ )
  {
    if( ! roSim.bGetOutPar( (eParType) uiPar ) ) continue;

    switch( uiPar )
    {
      case ePAR_VLT : osPar = wxT("V("); break;
      case ePAR_CUR : osPar = wxT("I("); break;
      case ePAR_PWR : osPar = wxT("P("); break;
      case ePAR_RES : osPar = wxT("R("); break;
      default: continue;
    }

    // Add any components and/or nodes to derivation list for this parameter
    os1 = rosMakeArgsPR( roSim, osPar );
    if( ! os1.IsEmpty( ) ) osCmdPR << os1;
  }

  // Add simulation commands to Simulation object
  roSim.bAddSimCmd( osCmdPR );
  roSim.bAddSimCmd( osCmdTR );
}

//*****************************************************************************
// Create the simulator command sequence for a fourier analysis.
//
// Argument List:
//   roSim - Simulation object

void  PrcGnuCap::MakeCmdFO( Simulation & roSim )
{
  wxString  osCmdPR;
  wxString  osCmdFO;
  wxString  osPar, os1;
  uint      uiPar;

  m_osInput.Empty( );

  // Create the FO command
  osCmdFO = wxT(".FO ");
  osCmdFO << roSim.rosGetSwpStart( );
  if( roSim.fGetSwpStop( ) != roSim.fGetSwpStart( ) )
  {
    osCmdFO << wxT(' ') << roSim.rosGetSwpStop( );
    osCmdFO << wxT(' ') << roSim.rosGetSwpStep( );
  }
  osCmdFO << wxT(" TE ") << roSim.rosGetTempC( );
  if( roSim.iGetSwpScale( ) == 0 ) osCmdFO << wxT(" COLD");
  osCmdFO << wxT(" BASIC");

  // Create the PRINT commands
  osCmdPR = wxT(".PR FO");
  for( uiPar=ePAR_FST; uiPar<=ePAR_LST; uiPar++ )
  {
    if( ! roSim.bGetOutPar( (eParType) uiPar ) ) continue;

    switch( uiPar )
    {
      case ePAR_VLT : osPar = wxT("V("); break;
      case ePAR_CUR : osPar = wxT("I("); break;
      case ePAR_PWR : osPar = wxT("P("); break;
      case ePAR_RES : osPar = wxT("R("); break;
      default: continue;
    }

    // Add any components and/or nodes to derivation list for this parameter
    os1 = rosMakeArgsPR( roSim, osPar );
    if( ! os1.IsEmpty( ) ) osCmdPR << os1;
  }

  // Add simulation commands to Simulation object
  roSim.bAddSimCmd( osCmdPR );
  roSim.bAddSimCmd( osCmdFO );
}

//*****************************************************************************
// Generate the simulator commands to be executed based on the simulation
// specifications and the NetList description.
//
// Arguments:
//   roSim - The net list and simulation specification
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bMakeCmds( Simulation & roSim )
{
  switch( roSim.eGetAnaType( ) )
  {
    case eANA_OP : MakeCmdOP( roSim ); break;
    case eANA_DC : MakeCmdDC( roSim ); break;
    case eANA_AC : MakeCmdAC( roSim ); break;
    case eANA_TR : MakeCmdTR( roSim ); break;
    case eANA_FO : MakeCmdFO( roSim ); break;
    default :      return( FALSE );
  }

  if( roSim.roasGetSimCmds( ).IsEmpty( ) ) return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Create a list of arguments for the Spice PRINT command based on the test
// components and test nodes specified in a Simulation object. A prefix may be
// specified which is prepended to each argument in the list.
//
// Argument List:
//   roSim     - The Simulation object
//   rosPrefix - The prefix to be prepended to each argument
//
// Return Values:
//   Success - A list of correctly formatted arguments
//   Failure - An empty string

wxString & PrcGnuCap::rosMakeArgsPR( Simulation & roSim, wxString & rosPrefix )
{
  static  wxString  osArgs;
  wxChar  oc1;
  size_t  sz1;

  osArgs.Empty( );

  // Add any components to derivation list for this parameter
  for( sz1=0; sz1<roSim.roasGetTstCpnts( ).GetCount( ); sz1++ )
    osArgs << wxT(' ') << rosPrefix << roSim.roasGetTstCpnts( ).Item( sz1 )
           << wxT(')');

  // Add any nodes to derivation list for voltage only
  for( oc1=0, sz1=0; sz1<rosPrefix.Length( ); sz1++ )
  {
    oc1 = rosPrefix.GetChar( sz1 );
    if( ! wxIsspace( oc1 ) ) break;
  }
  if( oc1==wxT('v') || oc1==wxT('V') )
  {
    for( sz1=0; sz1<roSim.roasGetTstNodes( ).GetCount( ); sz1++ )
      osArgs << wxT(' ') << rosPrefix << roSim.roasGetTstNodes( ).Item( sz1 )
             << wxT(')');
  }

  return( osArgs );
}

//*****************************************************************************
// Check / format the component lines in the Simulation object so that they are
// compatible with GnuCAP. Ie. add the substrate node to BJT components which
// are missing it, the substrate node is assumed to be connected to the
// collector. Eg. "Q1 nc nb ne BC109" would become "Q1 nc nb ne nc BC109".
//
// Argument List:
//   roasCpnts - The list of component lines
//
// Return Value:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bFmtCpnts( wxArrayString & roasCpnts )
{
  wxStringTokenizer  ostk1;
  wxString           os1, osNodeC;
  size_t             sz1;

  for( sz1=0; sz1<roasCpnts.GetCount( ); sz1++ )
  {
    ostk1.SetString( roasCpnts[ sz1 ] );

    os1 = ostk1.GetNextToken( );

    if( wxToupper( os1.GetChar( 0 ) ) != wxT('Q') ) continue;
    if( ostk1.CountTokens( ) != 4 )                 continue;

    osNodeC = ostk1.GetNextToken( );

    // Add substrate node to BJT definitions (ie. the collector node)
    os1 << wxT(' ') << osNodeC << wxT(' ') << ostk1.GetNextToken( );
    os1 << wxT(' ') << ostk1.GetNextToken( );
    os1 << wxT(' ') << osNodeC << wxT(' ') << ostk1.GetNextToken( );

    roasCpnts.RemoveAt( sz1 );
    roasCpnts.Insert( os1, sz1 );
  }

  return( TRUE );
}

//*****************************************************************************
// Format the contents of the results file so that gWave can read it ie. a
// header line at the top containing parameter names followed by the columns of
// data.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bFmtResults( void )
{
  bool  bRtnValue;

  m_osErrMsg.Empty( );

  // Attempt to open the results file
  if( ! roGetResultsFile( ).FileExists( ) )
  {
    m_osErrMsg << wxT("Results file doesn't exist : \n\n")
               << roGetResultsFile( ).GetFullPath( );
    return( FALSE );
  }
  if( ! m_oFileResults.Open( roGetResultsFile( ).GetFullPath( ) ) )
  {
    m_osErrMsg << wxT("Results file couldn't be opened : \n\n")
               << roGetResultsFile( ).GetFullPath( );
    return( FALSE );
  }

  // Format the simulation results
  if( roGetResultsFile( ).GetFullName( ).Right( 3 ) == wxT(".fo") )
       bRtnValue = bFmtResultsFO( );
  else bRtnValue = bFmtGeneric( );

  // Need last line to end with a '\n' or text controls will not load it
  if( m_oFileResults.GetLastLine( ).Last( ) != wxT('\n') )
    m_oFileResults.GetLastLine( ) << wxT('\n');

  m_oFileResults.Write( );  // Save the changes to disk
  m_oFileResults.Close( );  // Close the file

  return( bRtnValue );
}

//*****************************************************************************
// This function is a generic results file formatter which works in most
// circumstance.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bFmtGeneric( void )
{
  wxString  os1, os2;
  size_t    sz1, sz2;
  wxChar    oc1;

  // This function requires the result file to have been opened
  if( ! m_oFileResults.IsOpened( ) ) return( FALSE );

  // Find the beginning of the data area (ie. the last line beginning with '#')
  // and delete all the lines before the data area
  for( sz1=sz2=0; sz1<m_oFileResults.GetLineCount( ); sz1++ )
  {
    os1 = m_oFileResults.GetLine( sz1 );
    if( ! os1.IsEmpty( ) )
      if( os1.GetChar( 0 ) == wxT('#') )
        sz2 = sz1;
  }
  for( sz1=0; sz1<sz2; sz1++ ) m_oFileResults.RemoveLine( 0 );

  // Format the column header line
  bFmtColumnHdr( );

  // Delete any simulator error messages eg. "open circuit: internal node 3"
  // (All lines should start with a digit except for the first)
  for( sz1=1; !m_oFileResults.Eof() && sz1<m_oFileResults.GetLineCount(); sz1++ )
  {
    os1 = m_oFileResults.GetLine( sz1 );
    if( os1.Length( ) <= 1 )
    { // Delete empty lines
      m_oFileResults.RemoveLine( sz1 );
      sz1--;
      continue;
    }
    oc1 = os1.GetChar( 1 );
    if( ! wxIsdigit( oc1 ) )
    { // Delete non-data lines
      m_oFileResults.RemoveLine( sz1 );
      sz1--;
      continue;
    }
  }

  // Format the data lines
  bFmtDataLines( );

  return( TRUE );
}

//*****************************************************************************
// This function is a results file formatter for Fourier analysis result.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bFmtResultsFO( void )
{
  wxStringTokenizer  ostk1;
  wxConfig         * poConfig;
  wxString           os1, os2, os3;
  size_t             sz1, sz2;
  bool               b1;

  // This function requires the result file to have been opened
  if( ! m_oFileResults.IsOpened( ) ) return( FALSE );

  // Find the beginning of the data area
  for( sz1=0; sz1<m_oFileResults.GetLineCount( ); sz1++ )
  {
    os1 = m_oFileResults.GetLine( sz1 );
    if( ! os1.IsEmpty( ) )
      if( os1.Contains( wxT("--- actual ---") ) )
      {
        sz2 = sz1;
        break;
      }
  }

  // Delete all the lines before the data area
  for( sz1=0; sz1<sz2; sz1++ ) m_oFileResults.RemoveLine( 0 );

  // Delete unwanted header lines
  for( os1=m_oFileResults.GetFirstLine( ); !m_oFileResults.Eof( );
                                            os1=m_oFileResults.GetNextLine( ) )
  {
    if( os1.StartsWith( wxT("#freq") ) )
      m_oFileResults.RemoveLine( m_oFileResults.GetCurrentLine( ) );
  }

  // Delete unwanted columns of data
  poConfig = (wxConfig *) wxConfig::Get( );
  if( poConfig != NULL )
  {
    poConfig->SetPath( wxT("/Spice/GNU-Cap") );

    for( ostk1=m_oFileResults.GetFirstLine( ); !m_oFileResults.Eof( );
                                          ostk1=m_oFileResults.GetNextLine( ) )
    {
      if( ostk1.CountTokens( ) != 7 ) continue;

      // Get the operator field
      os1 = ostk1.GetNextToken( );

      // Get actual fields
      poConfig->Read( wxT("FO_Actual"), &b1, TRUE );
      if( b1 )
      {
        poConfig->Read( wxT("FO_MagDb"), &b1, TRUE );
        if( b1 )
        {
          ostk1.GetNextToken( );
          os1 << wxT(' ') << ostk1.GetNextToken( );
        }
        else
        {
          os1 << wxT(' ') << ostk1.GetNextToken( );
          ostk1.GetNextToken( );
        }
        poConfig->Read( wxT("FO_Phase"), &b1, FALSE );
        if( b1 ) os1 << wxT(' ') << ostk1.GetNextToken( );
        else     ostk1.GetNextToken( );
      }
      else for( sz1=0; sz1<3; sz1++ ) ostk1.GetNextToken( );

      // Get relative fields
      poConfig->Read( wxT("FO_Relative"), &b1, FALSE );
      if( b1 )
      {
        poConfig->Read( wxT("FO_MagDb"), &b1, TRUE );
        if( b1 )
        {
          ostk1.GetNextToken( );
          os1 << wxT(' ') << ostk1.GetNextToken( );
        }
        else
        {
          os1 << wxT(' ') << ostk1.GetNextToken( );
          ostk1.GetNextToken( );
        }
        poConfig->Read( wxT("FO_Phase"), &b1, FALSE );
        if( b1 ) os1 << wxT(' ') << ostk1.GetNextToken( );
      }

      m_oFileResults[ m_oFileResults.GetCurrentLine( ) ] = os1;
    }
  }

  // Construct the data column header
  os1 = wxT("#Freq");
  for( ostk1=m_oFileResults.GetFirstLine( ); !m_oFileResults.Eof( );
                                          ostk1=m_oFileResults.GetNextLine( ) )
  {
    // Look for header lines
    if( ostk1.GetNextToken( ) != wxT("#") ) continue;

    os2 = ostk1.GetNextToken( );

    // Get actual fields
    poConfig->Read( wxT("FO_Actual"), &b1, TRUE );
    if( b1 )
    {
      os3 = os2;
      poConfig->Read( wxT("FO_MagDb"), &b1, TRUE );
      if( b1 ) os3.insert( 1, wxT("ADB") );
      else     os3.insert( 1, wxT("A") );
      os1 << wxT(' ') << os3;

      poConfig->Read( wxT("FO_Phase"), &b1, FALSE );
      if( b1 )
      {
        os3 = os2;
        os3.insert( 1, wxT("AP") );
        os1 << wxT(' ') << os3;
      }
    }

    // Get relative fields
    poConfig->Read( wxT("FO_Relative"), &b1, TRUE );
    if( b1 )
    {
      os3 = os2;
      poConfig->Read( wxT("FO_MagDb"), &b1, TRUE );
      if( b1 ) os3.insert( 1, wxT("RDB") );
      else     os3.insert( 1, wxT("R") );
      os1 << wxT(' ') << os3;

      poConfig->Read( wxT("FO_Phase"), &b1, FALSE );
      if( b1 )
      {
        os3 = os2;
        os3.insert( 1, wxT("RP") );
        os1 << wxT(' ') << os3;
      }
    }

    // Remove the unwanted header lines
    if( m_oFileResults.GetCurrentLine( ) > 0 )
      m_oFileResults.RemoveLine( m_oFileResults.GetCurrentLine( ) );
  }
  m_oFileResults[ 0 ] = os1;

  // Determine the number of unique data lines
  ostk1 = m_oFileResults.GetFirstLine( );
  ostk1 = m_oFileResults.GetNextLine( );
  os1 = ostk1.GetNextToken( );
  for( ostk1=m_oFileResults.GetNextLine( ), sz1=1; !m_oFileResults.Eof( );
                                   ostk1=m_oFileResults.GetNextLine( ), sz1++ )
  {
    if( os1 == ostk1.GetNextToken( ) )
      break;
  }

  // Concatinate data lines according to frequency
  for( sz2=1; sz2<=sz1; sz2++ )
  {
    m_oFileResults.GoToLine( sz2-1 );
    ostk1=m_oFileResults.GetNextLine( );
    os1 = ostk1.GetNextToken( );
    for( ostk1=m_oFileResults.GetNextLine( ); !m_oFileResults.Eof( );
                                          ostk1=m_oFileResults.GetNextLine( ) )
    {
      if( os1 == ostk1.GetNextToken( ) )
        m_oFileResults[ sz2 ] << wxT(' ') << ostk1.GetString( );
    }
  }

  // Delete data lines which are no longer needed
  m_oFileResults.GoToLine( sz1+1 );
  while( ! m_oFileResults.Eof( ) )
    m_oFileResults.RemoveLine( m_oFileResults.GetCurrentLine( ) );

  // Format the header line and data lines
  bFmtColumnHdr( );
  bFmtDataLines( );

  return( TRUE );
}

//*****************************************************************************
// Format the column header in the results file. Set the label spacing and
// insert the input variable name into the column labels if necessary.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bFmtColumnHdr( void )
{
  wxStringTokenizer  ostk1;
  wxString           osLine, os1;
  int                i1;

  // This function requires the result file to have been opened
  if( ! m_oFileResults.IsOpened( ) ) return( FALSE );

  // Get the column header line
  ostk1 = m_oFileResults.GetFirstLine( );
  if( ostk1.CountTokens( ) < 1 )     return( FALSE );

  // Replace the first field if m_osInput is not empty
  os1 = ostk1.GetNextToken( );
// osLine = wxT("# "); // ??? 06/08/2005 gWave breaks if there's a space after initial '#'
  osLine = wxT('#');
  if( m_osInput.IsEmpty( ) ) osLine << os1.Right( os1.Length( )-1 );
  else                       osLine << m_osInput;

  // Format each field
  while( ostk1.HasMoreTokens( ) )
  {
    // Pad the column with spaces to the required width
    i1 = GNUCAP_COL_WD - (osLine.Length( ) % GNUCAP_COL_WD);
    osLine.Append( wxT(' '), i1 );

    // Add the next label to the line
    osLine << wxT(' ') << ostk1.GetNextToken( );
  }

  osLine.Trim( ); // Remove trailing space characters

  m_oFileResults.GetFirstLine( ) = osLine;

  return( TRUE );
}

//*****************************************************************************
// Format the data lines in the results file.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bFmtDataLines( void )
{
  wxStringTokenizer  ostk1;
  wxString           osLine, os1;
  size_t             sz1;
  double             f1;
  int                i1;

  // This function requires the result file to have been opened
  if( ! m_oFileResults.IsOpened( ) ) return( FALSE );

  for( sz1=1; sz1<m_oFileResults.GetLineCount( ); sz1++ )
  {
    // Get a line of data
    osLine = m_oFileResults.GetLine( sz1 );

    // Tokenize the string
    ostk1.SetString( osLine );
    if( ostk1.CountTokens( ) < 1 )   return( FALSE );

    // Format each field
    osLine.Empty( );
    while( ostk1.HasMoreTokens( ) )
    {
      // Add the next parameter to the line
      os1 = ostk1.GetNextToken( );
      if( ! ConvertType::bStrToFlt( os1, &f1 ) ) f1  =      -9.999e+99;
      if( ! ConvertType::bFltToStr( f1, os1 ) )  os1 = wxT("-9.999e+99");
      osLine << os1;

      // Pad the column with spaces to the required width
      i1 = GNUCAP_COL_WD - (osLine.Length( ) % GNUCAP_COL_WD);
      osLine.Append( wxT(' '), i1 );
    }

    osLine.Trim( ); // Remove trailing space characters

    m_oFileResults.GetLine( sz1 ) = osLine;
  }

  return( TRUE );
}

//*****************************************************************************
// Parse the simulation commands strings found in a Simulation object and load
// the values back into the Simulation object.
//
// Arguments:
//   roSim - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bParseSim( Simulation & roSim )
{
  const wxArrayString & roasCmds=roSim.roasGetSimCmds( );
  wxString  os1;
  size_t    sz1;

  // Go through the simulation commands and attempt to parse them
  for( sz1=0; sz1<roasCmds.GetCount( ); sz1++ )
  {
    os1 = roasCmds.Item( sz1 );

    if( bParseCmdGEN( os1, roSim ) ) continue;
    if( bParseCmdOP ( os1, roSim ) ) continue;
    if( bParseCmdDC ( os1, roSim ) ) continue;
    if( bParseCmdAC ( os1, roSim ) ) continue;
    if( bParseCmdTR ( os1, roSim ) ) continue;
    if( bParseCmdFO ( os1, roSim ) ) continue;
    if( bParseCmdPR ( os1, roSim ) ) continue;
  }

  return( TRUE );
}

//*****************************************************************************
// Create the simulation commands from the values found in a Simulation object
// and load the command string back into the Simulation object.
//
// Arguments:
//   roSim - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGnuCap::bMakeSim( Simulation & roSim )
{
  wxFileName  ofn1;
  wxString    os1;
  int         i1;

  // Clear error attributes
  m_osErrMsg.Empty( );

  // Only execute simulation if it isn't already running
  if( bIsExec( ) )
  {
    m_osErrMsg = wxT("Simulation already running");
    return( FALSE );
  }

  // Check / format lines for compatibility with GnuCAP
  if( ! bFmtCpnts( roSim.m_oasCpnts ) )
  {
    m_osErrMsg = wxT("Incompatibility with simulator");
    return( FALSE );
  }

  // Create the simulator commands
  if( ! bMakeCmds( roSim ) )
  {
    m_osErrMsg = wxT("Couldn't create simulator commands");
    return( FALSE );
  }

  // Save circuit description and SPICE commands to file
  ofn1 = roSim.rofnGetLoadFile( );
  if( ! roSim.bSaveFile( ofn1.GetFullPath( ) ) )
  {
    m_osErrMsg = wxT("Simulation couldn't be saved to file");
    return( FALSE );
  }

  // Construct the command line to execute the simulation
  os1 = wxT("-b ") + roSim.rofnGetSaveFile( ).GetFullPath( );
  if( ! bSetArgLst( os1.c_str( ) ) )
  {
    m_osErrMsg = wxT("Couldn't set argument list");
    return( FALSE );
  }

  // Create the results file name
  os1 = roSim.rofnGetLoadFile( ).GetPath( ) + wxT('/')
      + roSim.rofnGetLoadFile( ).GetName( );
  i1 = os1.Find( wxT(".gspiceui") );
  if( i1 > 0 ) os1 = os1.Truncate( (size_t) i1 );
  os1 << wxT('.') << rofnGetBinary( ).GetName( );
  switch( roSim.eGetAnaType( ) )
  {
    case eANA_OP : os1 << wxT(".op"); break;
    case eANA_DC : os1 << wxT(".dc"); break;
    case eANA_AC : os1 << wxT(".ac"); break;
    case eANA_TR : os1 << wxT(".tr"); break;
    case eANA_FO : os1 << wxT(".fo"); break;
    default :      return( FALSE );
  }
  if( ! bSetResultsFile( os1.c_str( ) ) )
  {
    m_osErrMsg = wxT("Couldn't set results file name");
    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Check for error messages in the simulator output ???

//  bData = FALSE;
//  for( os1=oFileCct.GetFirstLine(); !oFileCct.Eof(); os1=oFileCct.GetNextLine())
//  {
//    if( os1.Length( ) <= 1 ) continue;
//    oc1 = os1.GetChar( 0 );
//    if( bData )
//    { // Look for simulator error messages
//      if( !wxIsdigit( oc1 ) && oc1!=wxT(' ') && oc1!=wxT('-') )
//      { // Simulation error message found
//        osError = wxT("Error/s encountered running simulation");
//        break;
//      }
//    }
//    else
//    { // Look for the beginning of the simulation data
//      if( oc1 == wxT('#') ) bData = TRUE;
//    }
//  }

//*****************************************************************************
