//*****************************************************************************
//                              CmdNgSpiceAC.cpp                              *
//                             ------------------                             *
//  Started     : 11/10/2007                                                  *
//  Last Update : 21/02/2008                                                  *
//  Copyright   : (C) 2007 by M.S.Waters                                      *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "ngspice/commands/CmdNgSpiceAC.hpp"

//*****************************************************************************
// Constructor.

CmdNgSpiceAC::CmdNgSpiceAC( void )
{
  bClear( );
}

//*****************************************************************************
// Destructor.

CmdNgSpiceAC::~CmdNgSpiceAC( )
{
}

//*****************************************************************************
// Parse the command string.
//
// Eg.s : .AC LIN 50 1.00K 300.00K
//        .AC DEC 20 1.00K 300.00K
//        .AC OCT 10 1.00K 300.00K
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceAC::bParse( void )
{
  wxStringTokenizer  ostk1;
  wxString           os1;
  double             df1;
  long               li1;

  // Clear the object attributes
  os1 = *this;
  bClear( );
  *this = os1;

  // Tokenize the command string
  ostk1.SetString( *this );
  if( ostk1.CountTokens( ) != 5 ) return( FALSE );

  // Check command type
  os1 = ostk1.GetNextToken( ).Left( 3 ).Upper( );
  if( os1 != wxT(".AC") )         return( FALSE );
  m_osName = wxT("AC");

  m_bIsOk = TRUE;

  // Extract the sweep type: linear or log
  os1 = ostk1.GetNextToken( ).Upper( );
  if(      os1 == wxT("LIN") ) m_eScale = eSCALE_LIN;
  else if( os1 == wxT("DEC") ) m_eScale = eSCALE_DEC;
  else if( os1 == wxT("OCT") ) m_eScale = eSCALE_OCT;
  else                            m_bIsOk = FALSE;

  // Extract the step count
  os1 = ostk1.GetNextToken( );
  if( ConvertType::bStrToInt( os1, &li1 ) ) m_fStep  = (float) li1;
  else                            m_bIsOk = FALSE;

  // Extract the start frequency
  os1 = ostk1.GetNextToken( );
  if( ConvertType::bStrToFlt( os1, &df1 ) ) m_fStart = (float) df1;
  else                            m_bIsOk = FALSE;

  // Extract the stop frequency
  os1 = ostk1.GetNextToken( );
  if( ConvertType::bStrToFlt( os1, &df1 ) ) m_fStop  = (float) df1;
  else                            m_bIsOk = FALSE;

  return( m_bIsOk );
}

//*****************************************************************************
// Format the command string.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceAC::bFormat( void )
{
  wxString  osCmd, os1;

  m_bIsOk = FALSE;

  osCmd = wxT(".AC");

  switch( m_eScale )
  {
    case eSCALE_LIN : osCmd << wxT(" LIN"); break;
    case eSCALE_DEC : osCmd << wxT(" DEC"); break;
    case eSCALE_OCT : osCmd << wxT(" OCT"); break;
    default         :                                         return( FALSE );
  }

  osCmd << wxT(' ') << (int) lroundf( m_fStep );

  if( ! ConvertType::bFltToStrEng( (double) m_fStart, os1 ) ) return( FALSE );
  osCmd << wxT(' ') << os1;

  if( ! ConvertType::bFltToStrEng( (double) m_fStop,  os1 ) ) return( FALSE );
  osCmd << wxT(' ') << os1;

  *this = osCmd;

  m_bIsOk = TRUE;

  return( m_bIsOk );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceAC::bClear( void )
{
  CmdBase::bClear( );

  m_fStart = -FLT_MAX;
  m_fStop  = -FLT_MAX;
  m_fStep  = -FLT_MIN;

  m_eScale = eSCALE_NONE;

  return( TRUE );
}

//*****************************************************************************
//                                                                            *
//                                 Test Utility                               *
//                                                                            *
//*****************************************************************************

#ifdef TEST_UTIL

// System include files


// Application includes


// Function prototypes

void  Usage( char * psAppName );

//*****************************************************************************

int  main( int argc, char * argv[ ] )
{
  wxString  osCmd;
  wxString  os1;

  // Validate the argument count passed to the application
  if( argc > 2 )           { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }

  // Process the command line arguments
  os1 = wxConvLibc.cMB2WC( argv[ 1 ] );
  if( argc > 1 )
  {
    if( os1 == wxT("-h") ) { Usage( argv[ 0 ] ); exit( EXIT_SUCCESS ); }
    else                   { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }
  }

  // Display the utility banner
  cout << "\n  Class CmdNgSpiceAC Test Utility"
       << "\n     Version 1.00 (01/02/2008)\n";

  // Create a GNU-CAP AC command object
  CmdNgSpiceAC  oCmd_AC;

  // Use the following command example to check the formatter and the parser :
  osCmd = wxT(".AC DEC 30 1.00K 300.00K");

  // Set things up for a formatter test
  oCmd_AC.bClear( );
  oCmd_AC.m_fStart = 1.0E3;
  oCmd_AC.m_fStop  = 300.0E3;
  oCmd_AC.m_fStep  = 30;
  oCmd_AC.m_eScale = eSCALE_DEC;
  cout << "\nRun Formatter    : " << ( oCmd_AC.bFormat( ) ? "OK" : "FAULT" );
  cout << "\nTest Cmd Format  : " << ( oCmd_AC == osCmd   ? "OK" : "FAULT" );
  cout << "\nExample Command  : " << osCmd  .mb_str( );
  cout << "\noCmd_AC Contents : " << oCmd_AC.mb_str( ) << '\n';

  // Set things up for a parser test
  oCmd_AC.bClear( );
  oCmd_AC = osCmd;
  cout << "\nRun Parser       : " << ( oCmd_AC.bParse( ) ? "OK" : "FAULT" );
  oCmd_AC.bFormat( );
  cout << "\nTest Cmd Format  : " << ( oCmd_AC == osCmd  ? "OK" : "FAULT" );
  cout << "\nExample Command  : " << osCmd  .mb_str( );
  cout << "\noCmd_AC Contents : " << oCmd_AC.mb_str( ) << '\n';

  cout << '\n';

  exit( EXIT_SUCCESS );
}

//*****************************************************************************

void  Usage( char * psAppName )
{
  cout << "\nUsage   : " << psAppName << " [-OPTIONS]"
       << "\nOptions :"
       << "\n  -h : Print usage (this message)\n";
}

#endif // TEST_UTIL

//*****************************************************************************
