//*****************************************************************************
//                                Simulation.cpp                              *
//                               ----------------                             *
//  Started     : 23/09/2003                                                  *
//  Last Update : 29/02/2008                                                  *
//  Copyright   : (C) 2003 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "netlist/Simulation.hpp"

//*****************************************************************************
// Constructor.

Simulation::Simulation( void )
{
  // Initialize all object attributes
  bClear( );
}

//*****************************************************************************
// Destructor.

Simulation::~Simulation( )
{
}

//*****************************************************************************
// Extract the simulator type.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bExtractSimrType( void )
{
  wxString  os1;
  size_t    sz1;

  // Scan the circuit description for simulator type
  for( sz1=0; sz1<GetCount( ); sz1++ )
  {
    os1 = Item( sz1 );
    if( os1.IsEmpty( ) ) continue;

    os1.MakeUpper( );

    if( os1.Contains( wxT("GNU-CAP") ) )
    {
      m_eSimrType = eSIMR_GNUCAP;
      return( TRUE );
    }

    if( os1.Contains( wxT("NG-SPICE") ) )
    {
      m_eSimrType = eSIMR_NGSPICE;
      return( TRUE );
    }
  }

  return( FALSE );
}

//*****************************************************************************
// Extract all the simulator command lines from the circuit description.
// The following commands are currently detected and extracted:
//
//   .OPTIONS
//   .GENERATOR
//   .IC
//   .OP
//   .DC
//   .AC
//   .TRANSIENT
//   .FOURIER
//   .PRINT
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bExtractSimCmds( void )
{
  wxArrayString  oas1;
  wxString       os1, os2;
  size_t         sz1;

  // Scan the circuit description for simulation commands
  for( sz1=0; sz1<GetCount( ); sz1++ )
  {
    os1 = Item( sz1 );

    if( os1.IsEmpty( ) )               continue;
    if( ! os1.StartsWith( wxT(".") ) ) continue;

    os1.MakeUpper( );

    if(      os1.StartsWith( wxT(".OPT")   ) ) m_osOPTIONS = os1;
    else if( os1.StartsWith( wxT(".GEN")   ) ) ;
    else if( os1.StartsWith( wxT(".IC")    ) ) ;
    else if( os1.StartsWith( wxT(".OP")    ) ) ;
    else if( os1.StartsWith( wxT(".DC")    ) ) ;
    else if( os1.StartsWith( wxT(".AC")    ) ) ;
    else if( os1.StartsWith( wxT(".TR")    ) ) ;
    else if( os1.StartsWith( wxT(".FO")    ) ) ;
    else if( os1.StartsWith( wxT(".PR")    ) ) ;
    else if( os1.StartsWith( wxT(".PARAM") ) ) ;
    else if( os1.StartsWith( wxT(".SET")   ) ) ;
    else continue;

    os1 << wxT('\n');
    m_oasSimCmds.Add( Item( sz1 ) );
  }

  if( oas1.IsEmpty( ) ) return( FALSE );

  m_oasSimCmds = oas1;

  return( TRUE );
}

//*****************************************************************************
// Extract the source component.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bExtractSigSrc( void )
{
  wxString  os1;
  size_t    sz1;

  // Scan the circuit description for a source component
  for( sz1=0; sz1<GetCount( )-1; sz1++ )
  {
    os1 = Item( sz1 );

    if( os1.IsEmpty( ) )                        continue;
    if( ! os1.StartsWith( wxT("* Source (") ) ) continue;

    os1 = os1.AfterFirst( wxT('(') );
    os1 = os1.BeforeLast( wxT(')') );

    if( os1.Length( ) < 7 )                     continue; // Eg. "R 0 1 5"

    m_osSigSrc[ 0 ] = Item( sz1 + 1 );
    m_osSigSrc[ 1 ] = os1;

    return( TRUE );
  }

  return( FALSE );
}

//*****************************************************************************
// Clear all object attributes.

bool  Simulation::bClear( void )
{
  bool  bRtnValue=TRUE;
  uint  ui1;

  // Clear the simulation engine and analysis type specifiers
  if( ! bSetSimrType( eSIMR_NONE ) )            bRtnValue = FALSE;
  if( ! bSetAnaType( eANA_NONE ) )              bRtnValue = FALSE;

  // Clear the sweep parameter array
  for( ui1=0; ui1<eSWP_SIZE; ui1++ ) m_fSwp[ ui1 ] = -FLT_MAX;

  // Clear all parameter specifiers
  for( ui1=ePAR_FST; ui1<=ePAR_LST; ui1++ )
    if( ! bSetOutPar( (eParType) ui1, FALSE ) ) bRtnValue = FALSE;

  // Clear all complex part specifiers
  for( ui1=eCPX_FST; ui1<=eCPX_LST; ui1++ )
    if( ! bSetOutCpx( (eCpxType) ui1, FALSE ) ) bRtnValue = FALSE;

  // Clear temperature
  m_fTempC = -FLT_MAX;

  // Clear everything else
  m_oasTstNodes  .Empty( );
  m_oasTstCpnts  .Empty( );
  m_oasSimCmds   .Empty( );
  m_osSigSrc[ 0 ].Empty( );
  m_osSigSrc[ 1 ].Empty( );
  m_osErrMsg     .Empty( );

  // Clear the command strings
  m_osOPTIONS.Empty( );

  return( bRtnValue );
}

//*****************************************************************************
// Do the current simulation settings constitute a valid simulation?
//
// Return Values:
//   TRUE  - Settings are     valid
//   FALSE - Settings are not valid

bool  Simulation::bIsValid( void )
{
  uint  ui1;

  m_osErrMsg.Clear( );

  // Have any test components or test nodes been specified?
  if( m_oasTstNodes.GetCount( )>0 || m_oasTstCpnts.GetCount( )>0 )
  {
    m_osErrMsg = wxT("No nodes or components have been selected.");
    return( FALSE );
  }

  // Has at least one parameter been specified?
  for( ui1=ePAR_FST; ui1<=ePAR_LST; ui1++ )
  {
    if( bGetOutPar( (eParType) ui1 ) ) break;
    if( ui1>=ePAR_LST )
    {
      m_osErrMsg = wxT("No calculation parameters have been specified.");
      return( FALSE );
    }
  }

  return( TRUE );
}

//*****************************************************************************
// Load (or reload) the simulation from file.
//
// Arguments:
//   rosFName - The name of the file to be loaded
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bLoadFile( const wxString & rosFName )
{
  if( ! NetList::bLoadFile( rosFName ) ) return( FALSE );

  bExtractSimrType( );
  bExtractSimCmds( );
  bExtractSigSrc( );

  if( ! m_osOPTIONS.IsEmpty( ) )
  {
    CmdBase * poCmdOPTIONS;

    switch( m_eSimrType )
    {
      case eSIMR_GNUCAP:
        break;
      case eSIMR_NGSPICE:
        poCmdOPTIONS = new CmdNgSpiceOPT;
        (wxString) *poCmdOPTIONS = m_osOPTIONS;
        poCmdOPTIONS->bParse( );
        m_fTempC = ((CmdNgSpiceOPT *) poCmdOPTIONS)->m_fTEMP;
        delete ((CmdNgSpiceOPT *) poCmdOPTIONS);
        break;
      default:
        break;
    }
  }

  return( TRUE );
}

//*****************************************************************************
// Save (or resave) the simulation to file.
//
// Arguments:
//   rosFName - The name of the file to be saved
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bSaveFile( const wxString & rosFName )
{
  wxString  os1;
  size_t    sz1;

  // Save the net list to file
  if( ! NetList::bSaveFile( rosFName ) ) return( FALSE );

  // Open the file
  wxTextFile  oFileCct( m_ofnSave.GetFullPath( ) );
  if( ! oFileCct.Open( ) )               return( FALSE );

  // Find the source component and insert a comment before it
  if( !m_osSigSrc[ 0 ].IsEmpty( ) && !m_osSigSrc[ 1 ].IsEmpty( ) )
  {
    for( os1=oFileCct.GetFirstLine(); !oFileCct.Eof(); os1=oFileCct.GetNextLine() )
    {
      if( m_osSigSrc[ 0 ].BeforeFirst( wxT(' ') ) == os1.BeforeFirst( wxT(' ') ) )
      {
        os1.Empty( );
        os1 << wxT("* Source (") << m_osSigSrc[ 1 ] << wxT(')');
        sz1 = oFileCct.GetCurrentLine( );
        oFileCct.InsertLine( os1, sz1 );
        break;
      }
    }
  }

  // Temporarily remove the circuit description terminator ie. ".END"
  oFileCct.RemoveLine( oFileCct.GetLineCount( )-1 );

  // Append the simulation command lines to the end of the file
  os1 = wxT("* ");
  switch( m_eSimrType )
  {
    case eSIMR_GNUCAP:
      os1 << wxT("GNU-Cap ");
      break;
    case eSIMR_NGSPICE:
      os1 << wxT("NG-Spice ");
      break;
    default:
      break;
  }
  os1 << wxT("Simulation Commands");
  oFileCct.AddLine( os1 );
  oFileCct.AddLine( m_osOPTIONS );
  for( sz1=0; sz1<m_oasSimCmds.Count( ); sz1++ )
    oFileCct.AddLine( m_oasSimCmds[ sz1 ] );
  oFileCct.AddLine( wxT("") );

  // Add the circuit description terminator
  oFileCct.AddLine( wxT(".END") );
  oFileCct.AddLine( wxT("") );

  // Save the changes to disk
  oFileCct.Write( );

  oFileCct.Close( );

  return( TRUE );
}

//*****************************************************************************
// Select the simulation engine used.
//
// Arguments:
//   eSimr - The simulation engine type
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bSetSimrType( eSimrType eSimr )
{
  if( eSimr<eSIMR_FST || eSimr>eSIMR_LST ) return( FALSE );

  m_eSimrType = eSimr;

  return( TRUE );
}

//*****************************************************************************
// Select the analysis to be performed.
//
// Arguments:
//   eAna - The analysis type
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bSetAnaType( eAnaType eAna )
{
  if( eAna<eANA_FST || eAna>eANA_LST ) return( FALSE );

  m_eAnaType = eAna;

  return( TRUE );
}

//*****************************************************************************
// Set the sweep start point.
//
// Argument List:
//   fStart - The sweep starting point
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bSetSwpStart( float fStart )
{
  if( fStart<-FLT_MAX || fStart>FLT_MAX ) return( FALSE );

  m_fSwp[ eSWP_START ] = fStart;

  return( TRUE );
}

//*****************************************************************************
// Set the simulation stop point.
//
// Argument List:
//   fStop - The sweep end point
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bSetSwpStop( float fStop )
{
  if( fStop<-FLT_MAX || fStop>FLT_MAX ) return( FALSE );

  m_fSwp[ eSWP_STOP ] = fStop;

  return( TRUE );
}

//*****************************************************************************
// Set the sweep step size.
//
// Argument List:
//   fStep - The sweep step size
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bSetSwpStep( float fStep )
{
  if( fStep<-FLT_MAX || fStep>FLT_MAX ) return( FALSE );

  m_fSwp[ eSWP_STEP ] = fStep;

  return( TRUE );
}

//*****************************************************************************
// Set the sweep scale type.
//
// Argument List:
//   iScale - The sweep step scale type
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bSetSwpScale( int iScale )
{
  if( iScale<0 || iScale>3 ) return( FALSE );

  m_fSwp[ eSWP_SCALE ] = (float) iScale;

  return( TRUE );
}

//*****************************************************************************
// Enable or disable the derivation of an analysis parameter.
//
// Arguments:
//   ePar   - The parameter to set
//   bState - Indicate whether the parameter is to be derived or not
//            (TRUE - derive parameter, FALSE - don't derive parameter)
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bSetOutPar( eParType ePar, bool bState )
{
  if( ePar<ePAR_FST || ePar>ePAR_LST ) return( FALSE );

  m_bPar[ ePar ] = bState;

  return( TRUE );
}

//*****************************************************************************
// Enable or disable the derivation of a part of a complex analysis parameter.
//
// Arguments:
//   eCpx   - The complex part to set
//   bState - Indicate whether the parameter is to be derived or not
//            (TRUE - derive parameter, FALSE - don't derive parameter)
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bSetOutCpx( eCpxType eCpx, bool bState )
{
  if( eCpx<eCPX_FST || eCpx>eCPX_LST ) return( FALSE );

  m_bCpx[ eCpx ] = bState;

  return( TRUE );
}

//*****************************************************************************
// Set the analysis temperature in degrees Celcius.
//
// Argument List:
//   fTempC - The temperature in degrees Celcius
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bSetTempC( float fTempC )
{
  if( fTempC<-100.0 || fTempC>250.0 ) return( FALSE );

  m_fTempC = fTempC;

  return( TRUE );
}

//*****************************************************************************
// Get the simulation start point as a string.
//
// Return Values:
//   Success - The string value
//   Failure - An empty string

const wxString & Simulation::rosGetSwpStart( void )
{
  static  wxString  osStart;
  double  df1;

  df1 = (double) m_fSwp[ eSWP_START ];
  if( ! ConvertType::bFltToStrEng( df1, osStart ) )
    osStart.Empty( );

  return( osStart );
}

//*****************************************************************************
// Get the simulation stop point as a string.
//
// Return Values:
//   Success - The string value
//   Failure - An empty string

const wxString & Simulation::rosGetSwpStop( void )
{
  static  wxString  osStop;
  double  df1;

  df1 = (double) m_fSwp[ eSWP_STOP ];
  if( ! ConvertType::bFltToStrEng( df1, osStop ) )
    osStop.Empty( );

  return( osStop );
}

//*****************************************************************************
// Get the simulation step point as a string.
//
// Return Values:
//   Success - The string value
//   Failure - An empty string

const wxString & Simulation::rosGetSwpStep( void )
{
  static  wxString  osStep;
  double  df1;

  df1 = (double) m_fSwp[ eSWP_STEP ];
  if( ! ConvertType::bFltToStrEng( df1, osStep ) )
    osStep.Empty( );

  return( osStep );
}

//*****************************************************************************
// Get the simulation temperature as a string.
//
// Return Values:
//   Success - The string value
//   Failure - An empty string

const wxString & Simulation::rosGetTempC( void )
{
  static  wxString  osTempC;
  double  df1;

  df1 = (double) m_fTempC;
  if( ! ConvertType::bFltToStrEng( df1, osTempC ) )
    osTempC.Empty( );

  return( osTempC );
}

//*****************************************************************************
// Add a node to the list of test points.
//
// Arguments:
//   rosNode - The node to be add to the list of test points
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bAddTstNode( const wxString & rosNode )
{
  if( rosNode.IsEmpty( ) )                            return( FALSE );
  if( m_oasTstNodes.Index( rosNode ) != wxNOT_FOUND ) return( TRUE );

  m_oasTstNodes.Add( rosNode );
  m_oasTstNodes.Sort( &iCompare );

  return( TRUE );
}

//*****************************************************************************
// Add a component to the list of test points.
//
// Arguments:
//   rosCpnt - The component to be add to the list of test points
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bAddTstCpnt( const wxString & rosCpnt )
{
  if( rosCpnt.IsEmpty( ) )                            return( FALSE );
  if( m_oasTstCpnts.Index( rosCpnt ) != wxNOT_FOUND ) return( TRUE );

  m_oasTstCpnts.Add( rosCpnt );
  m_oasTstCpnts.Sort( &iCompare );

  return( TRUE );
}

//*****************************************************************************
// Add a command to the list of simulator commands.
//
// Arguments:
//   rosCpnt - The component to be add to the list of test points
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bAddSimCmd( const wxString & rosCmd )
{
  if( rosCmd.IsEmpty( ) ) return( FALSE );

  m_oasSimCmds.Add( rosCmd );

  return( TRUE );
}

//*****************************************************************************
// Set the value of the source (eg. voltage or current) string.
//
// Arguments:
//   rosSrc - The source string
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  Simulation::bSetSigSrc( const wxString & rosSrc )
{
  wxString  os1;
  size_t    sz1;

  ClrSigSrc( );

  // Validate a few things before proceeding
  if( m_oasCpnts.IsEmpty( ) ) return( FALSE );
  if( rosSrc    .IsEmpty( ) ) return( TRUE );

  // Check that the source component exists
  os1 = rosSrc.BeforeFirst( wxT(' ') );
  for( sz1=0; sz1<m_oasCpnts.GetCount( ); sz1++ )
    if( os1 == m_oasCpnts[ sz1 ].BeforeFirst( wxT(' ') ) )
      break;
  if( sz1 >= m_oasCpnts.GetCount( ) ) return( FALSE );

  // Set the source component
  m_osSigSrc[ 0 ]   = rosSrc;
  m_osSigSrc[ 1 ]   = m_oasCpnts[ sz1 ];
  m_oasCpnts[ sz1 ] = rosSrc;

  return( TRUE );
}

//*****************************************************************************
// Reset the value of the source string to its original value.

void  Simulation::ClrSigSrc( void )
{
  wxString  os1;
  size_t    sz1;

  // Check if  the source component is already clear
  if( m_osSigSrc[ 0 ].IsEmpty( ) ) return;
  if( m_oasCpnts     .IsEmpty( ) ) return;

  // Find the component used as a source and reset it to it's original value
  os1 = m_osSigSrc[ 0 ].BeforeFirst( wxT(' ') );
  m_osSigSrc[ 0 ].Empty( );
  for( sz1=0; sz1<m_oasCpnts.GetCount( ); sz1++ )
  { // Search for the source component
    if( os1 == m_oasCpnts[ sz1 ].BeforeFirst( wxT(' ') ) )
    { // Reset the component to its original value
      m_oasCpnts[ sz1 ] = m_osSigSrc[ 1 ];
      break;
    }
  }

  m_osSigSrc[ 0 ].Empty( );
  m_osSigSrc[ 1 ].Empty( );
}

//*****************************************************************************
// Print the object attributes.
//
// Argument List :
//   psPrefix - A prefix to every line displayed (usually just spaces)

void  Simulation::Print( const char * psPrefix )
{
  int     i1;
  size_t  sz1;

  cout << psPrefix << "m_eSimrType        : ";
  switch( m_eSimrType )
  {
    case eSIMR_GNUCAP  : cout << "eSIMR_GNUCAP";  break;
    case eSIMR_NGSPICE : cout << "eSIMR_NGSPICE"; break;
    case eSIMR_NONE    : cout << "eSIMR_NONE";    break;
  }
  cout << '\n';

  cout << psPrefix << "m_eAnaType         : ";
  switch( m_eAnaType )
  {
    case eANA_OP   : cout << "eANA_OP";   break;  // Operating point analysis
    case eANA_DC   : cout << "eANA_DC";   break;  // DC analysis
    case eANA_AC   : cout << "eANA_AC";   break;  // AC analysis
    case eANA_TR   : cout << "eANA_TR";   break;  // Transient analysis
    case eANA_FO   : cout << "eANA_FO";   break;  // Fourier analysis
    case eANA_DI   : cout << "eANA_DI";   break;  // Distortion analysis
    case eANA_NO   : cout << "eANA_NO";   break;  // Noise analysis
    case eANA_PZ   : cout << "eANA_PZ";   break;  // Pole-zero analysis
    case eANA_SE   : cout << "eANA_SE";   break;  // Sensitivity analysis
    case eANA_TF   : cout << "eANA_TF";   break;  // Transfer function analysis
    case eANA_NONE : cout << "eANA_NONE"; break;  // Analysis type not set
  }
  cout << '\n';

  cout << psPrefix << "m_fSwp[ n ]        :";
  for( i1=(int)eSWP_START; i1<=(int)eSWP_START; i1++ )
    cout << ' ' << m_fSwp[ i1 ];
  cout << '\n';

  cout << psPrefix << "m_bPar[ n ]        :";
  for( i1=(int)ePAR_FST; i1<=(int)ePAR_LST; i1++ )
    cout << ( m_bPar[ i1 ] ? " TRUE " : " FALSE" );
  cout << '\n';

  cout << psPrefix << "m_bCpx[ n ]        :";
  for( i1=(int)eCPX_FST; i1<=(int)eCPX_LST; i1++ )
    cout << ( m_bCpx[ i1 ] ? " TRUE " : " FALSE" );
  cout << '\n';

  cout << psPrefix << "m_fTempC           : " << m_fTempC << '\n';

  cout << psPrefix << "m_osaTstNodes[ n ] :";
  for( sz1=0; sz1<m_oasTstNodes.GetCount( ); sz1++ )
    cout << ' ' << m_oasTstNodes[ sz1 ].mb_str( );
  cout << '\n';

  cout << psPrefix << "m_osaTstCpnts[ n ] :";
  for( sz1=0; sz1<m_oasTstCpnts.GetCount( ); sz1++ )
    cout << ' ' << m_oasTstCpnts[ sz1 ].mb_str( );
  cout << '\n';

  cout << psPrefix << "m_osaSimCmds[ n ]  :";
  for( sz1=0; sz1<m_oasSimCmds.GetCount( ); sz1++ )
    cout << ' ' << m_oasSimCmds[ sz1 ].mb_str( );
  cout << '\n';

  cout << psPrefix << "m_osSigSrc[ 0 ]    : " << m_osSigSrc[ 0 ].mb_str( ) << '\n'
       << psPrefix << "m_osSigSrc[ 1 ]    : " << m_osSigSrc[ 1 ].mb_str( ) << '\n';

  cout << psPrefix << "m_osOPTIONS        : " << m_osOPTIONS.mb_str( ) << '\n';
}

//*****************************************************************************
