//*****************************************************************************
//                                PnlGnuCapFO.cpp                             *
//                               -----------------                            *
//  Started     : 26/04/2004                                                  *
//  Last Update : 05/10/2007                                                  *
//  Copyright   : (C) 2004 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "gnucap/panels/PnlGnuCapFO.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlGnuCapFO, PnlAnaBase )

  EVT_BUTTON( ID_BTN_SETUP, PnlGnuCapFO::OnBtnSetup )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.

PnlGnuCapFO::PnlGnuCapFO( wxWindow * poWin )
                        : PnlAnaBase( poWin ), m_oDlgSigGenCfg( poWin )
{
  bSetSimrType( eSIMR_GNUCAP );
  bSetAnaType ( eANA_FO );

  Create( );  // Create the analysis panel
  bClear( );  // Clear all object attributes
}

//*****************************************************************************
// Destructor.

PnlGnuCapFO::~PnlGnuCapFO( )
{
}

//*****************************************************************************
// Create the display objects.

void  PnlGnuCapFO::Create( void )
{
  wxStaticBox * poSbxSigSrc;
  int           ix, iy;

  // Set the frequency sweep parameter labels
  m_oSbxSwpPars.GetSize( &ix, &iy );
  m_oSbxSwpPars.SetSize(  ix, 167 );
  m_oSbxSwpPars.SetLabel( wxT(" Frequency Sweep ") );
  m_oPnlStart  .bSetName( wxT("Start Frequency") );
  m_oPnlStop   .bSetName( wxT("Stop Frequency") );
  m_oPnlStep   .bSetName( wxT("Step Increment") );

  // Set sweep parameter units
  m_oPnlStart.bSetUnitsType( ChoUnits::eUNITS_FREQ );
  m_oPnlStop .bSetUnitsType( ChoUnits::eUNITS_FREQ );
  m_oPnlStep .bSetUnitsType( ChoUnits::eUNITS_FREQ );

  // Create and add the initial conditions radio buttons
  wxString  osInitC[ 3 ] = { wxT("Warm "), wxT("Use ICs"), wxT("Cold ") };
  m_oRbxInitC.Create( this, ID_UNUSED, wxT(" Initial Conditions "),
                      wxPoint( 13, 123 ), wxDefaultSize, 3, osInitC, 3 );

  // Create and add generator controls
  poSbxSigSrc = new wxStaticBox( this, ID_UNUSED, wxT(" Signal Source "),
                           wxPoint( 6, 175 ),  wxSize( 241, 55 ) );
  m_oChoSigSrcCpnt.Create( this, ID_CHO_SIGSRCCPNT,
                           wxPoint( 18, 195 ),  wxSize( 107, 29 ) );
  m_oBtnSetup     .Create( this, ID_BTN_SETUP, wxT("Setup ..."),
                           wxPoint( 130, 195 ), wxSize( 105, 29 ) );
  poSbxSigSrc->SetOwnFont( FONT_SLANT );

  // Create and add output parameters
  CreateOutputs( );

  // Create and add the analysis temperature controls
  CreateTemp( );
}

//*****************************************************************************
// Create the parameters to included in the results file output.

void  PnlGnuCapFO::CreateOutputs( void )
{
  if( bIsCreatedOutputs( ) ) return;

  // Create and add output parameters
  m_oSbxOutputs.Create( this, ID_UNUSED,    wxT(" Outputs "),
                        wxPoint( 455,  5 ), wxSize( 110, 122 ) );
  m_oCbxAct    .Create( this, ID_CBX_ACT,   wxT("Act. Mag."),
                        wxPoint( 461, 24 ) );
  m_oCbxRel    .Create( this, ID_CBX_REL,   wxT("Rel. Mag."),
                        wxPoint( 461, 49 ) );
  m_oCbxPhase  .Create( this, ID_CBX_PHASE, wxT("Phase"),
                        wxPoint( 461, 74 ) );
  m_oCbxDbMag  .Create( this, ID_CBX_DBMAG, wxT("Mag. in dB"),
                        wxPoint( 461, 99 ) );

  m_oSbxOutputs.SetOwnFont( FONT_SLANT );
}

//*****************************************************************************
// Set the state of the initial conditions radio box.
//
// Argument List:
//   eInitC - The enumerated initial conditions specifier
//
// Return Values:
//   Success - TRUE
//   Failure - FALSE

bool  PnlGnuCapFO::bSetInitC( eInitCType eInitC )
{
  if( eInitC<eINITC_FST || eInitC>eINITC_LST )      return( FALSE );
#if wxCHECK_VERSION( 2,8,0 )
  if( m_oRbxInitC.GetCount( ) < (uint) eInitC + 1 ) return( FALSE );
#else
  if( m_oRbxInitC.GetCount( ) < (int)  eInitC + 1 ) return( FALSE );
#endif

  m_oRbxInitC.SetSelection( (int) eInitC );

  return( TRUE );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapFO::bClear( void )
{
  bool  bRtnValue=TRUE;

  // Clear the base class
  if( ! PnlAnaBase::bClear( ) )    bRtnValue = FALSE;

  // Set default sweep values
  m_oPnlStart.bSetValue( (float)   1.0 );
  m_oPnlStop .bSetValue( (float) 100.0 );
  m_oPnlStep .bSetValue( (float)   1.0 );
  m_oPnlStart.bSetUnits( wxT("kHz") );
  m_oPnlStop .bSetUnits( wxT("kHz") );
  m_oPnlStep .bSetUnits( wxT("kHz") );

  // Set default initial condition values
  if( ! bSetInitC( eINITC_WARM ) ) bRtnValue = FALSE;

  // Clear the signal generator configuration dialog
  m_oDlgSigGenCfg.bClear( );
  m_oDlgSigGenCfg.bSetDefaults( );

  // Set the output parameter check box default values
  m_oCbxAct  .SetValue( TRUE );
  m_oCbxRel  .SetValue( FALSE );
  m_oCbxPhase.SetValue( FALSE );
  m_oCbxDbMag.SetValue( TRUE );

  return( bRtnValue );
}

//*****************************************************************************
// Load information from a Simulation object.
//
// Argument List:
//   roSim - The Simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapFO::bLoad( Simulation & roSim )
{
  bool      bRtnValue=TRUE;
  wxString  os1;
  uint      ui1;

  // Load the components into the signal source choice box
  LoadSigSrcCpnts( roSim, wxT("VI") );

  // Don't go any further if the analysis type doesn't match
  if( roSim.eGetAnaType( ) != eGetAnaType( ) )            return( TRUE );

  // Set the initial conditions
  if( ! bSetInitC( (eInitCType) roSim.iGetSwpScale( ) ) ) bRtnValue = FALSE;

  // Set the sweep source
  if( ! PnlAnaBase::bSetSigSrc( roSim ) )                 bRtnValue = FALSE;

  // Load the generator setup dialog
  const wxArrayString & roasCmds = roSim.roasGetSimCmds( );
  for( ui1=0; ui1<roasCmds.GetCount( ); ui1++ )
  {
    os1 = roasCmds.Item( ui1 ).Upper( );
    if( os1.StartsWith( wxT(".GEN") ) )
    {
      if( ! m_oDlgSigGenCfg.bSetGenCmd( roasCmds.Item( ui1 ) ) )
                                                          bRtnValue = FALSE;
      break;
    }
  }

  // Perform any base class load tasks
  if( ! PnlAnaBase::bLoad( roSim ) )                      bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Save information to a Simulation object.
//
// Argument List:
//   roSim - The Simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapFO::bSave( Simulation & roSim )
{
  wxStringTokenizer  ostk1;
  wxString           os1;
  int                i1;
  size_t             sz1;

  // Execute base class save tasks
  PnlAnaBase::bSave( roSim );

  // Set the initial conditions
  i1 = m_oRbxInitC.GetSelection( );
  if( ! roSim.bSetSwpScale( i1 ) )
    SetErrMsg( wxT("Initial conditions couldn't be set.") );

  // Set the component to be used as the sweep source
  if( m_oChoSigSrcCpnt.GetSelection( ) > 0 )
  {
    // Find the component to used as a generator
    for( sz1=0; sz1<roSim.roasGetCpnts( ).GetCount( ); sz1++ )
    {
      os1 = roSim.roasGetCpnts( ).Item( sz1 );
      if( os1.BeforeFirst( wxT(' ') ) == m_oChoSigSrcCpnt.GetStringSelection( ) )
        break;
    }

    // Convert the component string to a generator string
    ostk1.SetString( os1 );
    os1.Empty( );
    for( sz1=0; sz1<3; sz1++ ) os1 << ostk1.GetNextToken( ) << wxT(' ');
    os1 << wxT("GENERATOR(1)");

    // Add the generator definition command line
    roSim.bAddSimCmd( m_oDlgSigGenCfg.rosGetGenCmd( ) );
  }
  else os1.Empty( );
  if( ! roSim.bSetSigSrc( os1 ) )
    SetErrMsg( wxT("Signal source (generator) component couldn't be set.") );

  // Store the output parameters to be extracted
  if( m_oCbxAct.IsChecked( ) || m_oCbxRel.IsChecked( ) )
  {
    roSim.bSetOutCpx( eCPX_MAG,   m_oCbxMag  .GetValue( ) );
    roSim.bSetOutCpx( eCPX_PHASE, m_oCbxPhase.GetValue( ) );
    roSim.bSetOutCpx( eCPX_REAL,  m_oCbxReal .GetValue( ) );
    roSim.bSetOutCpx( eCPX_IMAG,  m_oCbxImag .GetValue( ) );
    roSim.bSetOutCpx( eCPX_MAGDB, m_oCbxMagDb.GetValue( ) );
  }
  else
    SetErrMsg( wxT("An Outputs magnitude value hasn't been specified.") );

  return( bIsOk( ) );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Generator component setup button control event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  PnlGnuCapFO::OnBtnSetup( wxCommandEvent & roEvtCmd )
{
  m_oDlgSigGenCfg.CenterOnParent( );
  m_oDlgSigGenCfg.ShowModal( );
}

//*****************************************************************************
