/*
 * Copyright 2010 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.gradle.api.file;

import groovy.lang.Closure;
import org.gradle.api.Action;
import org.gradle.api.tasks.util.PatternFilterable;
import org.gradle.api.specs.Spec;

import java.util.Map;
import java.io.FilterReader;
import java.util.regex.Pattern;

/**
 * A set of specifications for copying files.  This includes:
 *
 * <ul>
 *
 * <li>source directories (multiples allowed)
 *
 * <li>destination directory
 *
 * <li>ANT like include patterns
 *
 * <li>ANT like exclude patterns
 *
 * <li>File relocating rules
 *
 * <li>renaming rules
 *
 * <li>content filters
 *
 * </ul>
 *
 * CopySpecs may be nested by passing a closure to one of the from methods.  The closure creates a child CopySpec and
 * delegates methods in the closure to the child. Child CopySpecs inherit any values specified in the parent. This
 * allows constructs like:
 * <pre>
 * into('webroot')
 * exclude('**&#47;.svn/**')
 * from('src/main/webapp') {
 *    include '**&#47;*.jsp'
 * }
 * from('src/main/js') {
 *    include '**&#47;*.js'
 * }
 * </pre>
 *
 * In this example, the <code>into</code> and <code>exclude</code> specifications at the root level are inherited by the
 * two child CopySpecs.
 *
 * @author Steve Appling
 * @see org.gradle.api.tasks.Copy Copy Task
 * @see org.gradle.api.Project#copy(groovy.lang.Closure) Project.copy()
 */
public interface CopySpec extends CopySourceSpec, CopyProcessingSpec, PatternFilterable {
    /**
     * Specifies whether case-sensitive pattern matching should be used.
     *
     * @return true for case-sensitive matching.
     */
    boolean isCaseSensitive();

    /**
     * Specifies whether case-sensitive pattern matching should be used for this CopySpec.
     *
     * @param caseSensitive true for case-sensitive matching.
     */
    void setCaseSensitive(boolean caseSensitive);

    /**
     * Tells if empty target directories will be included in the copy.
     *
     * @return <tt>true</tt> if empty target directories will be included in the copy, <tt>false</tt> otherwise
     */
    boolean getIncludeEmptyDirs();

    /**
     * Controls if empty target directories should be included in the copy.
     *
     * @param includeEmptyDirs <tt>true</tt> if empty target directories should be included in the copy, <tt>false</tt> otherwise
     */
    void setIncludeEmptyDirs(boolean includeEmptyDirs);

    /**
     * Adds the given specs as a child of this spec.
     * @param sourceSpecs The specs to add
     * @return this
     */
    CopySpec with(CopySpec... sourceSpecs);

    // CopySourceSpec overrides to broaden return type

    /**
     * {@inheritDoc}
     */
    CopySpec from(Object... sourcePaths);

    /**
     * {@inheritDoc}
     */
    CopySpec from(Object sourcePath, Closure c);

    // PatternFilterable overrides to broaden return type

    /**
     * {@inheritDoc}
     *
     * @see org.gradle.api.tasks.util.PatternFilterable Pattern Format
     */
    CopySpec setIncludes(Iterable<String> includes);

    /**
     * {@inheritDoc}
     *
     * @see org.gradle.api.tasks.util.PatternFilterable Pattern Format
     */
    CopySpec setExcludes(Iterable<String> excludes);

    /**
     * {@inheritDoc}
     *
     * @see org.gradle.api.tasks.util.PatternFilterable Pattern Format
     */
    CopySpec include(String... includes);

    /**
     * {@inheritDoc}
     *
     * @see org.gradle.api.tasks.util.PatternFilterable Pattern Format
     */
    CopySpec include(Iterable<String> includes);

    /**
     * {@inheritDoc}
     *
     * @see org.gradle.api.tasks.util.PatternFilterable Pattern Format
     */
    CopySpec include(Spec<FileTreeElement> includeSpec);

    /**
     * {@inheritDoc}
     *
     * @see org.gradle.api.tasks.util.PatternFilterable Pattern Format
     */
    CopySpec include(Closure includeSpec);

    /**
     * {@inheritDoc}
     *
     * @see org.gradle.api.tasks.util.PatternFilterable Pattern Format
     */
    CopySpec exclude(String... excludes);

    /**
     * {@inheritDoc}
     *
     * @see org.gradle.api.tasks.util.PatternFilterable Pattern Format
     */
    CopySpec exclude(Iterable<String> excludes);

    /**
     * {@inheritDoc}
     *
     * @see org.gradle.api.tasks.util.PatternFilterable Pattern Format
     */
    CopySpec exclude(Spec<FileTreeElement> excludeSpec);

    /**
     * {@inheritDoc}
     *
     * @see org.gradle.api.tasks.util.PatternFilterable Pattern Format
     */
    CopySpec exclude(Closure excludeSpec);

    // CopyProcessingSpec overrides to broaden return type

    /**
     * {@inheritDoc}
     */
    CopySpec into(Object destPath);

    /**
     * Creates and configures a child {@code CopySpec} with the given destination path. The destination path is evaluated based on its type, as follows:
     *
     * <ul>
     *
     * <li>A closure: The return value of the closure is recursively evaluated.</li>
     *
     * <li>Anything else: The {@code toString()} value of the given path.</li>
     *
     * </ul>
     *
     * @param destPath Path to the destination directory for a Copy
     * @param configureClosure The closure to use to configure the child {@code CopySpec}.
     * @return this
     */
    CopySpec into(Object destPath, Closure configureClosure);

    /**
     * {@inheritDoc}
     */
    CopySpec rename(Closure closure);

    /**
     * {@inheritDoc}
     */
    CopySpec rename(String sourceRegEx, String replaceWith);

    /**
     * {@inheritDoc}
     */
    CopyProcessingSpec rename(Pattern sourceRegEx, String replaceWith);

    /**
     * {@inheritDoc}
     */
    CopySpec filter(Map<String, ?> properties, Class<? extends FilterReader> filterType);

    /**
     * {@inheritDoc}
     */
    CopySpec filter(Class<? extends FilterReader> filterType);

    /**
     * {@inheritDoc}
     */
    CopySpec filter(Closure closure);

    /**
     * {@inheritDoc}
     */
    CopySpec expand(Map<String, ?> properties);

    /**
     * {@inheritDoc}
     */
    CopySpec eachFile(Action<? super FileCopyDetails> action);

    /**
     * {@inheritDoc}
     */
    CopySpec eachFile(Closure closure);
}
