#include "viewer.h"
#include "tabchild.h"
#include "analysisViewer.h"
#include "checkmateViewer.h"
#include "kifuViewer.h"
#include "quiescenceViewer.h"
#include "networkViewer.h"
#include "moveGeneratorDialog.h"
#include "ui_kisendialog4.h"
#include "ui_networkdialog4.h"
#include "kifuFile.h"
#include "evalDebug.h"
#include "kisenModel.h"
#include "progressDebug.h"

#include "httpwindow.h"
#include "boardEditor2.h"

#include "osl/search/simpleHashTable.h"
#include "osl/eval/progressEval.h"
#include "osl/state/numEffectState.h"
#include "osl/progress/effect5x3.h"
#include "osl/record/kisen.h"
#include "osl/record/kakinoki.h"
#include "osl/record/csa.h"
#include "osl/record/csaString.h"
#include "osl/search/simpleHashRecord.h"
#include "osl/threatmate/mlPredictor.h"
#include "osl/threatmate/treePredictor.h"
#include <QFileDialog>
#include <qlayout.h>
#include <qtabwidget.h>
#include <qinputdialog.h>
#include <qmessagebox.h>
#include <qtextcodec.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <QClipboard>

#include <boost/scoped_ptr.hpp>

#include <stdexcept>

Viewer::Viewer(QWidget *parent, const char *name)
  : QTabWidget(parent, name), chatIcon(false)
{
  osl::eval::ProgressEval::setUp();
  osl::eval::ml::OpenMidEndingEval::setUp();
  osl::eval::ml::NewProgress::setUp();
  kifuViewer = new KifuViewer(this);
  addTab(kifuViewer, "Kifu");
  analysisViewer = 0;
  checkmateViewer = new CheckmateViewer(this);
  addTab(checkmateViewer, "Check Search");
  quiescenceViewer = new QuiescenceViewer(this);
  addTab(quiescenceViewer, "Quiescence Search");
  networkViewer = new NetworkViewer(this);
  addTab(networkViewer, "Network");
  editor = new BoardEditor2(this);
  addTab(editor, "Edit Board");

  QPixmap close = QPixmap(":/images/close.png");

  QPushButton *closeButton = new QPushButton(close, "", this);
  setCornerWidget(closeButton);

  connect(this, SIGNAL(currentChanged(QWidget *)),
	  this, SIGNAL(statusChanged()));
  connect(kifuViewer, SIGNAL(statusChanged()),
	  this, SIGNAL(statusChanged()));
  connect(kifuViewer, SIGNAL(analyzeOnlineDisabled()),
	  this, SIGNAL(analyzeOnlineDisabled()));
  connect(checkmateViewer, SIGNAL(statusChanged()),
	  this, SIGNAL(statusChanged()));
  connect(quiescenceViewer, SIGNAL(statusChanged()),
	  this, SIGNAL(statusChanged()));
  connect(networkViewer, SIGNAL(statusChanged()),
	  this, SIGNAL(statusChanged()));
  connect(closeButton, SIGNAL(clicked()),
	  this, SLOT(closeTab()));
  connect(this, SIGNAL(currentChanged(QWidget *)),
	  this, SLOT(notifyOrientation(QWidget *)));
  connect(this, SIGNAL(currentChanged(QWidget *)),
	  this, SLOT(notifyEffect(QWidget *)));

  connect(networkViewer, SIGNAL(chatReceived()),
	  this, SLOT(chatReceived()));
  connect(networkViewer, SIGNAL(painted()),
	  this, SLOT(chatDisplayed()));

  resetIcon();
}

const viewer::EvalInfo Viewer::evalState()
{
  viewer::EvalInfo result;
  TabChild *child = (TabChild *)currentPage();
  osl::state::NumEffectState nState(child->getState());
  if (nState.kingPiece(osl::BLACK).isOnBoard() &&
      nState.kingPiece(osl::WHITE).isOnBoard())
  {
    osl::eval::ml::OpenMidEndingEval eval(nState);
#ifdef EVAL_QUAD
    const int piece_scale = osl::progress::ml::NewProgress::maxProgress()/3;
#else
    const int piece_scale = osl::progress::ml::NewProgress::maxProgress();
#endif
    result.total = eval.value() / piece_scale;
    result.opening = eval.openingValue();
    result.mid1 = eval.midgameValue();
#ifdef EVAL_QUAD
    result.mid2 = eval.midgame2Value();
#endif
    result.ending  = eval.endgameValue();
    osl::eval::PieceEval piece(nState);
    result.tension = result.total - piece.value();
  }
  return result;
}

void Viewer::progressOfState(osl::stl::vector<int>& out)
{
  TabChild *child = (TabChild *)currentPage();
  out.clear();
  // TODO: better abstraction needed
  if (currentPage() == editor)
  {
    for (int i = 0; i < 3; i++)
      out.push_back(0);
    return;
  }
  osl::state::NumEffectState nState(child->getState());
  if (nState.kingPiece(osl::BLACK).isOnBoard() &&
      nState.kingPiece(osl::WHITE).isOnBoard())
  {
    osl::progress::ml::NewProgress progress(nState);
    out.push_back(progress.progress16().value());
    out.push_back(progress.progress16(osl::BLACK).value());
    out.push_back(progress.progress16(osl::WHITE).value());
  }
  else
  {
    for (int i = 0; i < 3; i++)
      out.push_back(0);
  }
}

std::pair<double,double> Viewer::checkmateProbability()
{
  TabChild *child = (TabChild *)currentPage();
  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state = child->getStateAndMovesToCurrent(moves);

  if ((moves.size() == 0) || (! moves.back().isNormal()))
    return std::make_pair(0.0, 0.0);

  if (!state.kingPiece(osl::BLACK).isOnBoard() ||
      !state.kingPiece(osl::WHITE).isOnBoard())
    return std::make_pair(0.0, 0.0);

  osl::state::NumEffectState nState(state);
  for (size_t i=0; i<moves.size(); ++i)
    nState.makeMove(moves[i]);
  osl::threatmate::MlPredictor predictor;
  osl::threatmate::TreePredictor tree;
  return std::make_pair(predictor.probability(nState, moves.back()),
			tree.probability(nState, moves.back()));
}


void Viewer::forward()
{
  setChatIcon();
  TabChild *child = (TabChild *)currentPage();
  child->forward();
}

void Viewer::backward()
{
  resetIcon();
  TabChild *child = (TabChild *)currentPage();
  child->backward();
}

void Viewer::toInitialState()
{
  TabChild *child = (TabChild *)currentPage();
  child->toInitialState();
}

void Viewer::toLastState()
{
  TabChild *child = (TabChild *)currentPage();
  child->toLastState();
}

void Viewer::open(const std::string &fileName)
{
  kifuViewer->open(new CsaFile(fileName.c_str()));
  showPage(kifuViewer);
}

void Viewer::openKakinoki(const std::string &fileName)
{
  kifuViewer->open(new KakinokiFile(fileName.c_str()));
  showPage(kifuViewer);
}

void Viewer::openKi2(const std::string &fileName)
{
  kifuViewer->open(new Ki2File(fileName.c_str()));
  showPage(kifuViewer);
}

void Viewer::openUsi(const std::string &fileName)
{
  kifuViewer->open(new UsiFile(fileName.c_str()));
  showPage(kifuViewer);
}

void Viewer::open(const std::string &fileName, int index)
{
  kifuViewer->open(new KisenFile(fileName.c_str(), index));
  showPage(kifuViewer);
}

void Viewer::openEvalGraph()
{
  kifuViewer->openEvalGraph(true);
  showPage(kifuViewer);
}

void Viewer::open()
{
  QFileDialog *dialog = new QFileDialog(this);
  QString filename = kifuViewer->getFilename();
  if (!filename.isNull())
  {
    QFileInfo fileInfo(filename);
    dialog->setDir(fileInfo.dir());
  }
    
  QStringList filters = dialog->filters();
  filters << "Kisen (*.kpf *.kif)";
  filters << "CSA (*.csa)";
  filters << "Kakinoki (*.kif)";
  filters << "Usi (*.usi)";
  filters << "Shogi files (*.csa *.kpf *.kif *.ki2 *.usi)";
  dialog->setFilters(filters);
  if (dialog->exec() == QDialog::Accepted)
  {
    const QString& selectedFile = dialog->selectedFile();
    std::string fileName(selectedFile.ascii());
    if (selectedFile.endsWith(".csa"))
    {
      open(fileName);
    }
    else if (selectedFile.endsWith(".usi"))
    {
      openUsi(fileName);
    }
    else if ( selectedFile.endsWith(".ki2") || selectedFile.endsWith(".KI2") )
    {
      openKi2(fileName);
    }
    else if (selectedFile.endsWith(".kif")
	     && osl::KakinokiFile::isKakinokiFile(fileName))
    {
      openKakinoki(fileName);
    }
    else if (selectedFile.endsWith(".kif") || selectedFile.endsWith(".kpf"))
    {
      QString ipxFile(selectedFile);
      ipxFile.replace(ipxFile.length() - 3, 3, "ipx");
      osl::record::KisenIpxFile ipx(ipxFile.ascii());
      if (ipx.size() > 1000)
      {
	bool ok;
	int index = QInputDialog::getInteger("Kisen Index",
					     QString("Choose Kisen Index %1").arg(ipx.size()),
					     0, 0, ipx.size(), 1, &ok, this);
	if (ok)
	{
	  open(fileName, index);
	}
      }
      else
      {
	boost::scoped_ptr<QDialog> qdialog(new QDialog(this));
	boost::scoped_ptr<Ui_kisenDialog> dialog(new Ui_kisenDialog());
        dialog->setupUi(qdialog.get());
        boost::scoped_ptr<KisenModel> kisen_model(new KisenModel(&ipx, this));
        dialog->tableView->setModel(kisen_model.get());
        dialog->tableView->verticalHeader()->hide();
        dialog->tableView->resizeColumnsToContents();
        dialog->tableView->setShowGrid(false);
	int ret = qdialog->exec();
	if (ret == QDialog::Accepted)
	{
          QModelIndexList selected =
            dialog->tableView->selectionModel()->selection().indexes();
	  if (!selected.empty())
	  {
	    open(fileName, selected.first().row());
	  }
	}
      }
    }
    else
    {
      QMessageBox::warning(this, "Open File",
			   "Unknown file type " + selectedFile);
      return;
    }
    emit statusChanged();
  }
}

void Viewer::nextFile()
{
  const KifuFile *file = kifuViewer->getKifuFile();
  if (file == NULL)
    return;
  KifuFile *next_file = file->nextFile();
  if (next_file)
  {
    kifuViewer->open(next_file);
    showPage(kifuViewer);
    emit statusChanged();
  }
}

void Viewer::prevFile()
{
  const KifuFile *file = kifuViewer->getKifuFile();
  if (file == NULL)
    return;
  KifuFile *prev_file = file->prevFile();
  if (prev_file)
  {
    kifuViewer->open(prev_file);
    showPage(kifuViewer);
    emit statusChanged();
  }
}

void Viewer::watchFile(bool enable)
{
  // FIXME allow each kifuviewer to watch its own file
  kifuViewer->watchFile(enable);
}

void Viewer::openUrl(bool reload)
{
  HttpWindow *http = new HttpWindow(this, reload);
  http->show();
  if (http->exec() == QDialog::Accepted)
  {
    std::string filename = http->getFilename().toStdString();
    if (http->getFilename().endsWith(".kif"))
      openKakinoki(filename);
    else if (http->getFilename().endsWith(".ki2") || http->getFilename().endsWith(".KI2"))
      openKi2(filename);
    else if (http->getFilename().endsWith(".usi"))
      openUsi(filename);
    else
      open(filename);
    kifuViewer->toLastState();
  }
}

void Viewer::openUrl()
{
  openUrl(false);
}

void Viewer::reloadUrl()
{
  openUrl(true);
}

void Viewer::hirate()
{
  TabChild *child = (TabChild *)currentPage();
  if (!child->inherits("KifuViewer"))
    return;
  KifuViewer *kifu = (KifuViewer *)child;

  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state(osl::HIRATE);
  kifu->open(state, moves);
  showPage(kifu);
}

void Viewer::network()
{
  boost::scoped_ptr<QDialog> qdialog(new QDialog(this));
  boost::scoped_ptr<Ui::NetworkDialog> dialog(new Ui::NetworkDialog());
  dialog->setupUi(qdialog.get());

  qdialog->exec();

  networkViewer->setHostname(dialog->servername->text());
  networkViewer->setUsername(dialog->username->text());
  networkViewer->setPassword(dialog->password->text());
  networkViewer->connect();

  showPage(networkViewer);
}

void Viewer::view()
{
  TabChild *child = (TabChild *)currentPage();
  if (child == kifuViewer)
    return;

  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state = child->getStateAndMovesToCurrent(moves);
  kifuViewer->open(state, moves);
  kifuViewer->toLastState();
  showPage(kifuViewer);
}

void Viewer::viewInNewTab()
{
  TabChild *child = (TabChild *)currentPage();

  KifuViewer *viewer = new KifuViewer(this);
  connect(viewer, SIGNAL(statusChanged()), this, SIGNAL(statusChanged()));
  connect(viewer, SIGNAL(analyzeOnlineDisabled()),
	  this, SIGNAL(analyzeOnlineDisabled()));
  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state = child->getStateAndMovesToCurrent(moves);
  viewer->open(state, moves);
  viewer->toLastState();
  addTab(viewer, "Kifu");
  showPage(viewer);
}

int Viewer::moveCount() const
{
  TabChild *child = (TabChild *)currentPage();
  return child->moveCount();
}

osl::Player Viewer::turn() const
{
  TabChild *child = (TabChild *)currentPage();
  return child->turn();
}

const osl::state::SimpleState& Viewer::getState()
{
  TabChild *child = (TabChild *)currentPage();
  return child->getState();
}

void Viewer::analyze()
{
  analyze(false);
}

void Viewer::analyzeInNewTab()
{
  analyze(true);
}

void Viewer::analyze(bool newTab)
{
  bool tabCreated = false;
  TabChild *child = (TabChild *)currentPage();
  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state = child->getStateAndMovesToCurrent(moves);
  AnalysisViewer *av;

  if (!analysisViewer || newTab)
  {
    tabCreated = true;
    av = new AnalysisViewer(this);
    connect(av, SIGNAL(statusChanged()),
	    this, SIGNAL(statusChanged()));
  }
  else if (child->inherits("AnalysisViewer"))
  {
    av = (AnalysisViewer *)child;
  }
  else
    av = analysisViewer;

  bool success = av->analyze(state, moves);

  if (success)
  {
    if (tabCreated)
    {
      if (!analysisViewer)
	analysisViewer = av;
      addTab(av, "Analyze position");
    }
    showPage(av);
  }
  else
  {
    if (tabCreated)
    {
      delete av;
    }
  }
}

void Viewer::checkmateSearch()
{
  TabChild *child = (TabChild *)currentPage();
  const int node_limit = 100000;
  bool success = checkmateViewer->analyze(child->getState(), node_limit);
  if (success)
    showPage(checkmateViewer);
}

void Viewer::altCheckmateSearch()
{
  TabChild *child = (TabChild *)currentPage();
  const int node_limit = 100000;
  bool success = checkmateViewer->analyze(child->getState(), node_limit, true);
  if (success)
    showPage(checkmateViewer);
}

void Viewer::checkmateSimulation()
{
  TabChild *child = (TabChild *)currentPage();
  bool success = checkmateViewer->analyze(child->getState(), 0);
  if (success)
    showPage(checkmateViewer);
}

void Viewer::quiescenceSearch()
{
  TabChild *child = (TabChild *)currentPage();
  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state = child->getStateAndMovesToCurrent(moves);
  bool success = quiescenceViewer->analyze(state, moves);
  if (success)
    showPage(quiescenceViewer);
}

void Viewer::quiescenceSearchHalf()
{
  TabChild *child = (TabChild *)currentPage();
  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state = child->getStateAndMovesToCurrent(moves);
  bool success = quiescenceViewer->analyzeHalfDepth(state, moves);
  if (success)
    showPage(quiescenceViewer);
}

void Viewer::moveGenerateDialog()
{
  if (currentPage()->inherits("BoardAndListTabChild"))
  {
    ((BoardAndListTabChild *)currentPage())->moveGenerateDialog();
    return;
  }

  TabChild *child = (TabChild *)currentPage();
  child->moveGenerateDialog();
}

void Viewer::toggleOrientation()
{
  TabChild *child = (TabChild *)currentPage();
  child->toggleOrientation();
}

QString Viewer::getFilenameToSave()
{
  QString s = QFileDialog::getSaveFileName(0,
					   "CSA (*.csa)",
					   this,
					   "save file",
					   "Save moves up to current state");
  return s;
}

void Viewer::saveMovesToCurrent()
{
  QString s = getFilenameToSave();
  if (s.isNull())
    return;
  TabChild *child = (TabChild *)currentPage();
  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state = child->getStateAndMovesToCurrent(moves);
  std::ofstream os(s.ascii());
  os << state;
  for (int i = 0; i < (int)moves.size(); i++)
  {
    os << osl::record::csa::show(moves[i]) << std::endl;
  }
}

void Viewer::exportCurrent()
{
  QString s = getFilenameToSave();
  if (s.isNull())
    return;
  TabChild *child = (TabChild *)currentPage();
  std::ofstream os(s.ascii());
  os << child->getState();
}

void Viewer::moveTo(int index)
{
  TabChild *child = (TabChild *)currentPage();
  if (child == kifuViewer)
  {
    kifuViewer->updateIndex(index);
  }
}

void Viewer::copy()
{
  TabChild *child = (TabChild *)currentPage();
  child->copy();
}

void Viewer::copyBoardAndMoves()
{
  TabChild *child = (TabChild *)currentPage();
  child->copyBoardAndMoves();
}

void Viewer::copyUsi()
{
  TabChild *child = (TabChild *)currentPage();
  child->copyUsi();
}

void Viewer::copyBoardAndMovesUsi()
{
  TabChild *child = (TabChild *)currentPage();
  child->copyBoardAndMovesUsi();
}

void Viewer::paste()
{
  QWidget *widget = currentPage();
  if (widget->inherits("KifuViewer"))
  {
    QClipboard *clipboard = QApplication::clipboard();
    QString text = clipboard->text();
    if (!text.isEmpty())
    {
      try
      {
        osl::record::csa::CsaString csa(text.toUtf8());
        osl::stl::vector<osl::Move> board_moves;
        ((KifuViewer *)widget)->open(csa.getInitialState(), board_moves);
      }
      catch(...)
      {
      }
    }
  }  
}

QString Viewer::getFilename()
{
  TabChild *child = (TabChild *)currentPage();
  QString filename = child->getFilename();
  if (filename.isNull())
    return "";

  QFileInfo fi(filename);
  return fi.fileName();
}

void Viewer::closeTab()
{
  QWidget *widget = currentPage();
  if (widget->inherits("AnalysisViewer"))
  {
    removePage(widget);
    if (widget == analysisViewer)
    {
      analysisViewer = 0;
      for (int i = 0; i < count(); i++)
      {
	QWidget *child = page(i);
	if (child->inherits("AnalysisViewer"))
	{
	  analysisViewer = (AnalysisViewer *)child;
	  break;
	}
      }
    }
    delete widget;
  }
  else if (widget != kifuViewer && widget->inherits("KifuViewer"))
  {
    removePage(widget);
    delete widget;
  }
}

bool Viewer::isSenteView() const
{
  if (currentPage()->inherits("BoardTabChild"))
  {
    return ((BoardTabChild *)currentPage())->isSenteView();
  }
  return true;
}

void Viewer::setOrientation(bool gote)
{
  if (currentPage()->inherits("BoardTabChild"))
  {
    return ((BoardTabChild *)currentPage())->setOrientation(!gote);
  }
}

void Viewer::setAnalyzeOnline(bool enable)
{
  KifuViewer::setAnalyzeOnline(enable);
  if (enable)
  {
    if (currentPage()->inherits("KifuViewer")) {
      osl::state::SimpleState s;
      osl::stl::vector<osl::Move> m;
      s = ((KifuViewer *)currentPage())->getStateAndMovesToCurrent(m);
      ((KifuViewer *)currentPage())->prepareUpdateForAnalysis(s, m, 0, osl::Move());
    }
    else
      showPage(kifuViewer);
  }
}

void Viewer::notifyOrientation(QWidget *widget)
{
  bool sente;
  if (widget->inherits("BoardTabChild"))
    sente = ((BoardTabChild *)widget)->isSenteView();
  else
    sente = true;

  emit orientationChanged(sente);
}

void Viewer::enableEffect(bool on)
{
  ((TabChild *)currentPage())->enableEffect(on);
}

void Viewer::highlightLastMove(bool on)
{
  ((TabChild *)currentPage())->highlightLastMove(on);
}
void Viewer::highlightBookMove(bool on)
{
  ((TabChild *)currentPage())->highlightBookMove(on);
}

void Viewer::notifyEffect(QWidget *widget)
{
  if (widget->inherits("BoardTabChild"))
    emit effectChanged(((BoardTabChild *)widget)->effectEnabled());
}

void Viewer::kifuAnalyze()
{
  if (((TabChild *)currentPage())->inherits("KifuViewer"))
  {
    ((KifuViewer *)currentPage())->analyze();
  }
}

void Viewer::resetIcon()
{
#if defined(__APPLE__) || defined(__FreeBSD__)
  qApp->setWindowIcon(QIcon(":/images/icon.png"));
#endif
}

void Viewer::setChatIcon()
{
#if defined(__APPLE__) || defined(__FreeBSD__)
  qApp->setWindowIcon(QIcon(":/images/icon-chat.png"));
#endif
}

void Viewer::chatReceived()
{
  if (!chatIcon)
  {
    chatIcon = true;
    setChatIcon();
  }
}

void Viewer::chatDisplayed()
{
  if (chatIcon)
  {
    chatIcon = false;
    resetIcon();
  }
}

void Viewer::editState()
{
  TabChild *child = (TabChild *)currentPage();
  editor->setState(child->getState());
  showPage(editor);
}

void Viewer::testEvalDebug()
{
  TabChild *child = (TabChild *)currentPage();
  OpenMidEndingEvalDebugDialog *dialog =
    new OpenMidEndingEvalDebugDialog(child->getState(), this);
  if (child->inherits("BoardTabChild"))
  {
    BoardTabChild *board_tab_child = (BoardTabChild *)child;
    connect(board_tab_child,
	    SIGNAL(statusChanged(const osl::state::SimpleState &,
				 const osl::stl::vector<osl::Move> &,
				 int, osl::Move)),
	    dialog, SLOT(setStatus(const osl::state::SimpleState &,
				   const osl::stl::vector<osl::Move> &,
				   int, osl::Move)));
  }
  dialog->show();
  dialog->raise();
}

void Viewer::setState1()
{
  TabChild *child = (TabChild *)currentPage();
  state1.reset(new osl::state::SimpleState(child->getState()));
}

void Viewer::setState2()
{
  TabChild *child = (TabChild *)currentPage();
  state2.reset(new osl::state::SimpleState(child->getState()));
}

void Viewer::showEvalDiff()
{
  if (state1.get() != NULL && state2.get() != NULL)
  {
    OpenMidEndingEvalDiffDialog *dialog =
      new OpenMidEndingEvalDiffDialog(osl::state::NumEffectState(*state1),
                                      osl::state::NumEffectState(*state2), this);
    dialog->show();
    dialog->raise();
  }
}

void Viewer::showProgressDebug()
{
  TabChild *child = (TabChild *)currentPage();
  osl::state::NumEffectState state(child->getState());
  NewProgressDebugDialog *dialog = new NewProgressDebugDialog(state, this);
  dialog->setAttribute(Qt::WA_DeleteOnClose);
  dialog->show();
  dialog->raise();
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
