;;; Copyright (C) 2011 Team GPS.
;;; 
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;; 
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;; 
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

(ns twitter_clj.snapshot
  (:require [twitter_clj.common :as common]
            [twitter_clj.ki2 :as ki2]
            [twitter_clj.twitter :as twitter]
            [clojure.string :as str]
            [clojure.contrib.logging :as log]))

;; ==================================================
;; Global variables
;; ==================================================

(def timers
  (concat [0 60 300] (iterate #(+ % 900) 900)))


;; ==================================================
;; Functions
;; ==================================================

(defn take-pv
  [nmove]
  (letfn [(format-pv [m]
            (let [ki2moves (if (= "resign" (:pv m))
                             ""
                             (ki2/show (:pv m)))]
              (format "%d %s"
                      (* (:cp m) (if (odd? (inc nmove)) 1 -1))
                      ki2moves)))]
    (let [coll (common/sort-pv @common/pv)]
      (str/join " / " (map format-pv coll)))))
  

(defn take-snapshot
  [nmove last-move sec]
  (format "[(%d) %s] <%ds> %s" nmove last-move sec (take-pv nmove)))


(defn interval-seconds
  "Returns an interval in seconds from a start time to now."
  [start]
  (let [now (java.util.Date.)
        interval (quot (- (.getTime now) (.getTime start)) 1000)]
    interval))


(defn start-snapshot-thread
  []
  (let [[nmove last-move] (ki2/current-info)
        start-time (java.util.Date.)]
    (letfn [(tweet [sec]
              (let [s (take-snapshot nmove last-move sec)]
                (twitter/post-move s)
                s))
            (thread-main []
              (let [i (atom 0)]
                (try
                  (log/debug "Started a new snapshot thread")
                  (loop [last-pv ""]
                    (let [[t1 t2] (take 2 (drop @i timers))
                          interval (- t2 t1)]
                      (log/debug (format "Sleeping a while: %d secs..." interval))
                      (Thread/sleep (* interval 1000))
                      (log/debug "Awake!!!")
                      (swap! i inc)
                      (let [pv (take-pv nmove)]
                        (if (= last-pv pv)
                          (do
                            (log/info "The pv is still unchanged. Skipped tweeting it.")
                            (recur pv))
                          (do
                            (tweet (nth timers @i))
                            (recur pv))))))
                  (catch InterruptedException e
                    (tweet (interval-seconds start-time)))))
              (log/debug "The snapshot thread finished"))]
      (let [thread (Thread. thread-main)]
        (.start thread)
        thread))))

