/* analyzer.cc
 */
#include "analyzer.h"
#include "eval/eval.h"
#include "quiesce.h"
#include "osl/progress/effect5x3.h"
#include "osl/move_generator/legalMoves.h"
#include "osl/apply_move/applyMove.h"
#include "osl/eval/pieceEval.h"
#include <boost/thread/mutex.hpp>
#include <boost/foreach.hpp>
#include <ext/algorithm>
#include <iostream>
#include <cstdio>

// #define DEBUG_ALL

const double gpsshogi::SigmoidUtil::eps;

/* ------------------------------------------------------------------------- */

namespace 
{
  struct sort_by_index
  {
    bool operator()(const std::pair<int,double>& l, 
		    const std::pair<int,double>& r) const 
    {
      return l.first < r.first;
    }
  };
}

void gpsshogi::MoveData::sort()
{
  std::sort(diffs.begin(), diffs.end(), sort_by_index());
}

std::ostream& gpsshogi::operator<<(std::ostream& os, const gpsshogi::MoveData& md)
{
  os << "value " << md.value << " data";
  for (size_t i=0; i<md.diffs.size(); ++i) {
    os << "  " << md.diffs[i].first << " " << md.diffs[i].second;
  }
  return os << "\n";
}

/* ------------------------------------------------------------------------- */

void gpsshogi::
Analyzer::makeLeaf(HistoryState& state, const PVVector& pv)
{
  BOOST_FOREACH(Move m, pv)
    state.makeMove(m);
}

void gpsshogi::
Analyzer::analyzeLeaf(const NumEffectState& state_org, 
		      const PVVector& pv, Eval& eval,
		      MoveData& data)
{
  HistoryState state(state_org);
  makeLeaf(state, pv);

  assert(! state.state().inCheck(BLACK));
  assert(! state.state().inCheck(WHITE));

#ifdef DEBUG_ALL
  std::cerr << state;
  std::cerr << value << "\n";
#endif  
  data.clear();
  double value;
  data.diffs.reserve(eval.maxActive()*2);
  eval.features(state, value, data.diffs);
  data.value = (int)value;
#ifndef L1BALL_NO_SORT
  assert(__gnu_cxx::is_sorted(data.diffs.begin(), data.diffs.end()));
#endif
  osl::PieceEval pe(state);
  data.piece_value = pe.value();
#ifdef DEBUG_ALL
  std::cerr << "\n";
#endif
}

boost::shared_ptr<gpsshogi::InstanceData> 
gpsshogi::
Analyzer::makeInstance(double turn_coef, const MoveData& selected, const MoveData& sibling)
{
  vector<size_t> dummy;
  return makeInstance(turn_coef, selected, sibling, dummy, 0);
}

void gpsshogi::
Analyzer::makeInstance(double turn_coef, const MoveData& selected, const MoveData& sibling, InstanceData& out)
{
  vector<size_t> dummy;
  makeInstance(turn_coef, selected, sibling, dummy, 0, out);
}

boost::shared_ptr<gpsshogi::InstanceData> 
gpsshogi::
Analyzer::makeInstance(double turn_coef, const MoveData& selected, const MoveData& sibling,
		       const vector<size_t>& frequency, int min_frequency)
{
  boost::shared_ptr<InstanceData> instance(new InstanceData);
  makeInstance(turn_coef, selected, sibling, frequency, min_frequency, *instance);
  return instance;
}

void
gpsshogi::
Analyzer::makeInstance(double turn_coef, const MoveData& selected, const MoveData& sibling,
		       const vector<size_t>& frequency, int min_frequency, InstanceData& instance)
{
  assert(__gnu_cxx::is_sorted(selected.diffs.begin(), selected.diffs.end()));
  assert(__gnu_cxx::is_sorted(sibling.diffs.begin(), sibling.diffs.end()));

  instance.index.reserve(selected.diffs.size()+sibling.diffs.size()+1);
  instance.value.reserve(selected.diffs.size()+sibling.diffs.size()+1);
  instance.y = 1-std::max(0.0, turn_coef); 
  
  MoveData::vector_t::const_iterator p = sibling.diffs.begin();
  MoveData::vector_t::const_iterator q = selected.diffs.begin();

  while (p != sibling.diffs.end() && q != selected.diffs.end()) {
    if (min_frequency) {
      if (frequency[p->first] < min_frequency) {
	++p;
	continue;
      }
      if (frequency[q->first] < min_frequency) {
	++q;
	continue;
      }
    }
    const int index = std::min(p->first, q->first);
    double diff;		// sibling[i] - selected[i];
    if (p->first == q->first) {
      diff = p->second - q->second;
      ++p, ++q;
    }
    else if (p->first < q->first) {
      diff = p->second;		// seleceted[index] == 0
      ++p;
    }
    else {
      assert(p->first > q->first);
      diff = - (q->second);	// sibling[index] == 0
      ++q;
    }
    if (fabs(diff) > SigmoidUtil::eps) {
      instance.index.push_back(index);
      instance.value.push_back(diff);
    }
  }
  while (p != sibling.diffs.end()) {
    const int index = p->first;
    const double diff = p->second; // seleceted[index] == 0
    ++p;
    if (fabs(diff) > SigmoidUtil::eps) {
      instance.index.push_back(index);
      instance.value.push_back(diff);
    }
  }
  while (q != selected.diffs.end()) {
    const int index = q->first;
    const double diff = - (q->second); // sibling[index] == 0
    ++q;
    if (fabs(diff) > SigmoidUtil::eps) {
      instance.index.push_back(index);
      instance.value.push_back(diff);
    }
  }
}

void
gpsshogi::
Analyzer::makeInstanceOne(double turn_coef, const MoveData& data, 
			  const vector<size_t>& frequency, int min_frequency, InstanceData& instance)
{
  instance.index.reserve(data.diffs.size()+1);
  instance.value.reserve(data.diffs.size()+1);
  instance.y = 1-std::max(0.0, turn_coef);  
  
  for (MoveData::vector_t::const_iterator p = data.diffs.begin();
       p!=data.diffs.end(); ++p) {
    if (min_frequency && frequency[p->first] < min_frequency) 
      continue;
    const int index = p->first;
    double diff = p->second;
    instance.index.push_back(index);
    instance.value.push_back(diff);
  }
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
