/*
 * GooCanvas. Copyright (C) 2005 Damon Chaplin.
 * Released under the GNU LGPL license. See COPYING for details.
 *
 * goocanvasrect.c - rectangle item.
 */

/**
 * SECTION:goocanvasrect
 * @Title: GooCanvasRect
 * @Short_Description: a rectangle item.
 *
 * GooCanvasRect represents a rectangle item.
 *
 * It is a subclass of #GooCanvasItemSimple and so inherits all of the style
 * properties such as "stroke-color", "fill-color" and "line-width".
 *
 * It also implements the #GooCanvasItem interface, so you can use the
 * #GooCanvasItem functions such as goo_canvas_item_raise() and
 * goo_canvas_item_rotate().
 *
 * To create a #GooCanvasRect use goo_canvas_rect_new().
 *
 * To get or set the properties of an existing #GooCanvasRect, use
 * g_object_get() and g_object_set().
 *
 * To respond to events such as mouse clicks on the rect you must connect
 * to the signal handlers of the corresponding #GooCanvasRectView objects.
 * (See goo_canvas_view_get_item_view() and #GooCanvasView::item-view-created.)
 */
#include <config.h>
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include "goocanvasrect.h"
#include "goocanvasrectview.h"


enum {
  PROP_0,

  PROP_X,
  PROP_Y,
  PROP_WIDTH,
  PROP_HEIGHT,
  PROP_RADIUS_X,
  PROP_RADIUS_Y
};


static void item_interface_init (GooCanvasItemIface *iface);
static void goo_canvas_rect_get_property (GObject            *object,
					  guint               param_id,
					  GValue             *value,
					  GParamSpec         *pspec);
static void goo_canvas_rect_set_property (GObject            *object,
					  guint               param_id,
					  const GValue       *value,
					  GParamSpec         *pspec);

G_DEFINE_TYPE_WITH_CODE (GooCanvasRect, goo_canvas_rect,
			 GOO_TYPE_CANVAS_ITEM_SIMPLE,
			 G_IMPLEMENT_INTERFACE (GOO_TYPE_CANVAS_ITEM,
						item_interface_init))


static void
goo_canvas_rect_class_init (GooCanvasRectClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass*) klass;

  gobject_class->get_property = goo_canvas_rect_get_property;
  gobject_class->set_property = goo_canvas_rect_set_property;

  g_object_class_install_property (gobject_class, PROP_X,
				   g_param_spec_double ("x",
							"X",
							_("The x coordinate of the rectangle"),
							-G_MAXDOUBLE,
							G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_Y,
				   g_param_spec_double ("y",
							"Y",
							_("The y coordinate of the rectangle"),
							-G_MAXDOUBLE,
							G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_WIDTH,
				   g_param_spec_double ("width",
							_("Width"),
							_("The width of the rectangle"),
							0.0, G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_HEIGHT,
				   g_param_spec_double ("height",
							_("Height"),
							_("The height of the rectangle"),
							0.0, G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_RADIUS_X,
				   g_param_spec_double ("radius_x",
							_("Radius X"),
							_("The horizontal radius to use for rounded corners"),
							0.0, G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_RADIUS_Y,
				   g_param_spec_double ("radius_y",
							_("Radius Y"),
							_("The vertical radius to use for rounded corners"),
							0.0, G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));
}


static void
goo_canvas_rect_init (GooCanvasRect *canvas_rect)
{

}


/**
 * goo_canvas_rect_new:
 * @parent: the parent item, or %NULL. If a parent is specified, it will assume
 *  ownership of the item, and the item will automatically be freed when it is
 *  removed from the parent. Otherwise call g_object_unref() to free it.
 * @x: the x coordinate of the left of the rectangle.
 * @y: the y coordinate of the top of the rectangle.
 * @width: the width of the rectangle.
 * @height: the height of the rectangle.
 * @first_property: the name of the first property to set, or %NULL.
 * @...: the remaining property names and values to set, terminated with a
 *  %NULL.
 * 
 * Creates a new rectangle item.
 * 
 * <!--PARAMETERS-->
 *
 * Here's an example showing how to create a rectangle at (100,100) with a
 * width of 200 and a height of 100.
 *
 * <informalexample><programlisting>
 *  GooCanvasItem *rect = goo_canvas_rect_new (mygroup, 100.0, 100.0, 200.0, 100.0,
 *                                             "stroke-color", "red",
 *                                             "line-width", 5.0,
 *                                             "fill-color", "blue",
 *                                             NULL);
 * </programlisting></informalexample>
 * 
 * Returns: a new rectangle item.
 **/
GooCanvasItem*
goo_canvas_rect_new (GooCanvasItem *parent,
		     gdouble        x,
		     gdouble        y,
		     gdouble        width,
		     gdouble        height,
		     const gchar   *first_property,
		     ...)
{
  GooCanvasItem *item;
  GooCanvasRect *rect;
  va_list args;

  item = g_object_new (GOO_TYPE_CANVAS_RECT, NULL);
  rect = GOO_CANVAS_RECT (item);

  rect->x = x;
  rect->y = y;
  rect->width = width;
  rect->height = height;
  rect->radius_x = 0;
  rect->radius_y = 0;

  va_start (args, first_property);
  g_object_set_valist (G_OBJECT (item), first_property, args);
  va_end (args);

  if (parent)
    {
      goo_canvas_item_add_child (parent, item, -1);
      g_object_unref (item);
    }

  return item;
}


static GooCanvasItemView*
goo_canvas_rect_create_view (GooCanvasItem     *item,
			     GooCanvasView     *canvas_view,
			     GooCanvasItemView *parent_view)
{
  return goo_canvas_rect_view_new (canvas_view, parent_view,
				   (GooCanvasRect*) item);
}


static void
item_interface_init (GooCanvasItemIface *iface)
{
  iface->create_view = goo_canvas_rect_create_view;
}


static void
goo_canvas_rect_get_property (GObject              *object,
			      guint                 prop_id,
			      GValue               *value,
			      GParamSpec           *pspec)
{
  GooCanvasRect *rect = (GooCanvasRect*) object;

  switch (prop_id)
    {
    case PROP_X:
      g_value_set_double (value, rect->x);
      break;
    case PROP_Y:
      g_value_set_double (value, rect->y);
      break;
    case PROP_WIDTH:
      g_value_set_double (value, rect->width);
      break;
    case PROP_HEIGHT:
      g_value_set_double (value, rect->height);
      break;
    case PROP_RADIUS_X:
      g_value_set_double (value, rect->radius_x);
      break;
    case PROP_RADIUS_Y:
      g_value_set_double (value, rect->radius_y);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}


static void
goo_canvas_rect_set_property (GObject              *object,
			      guint                 prop_id,
			      const GValue         *value,
			      GParamSpec           *pspec)
{
  GooCanvasRect *rect = (GooCanvasRect*) object;

  switch (prop_id)
    {
    case PROP_X:
      rect->x = g_value_get_double (value);
      break;
    case PROP_Y:
      rect->y = g_value_get_double (value);
      break;
    case PROP_WIDTH:
      rect->width = g_value_get_double (value);
      break;
    case PROP_HEIGHT:
      rect->height = g_value_get_double (value);
      break;
    case PROP_RADIUS_X:
      rect->radius_x = g_value_get_double (value);
      break;
    case PROP_RADIUS_Y:
      rect->radius_y = g_value_get_double (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }

  g_signal_emit_by_name (rect, "changed", TRUE);
}

