/* pmud Monitor panel applet
 *
 * By Uwe Steinmann <uwe@steinmann.cx>
 *
 * Enjoy.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or 
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License  
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>

#include "panel-applet.h"
#include "gpmudmon.h"
#include "gtkgauge.h"

GtkWidget *counter;
GtkTooltips *tooltip;
#ifdef DEBUG
FILE *logfile;
#endif

extern void prefs_cb(BonoboUIComponent *uic, gpointer data, const gchar *verbname);
#ifdef PMUD
static void sleep_cb(BonoboUIComponent *uic, gpointer data, const gchar *verbname);
#endif
static void help_cb (BonoboUIComponent *uic, gpointer data, const gchar *verbname);
static void about_cb(BonoboUIComponent *uic, gpointer data, const gchar *verbname);

int count_timer(gpointer data);
int open_pmudinfo(PmudData *pd);
static PMUD_STATUS * get_pmud_status(PmudData *pd);

static const BonoboUIVerb GPmudMon_Applet_menu_verbs [] = {
	        BONOBO_UI_VERB ("GPmudMon_Properties", prefs_cb),
	        BONOBO_UI_VERB ("GPmudMon_Help",       help_cb),
	        BONOBO_UI_VERB ("GPmudMon_About",      about_cb),
	        BONOBO_UI_VERB_END
};

/* write_log() {{{
 * Output an log entry.
 */
void write_log(const char *fmt, ...) {
#ifdef DEBUG
	char msg[256];
	va_list ap;

	va_start(ap, fmt);
	vsprintf(msg, fmt, ap);

	fprintf(logfile, msg);
	fflush(logfile);

	va_end(ap);
#endif
}
/* }}} */

/* gpmudmon_load_pixbuf() {{{
 *
 */
GdkPixbuf *
gpmudmon_load_pixbuf( const char *file )
{
	char *filelocation;
	GError *error = NULL;
	GdkPixbuf *pixbuf;
	
	filelocation = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_PIXMAP, file, FALSE, NULL);
	pixbuf = gdk_pixbuf_new_from_file (filelocation, &error);
	g_free (filelocation);
	
	if (error) {
		g_warning (G_STRLOC ": cannot open %s: %s", file, error->message);
		g_error_free (error);
		return 0;
	}
	
	return pixbuf;
}
/* }}} */

/* about_cb() {{{
 *
 */
static void
about_cb(BonoboUIComponent *uic, gpointer d, const gchar *verbname)
{
	const gchar *authors[4];
	GtkWidget *about;
	GdkPixbuf *pixbuf;

	const gchar *documenters[] = {
		NULL
	};
	
	authors[0] = "Uwe Steinmann <uwe@steinmann.cx>";
	authors[1] = "http://sourceforge.net/projects/gpmudmon";
	authors[2] = NULL;

	pixbuf = gpmudmon_load_pixbuf( "gpmudmon-applet/gpmudmon-logo.png");

	about = GTK_WIDGET(GTK_DIALOG(gnome_about_new (
		"GNOME Pmu Monitor",
		VERSION,
		"(C) 2000-2004 Uwe Steinmann",
		_("This applet displays the battery status as provided by the power mananagement unit on apple computers."),
		authors,
		documenters,
		NULL,
		pixbuf)));

	if(pixbuf)
		gdk_pixbuf_unref (pixbuf);

	gtk_window_set_wmclass (GTK_WINDOW (about), "GNOME PMU monitor", "GNOME PMU monitor");
	gtk_widget_show(about);
	
	return;
}
/* }}} */

/* help_cb() {{{
 *
 */
static void
help_cb (BonoboUIComponent *uic, gpointer data, const gchar *verbname)
{
	GError *error = NULL;

	gnome_help_display(PACKAGE, NULL, &error);
	if (error) {
		gnome_error_dialog(error->message);
		g_error_free (error);
		error = NULL;
	}
}
/* }}} */

#ifdef PMUD
/* sleep_cb() {{{
 *
 */
static void
sleep_cb(BonoboUIComponent *uic, gpointer data, const gchar *verbname)
{
	int error;
	PmudData *pd = (PmudData *) data;
	error = write(pd->pmudinfo, "sleep\n", 6);
	return;
}
/* }}} */
#endif

/* change_size_cb() {{{
 *
 */
void change_size_cb(PanelApplet *applet, gint size, PmudData *pd)
{
// 	gtk_widget_set_size_request(data->globalbox, 2*size+30, size);
	gtk_widget_set_usize(pd->globalbox, 2*size+30, size);
	gtk_widget_set_usize(pd->bar1, size/4, -1);
	gtk_widget_set_usize(pd->bar2, size/4, -1);
	gtk_widget_set_usize(pd->gaugeframe1, size*1.7, size);
	gtk_widget_set_usize(pd->gaugeframe2, size*1.7, size);
}
/* }}} */

/* change_orient_cb() {{{
 *
 */
void change_orient_cb(PanelApplet *applet, PanelAppletOrient o, PmudData *data)
{
}
/* }}} */

/* count_timer() {{{
 *
 */
int
count_timer(gpointer data)
{
	PmudData *pd = (PmudData *) data;
	char *tiptext;
	char buffer[100];
	int tmin;
	BAT_STATUS *batleft, *batright;

	pd->pmud_status = get_pmud_status(pd);
	if(pd->pmud_status == NULL) {
		gtk_tooltips_set_tip(GTK_TOOLTIPS(pd->tooltip), GTK_WIDGET(pd->app), pd->pmud_status->errmsg, NULL);
		return 1;
	}

	if(pd->valid_data == 0) {
		tiptext = malloc(1);
		tiptext[0] = '\0';

		batright = pd->pmud_status->bat;
		batleft = batright++;
		write_log("batleft = 0x%X\n", batleft);
		write_log("batright = 0x%X\n", batright);
		/* The first two flags for left and right seems to the same */
		if(batleft->charging)
			gtk_image_set_from_pixbuf(GTK_IMAGE(pd->status_pixmap), pd->charge_in_pixmap);
		else if(batleft->ac_on)
			gtk_image_set_from_pixbuf(GTK_IMAGE(pd->status_pixmap), pd->ac_in_pixmap);
		else
			gtk_image_set_from_pixbuf(GTK_IMAGE(pd->status_pixmap), pd->battery_in_pixmap);

		// Battery left
		if(batleft->present) {
			gtk_gauge_set_value(GTK_GAUGE(pd->gauge1), batleft->current);
			gtk_widget_show(pd->bar1);
			if(batleft->charging)
				tmin = (batleft->maxcharge - batleft->charge) * 66.0 / batleft->current;
			else
				tmin = batleft->charge * 59.2 / -batleft->current;
			if(batleft->charge == batleft->maxcharge)
				tmin = 0;
			sprintf(buffer, _("Battery left: %2.2fV, %2.1f%%, %d:%02dh"), batleft->voltage,
		                     	(float) batleft->charge/batleft->maxcharge*100,
		                     	tmin/60, tmin%60);
			tiptext = realloc(tiptext, strlen(tiptext)+strlen(buffer)+1);
			strcat(tiptext, buffer);

			gtk_progress_bar_update(GTK_PROGRESS_BAR(pd->bar1), (float) batleft->charge/batleft->maxcharge);
		} else {
			sprintf(buffer, _("Battery left not present"));
			tiptext = realloc(tiptext, strlen(tiptext)+strlen(buffer)+1);
			strcat(tiptext, buffer);
			gtk_widget_hide(pd->bar1);
		}

		// Battery right
		if(batright->present) {
			if(batleft->present) {
				tiptext = realloc(tiptext, strlen(tiptext)+2+1);
				strcat(tiptext, "\n");
			}
			gtk_gauge_set_value(GTK_GAUGE(pd->gauge2), batright->current);
			gtk_widget_show(pd->bar2);
			if(batright->charging)
				tmin = (batright->maxcharge - batright->charge) * 66.0 / batright->current;
			else
				tmin = batright->charge * 59.2 / -batright->current;
			if(batright->charge == batright->maxcharge)
				tmin = 0;
			buffer[0] = '\n';
			sprintf(&buffer[1], _("Battery right: %2.2fV, %2.1f%%, %d:%02dh"),
					           batright->voltage,
			               (float) batright->charge/batright->maxcharge*100,
			               tmin/60, tmin%60);
			tiptext = realloc(tiptext, strlen(tiptext)+strlen(buffer)+1);
			strcat(tiptext, buffer);

			gtk_progress_bar_update(GTK_PROGRESS_BAR(pd->bar2), (float) batright->charge/batright->maxcharge);
		} else {
			buffer[0] = '\n';
			sprintf(&buffer[1], _("Battery right not present"));
			tiptext = realloc(tiptext, strlen(tiptext)+strlen(buffer)+1);
			strcat(tiptext, buffer);
			gtk_widget_hide(pd->bar2);
		}

		if(batleft->present && pd->show_current)
			gtk_widget_show(pd->gaugeframe1);
		else
			gtk_widget_hide(pd->gaugeframe1);
		if(batright->present && pd->show_current)
			gtk_widget_show(pd->gaugeframe2);
		else
			gtk_widget_hide(pd->gaugeframe2);

		gtk_tooltips_set_tip(GTK_TOOLTIPS(pd->tooltip), GTK_WIDGET(pd->app), tiptext, NULL);
		free(tiptext);
	} else {
#ifdef PMUD
		fprintf(stderr, "Try to reopen connection\n");
		pd->pmudinfo = open_pmudinfo(pd);
#endif
	}

	return(1);
}
/* }}} */

#ifdef PMUD
/* open_pmudinfo() {{{
 *
 */
#define IM_BUFSIZE  100
int
open_pmudinfo(PmudData *pd)
{
	int s;
	int ourPort;
	struct sockaddr_in sai;
	struct hostent *he;
	char buf[IM_BUFSIZE];

	if(pd->pmudinfo) {
		close(pd->pmudinfo);
		pd->pmudinfo = 0;
	}

	ourPort = pd->port;
	if (ourPort == 0)
		ourPort = 879;

	s = socket(PF_INET, SOCK_STREAM, 0);
	if (s) {
		he = gethostbyname(pd->servername);
		if (he) {
			sai.sin_family = AF_INET;
			sai.sin_port = htons(ourPort);
			sai.sin_addr = *(struct in_addr*) he->h_addr;

			if (connect(s, (struct sockaddr*) &sai, sizeof(sai)) == 0) {
				pd->pmudinfo = s;
//				fprintf(stderr, "Open connection: %d\n", s);
			} else
				close(s);
		} else
			close(s);
	}

	read(pd->pmudinfo, buf, IM_BUFSIZE);
	return(pd->pmudinfo);
}
/* }}} */

/* get_pmud_status() {{{
 *
 */
static PMUD_STATUS *
get_pmud_status(PmudData *pd)
{
	char buf[IM_BUFSIZE], *ptr1;
	fd_set fs;
	struct timeval tv;
	int n12, n13, n14, n15;
	PMUD_STATUS *status;

	if(pd->pmudinfo) {
		int error;
		FD_ZERO(&fs);
		FD_SET(pd->pmudinfo, &fs);
		tv.tv_sec = 3;
		tv.tv_usec = 0;
	
		if ((select(FD_SETSIZE, &fs, 0, 0, &tv) > 0)) { // && (!feof(pd->pmudinfo))) {
			if(pd->pmud_status)
				free(pd->pmud_status);
			pd->pmud_status = (PMUD_STATUS *) malloc(sizeof(PMUD_STATUS));
			status = pd->pmud_status;
		
			error = 0;
			memset(buf, 0, IM_BUFSIZE);
			error = read(pd->pmudinfo, buf, IM_BUFSIZE);

			error = 0;
			ptr1 = strchr(buf, '{');
			ptr1++; /* Points now on first character after first '{', which are the flags */
			status->bat[0].present = ptr1[2] == '1' ? 1 : 0;
			status->bat[0].charging = ptr1[1] == '1' ? 1 : 0;
			status->bat[0].ac_on = ptr1[0] == '1' ? 1 : 0;
			ptr1 += 3; /* Skip the flags */
			
			/* If left battery is present then ptr1 should point to ' ' and not '}'.
			   We better test for it
			*/
			if(status->bat[0].present && (*ptr1 == ' ')) {
				sscanf(ptr1, "%d %d %d %d", &n12, &n13, &n14, &n15);
				status->bat[0].charge = n12;
				status->bat[0].maxcharge = n13;
				status->bat[0].current = (float) n14;
				status->bat[0].voltage = n15/1000.0;
			}
			ptr1 = strchr(ptr1, '{'); /* Find the next '{' for rightt battery */
			if(ptr1 == NULL) {  /* This should not happen */
				free(pd->pmud_status);
				return(NULL);
			}
			ptr1++; /* Now points on first flag */
			status->bat[1].present = ptr1[2] == '1' ? 1 : 0;
			status->bat[1].charging = ptr1[1] == '1' ? 1 : 0;
			status->bat[1].ac_on = ptr1[0] == '1' ? 1 : 0;
			ptr1 += 3; /* Skip the flags */

			/* If right battery is present then ptr1 should point to ' ' and not '}'.
			   We better test for it
			*/
			if(status->bat[1].present && (*ptr1 == ' ')) {
				sscanf(ptr1, "%d %d %d %d", &n12, &n13, &n14, &n15);
				status->bat[1].charge = n12;
				status->bat[1].maxcharge = n13;
				status->bat[1].current = (float) n14;
				status->bat[1].voltage = n15/1000.0;
			}
		}
		error = write(pd->pmudinfo, "\n", 1);
	} else {
		fprintf(stderr, "Try to reopen connection\n");
		pd->pmudinfo = open_pmudinfo(pd);
		return(NULL);
	}

	return(pd->pmud_status);
}
#undef IM_BUFSIZE
/* }}} */
#else
/* open_pmudinfo() {{{
 *
 */
#define IM_BUFSIZE  100
int
open_pmudinfo(PmudData *pd)
{
	return(1);
}
/* }}} */

void get_value_string(char *buf, char *ret) {
	char *str;
	str = strchr(buf, ':');
	if(str) {
		str++;
		sscanf(str, "%s", ret);
	} else {
		ret[0] = '\0';
	}
}

void get_value_int(char *buf, int *ret) {
	char *str;
	str = strchr(buf, ':');
	if(str) {
		str++;
		sscanf(str, "%d", ret);
	} else {
		*ret = 0;
	}
}

/* get_pmud_status() {{{
 *
 */
static PMUD_STATUS *
get_pmud_status(PmudData *pd)
{
	char buf[IM_BUFSIZE], buf2[IM_BUFSIZE], *ptr1;
	FILE *pmuinfo, *pmubat[2];
	int i, numbats, acpower;
	PMUD_STATUS *status = NULL;

	if(!pd->pmud_status) {
		pd->pmud_status = (PMUD_STATUS *) calloc(sizeof(PMUD_STATUS), 1);
	}
	status = pd->pmud_status;

	write_log("--------\nReading status of pmu\n");
	pmuinfo = fopen("/proc/pmu/info", "r");
	if(NULL == pmuinfo) {
		snprintf(status->errmsg, 99, _("Could not open /proc/pmu/info"));
		return NULL;
	}
	numbats = 0;
	while(!feof(pmuinfo)) {
		char *str;
		fgets(buf, IM_BUFSIZE-1, pmuinfo);
		if(!strncmp(buf, "Battery count", 13)) {
			get_value_int(buf, &numbats);
			write_log("Battery count = %d\n", numbats);
		} else if(!strncmp(buf, "AC Power", 8)) {
			get_value_int(buf, &acpower);
			write_log("AC Power = %d\n", acpower);
		}
	}
	fclose(pmuinfo);

	write_log("Found %d batteries\n", numbats);
	if(numbats) {
		int error;
		char flags[20];

		for(i=0; i<numbats; i++) {
			int dummy;
			sprintf(buf, "/proc/pmu/battery_%d", i);
			pmubat[i] = fopen(buf, "r");
			if(pmubat[i]) {
				write_log("Successfully opened file '%s'\n", buf);
				while(!feof(pmubat[i])) {
					char *str;
					fgets(buf, IM_BUFSIZE-1, pmubat[i]);
					if(!strncmp(buf, "flags", 5)) {
						get_value_string(buf, flags);
						if(flags[0] != '\0') {
							write_log("flags = %s\n", flags);
							status->bat[i].present = flags[7] >= '1' ? 1 : 0;
							status->bat[i].charging = flags[7] == '3' ? 1 : 0;
						}
					} else if(!strncmp(buf, "charge", 6)) {
						get_value_int(buf, &(status->bat[i].charge));
						write_log("charge = %d\n", (status->bat[i].charge));
					} else if(!strncmp(buf, "max_charge", 10)) {
						get_value_int(buf, &(status->bat[i].maxcharge));
						write_log("max_charge = %d\n", (status->bat[i].maxcharge));
					} else if(!strncmp(buf, "current", 7)) {
						get_value_int(buf, &dummy);
						status->bat[i].current = (float) dummy;;
						write_log("current = %f\n", (status->bat[i].current));
					} else if(!strncmp(buf, "voltage", 7)) {
						get_value_int(buf, &dummy);
						status->bat[i].voltage = dummy / 1000.0;
						write_log("voltage = %f\n", (status->bat[i].voltage));
					} else if(!strncmp(buf, "time rem.", 9)) {
					}
				}
				status->bat[i].ac_on = acpower;
				fclose(pmubat[i]);
			} else {
				write_log("Failed to open '%s'\n", buf);
				status->bat[i].present = 0;
			}
		}
	}
	return(status);
}
#undef IM_BUFSIZE
/* }}} */
#endif

/* create_monitor() {{{
 *
 */
static GtkWidget
*create_monitor(PmudData *pd)
{
	GtkWidget *hbox;
	GtkAdjustment *adj;
	GtkAdjustment *adjgauge;

	hbox = gtk_hbox_new(FALSE, 4);

//	label=gtk_label_new(_("Refresh rate (in milli seconds)"));
	adj=(GtkAdjustment *)gtk_adjustment_new(200,0,1000,1000,100,10);
	pd->bar1 = gtk_progress_bar_new_with_adjustment(adj);
	gtk_widget_set_usize(pd->bar1, 10, 16);
  gtk_progress_bar_set_discrete_blocks(GTK_PROGRESS_BAR(pd->bar1), 10);
	gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(pd->bar1), GTK_PROGRESS_BOTTOM_TO_TOP);
	gtk_box_pack_start(GTK_BOX(hbox),pd->bar1,FALSE,FALSE,0);

	adj=(GtkAdjustment *)gtk_adjustment_new(500,0,1000,1000,100,10);
	pd->bar2 = gtk_progress_bar_new_with_adjustment(adj);
	gtk_widget_set_usize(pd->bar2, 10, 16);
	gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(pd->bar2), GTK_PROGRESS_BOTTOM_TO_TOP);
	gtk_box_pack_start(GTK_BOX(hbox),pd->bar2,FALSE,FALSE,0);

	pd->battery_in_pixmap = gpmudmon_load_pixbuf("gpmudmon-applet/battery.png");
	pd->ac_in_pixmap = gpmudmon_load_pixbuf("gpmudmon-applet/ac.png");
	pd->charge_in_pixmap = gpmudmon_load_pixbuf("gpmudmon-applet/charge.png");

	pd->status_pixmap = gtk_image_new();
	gtk_image_set_from_pixbuf(GTK_IMAGE(pd->status_pixmap), pd->battery_in_pixmap);
	gtk_box_pack_start(GTK_BOX(hbox),pd->status_pixmap, FALSE, FALSE, 0);

	/* left battery */
	pd->gaugeframe1 = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(pd->gaugeframe1), GTK_SHADOW_IN);
	adjgauge = GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, 100, 0.01, 0.1, 0));
	pd->gauge1 = gtk_gauge_new(adjgauge);
	gtk_gauge_set_range(GTK_GAUGE(pd->gauge1), -1500, 1500);
	gtk_container_add(GTK_CONTAINER(pd->gaugeframe1), pd->gauge1);
	gtk_widget_set_usize(pd->gaugeframe1, 80, 38);
	gtk_box_pack_start(GTK_BOX(hbox),pd->gaugeframe1,FALSE,FALSE,0);

	/* right battery */
	pd->gaugeframe2 = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(pd->gaugeframe2), GTK_SHADOW_IN);
	adjgauge = GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, 100, 0.01, 0.1, 0));
	pd->gauge2 = gtk_gauge_new(adjgauge);
	gtk_gauge_set_range(GTK_GAUGE(pd->gauge2), -1500, 1500);
	gtk_container_add(GTK_CONTAINER(pd->gaugeframe2), pd->gauge2);
	gtk_widget_set_usize(pd->gaugeframe2, 80, 38);
	gtk_box_pack_start(GTK_BOX(hbox),pd->gaugeframe2,FALSE,FALSE,0);

	gtk_widget_show(hbox);
	return hbox;
}
/* }}} */

/* save_config() {{{
 *
 */
void
save_config(PmudData *pd) {
	panel_applet_gconf_set_int(PANEL_APPLET(pd->app), "updatetime", pd->updatetime, NULL);
	panel_applet_gconf_set_int(PANEL_APPLET(pd->app), "width", pd->width, NULL);
	panel_applet_gconf_set_int(PANEL_APPLET(pd->app), "height", pd->height, NULL);
	panel_applet_gconf_set_int(PANEL_APPLET(pd->app), "port", pd->port, NULL);
	panel_applet_gconf_set_string(PANEL_APPLET(pd->app), "name", pd->servername, NULL);
	panel_applet_gconf_set_int(PANEL_APPLET(pd->app), "showcurrent", pd->show_current, NULL);
}
/* }}} */

/* applet_pixel_size_changed_cb() {{{
 *
 */
static void
applet_pixel_size_changed_cb (GtkWidget *applet, int size, PmudData *pd)
{
//	if(pd->height <= 15)
	gtk_widget_set_usize(pd->globalbox, -1, size);
	gtk_widget_set_usize(pd->bar1, size/4, -1);
	gtk_widget_set_usize(pd->bar2, size/4, -1);
	gtk_widget_set_usize(pd->gaugeframe1, size*1.7, size);
	gtk_widget_set_usize(pd->gaugeframe2, size*1.7, size);
}
/* }}} */

/* load_config() {{{
 *
 */
void
load_config(PmudData *pd) {
	GError *error;

	pd->updatetime = panel_applet_gconf_get_int(PANEL_APPLET(pd->app), "updatetime", NULL);
	if(pd->updatetime < 1000)
		pd->updatetime = 1000;
	pd->width = panel_applet_gconf_get_int(PANEL_APPLET(pd->app), "width", NULL);
	pd->height = panel_applet_gconf_get_int(PANEL_APPLET(pd->app), "height", NULL);
	pd->port = panel_applet_gconf_get_int(PANEL_APPLET(pd->app), "port", NULL);
	if(pd->port == 0)
		pd->port = 879;
	pd->servername = (char *) panel_applet_gconf_get_string(PANEL_APPLET(pd->app), "name", NULL);
	if(pd->servername == NULL)
		pd->servername = (char *) strdup("localhost");
	pd->show_current = panel_applet_gconf_get_int(PANEL_APPLET(pd->app), "showcurrent", NULL);
}
/* }}} */

/* gpmudmon_applet_fill() {{{
 *
 */
static gboolean
gpmudmon_applet_fill (PanelApplet *applet)
{
	PmudData *pd;

	pd = g_new0(PmudData, 1);
	pd->app = applet;
	pd->port = 879;
	pd->servername = (char *) strdup("localhost");
	pd->updatetime = 10000;
	pd->show_current = 1;

	load_config(pd);
	pd->globalbox = create_monitor(pd);

	pd->valid_data = 0;
#ifdef PMUD
	pd->pmudinfo = 0;
	pd->pmudinfo = open_pmudinfo(pd);
#endif
	pd->pmud_status = NULL;
	gtk_container_add(GTK_CONTAINER(pd->app), pd->globalbox);

  panel_applet_setup_menu_from_file (PANEL_APPLET(applet),
			             NULL,
		              "GNOME_GPmudMon_Applet.xml",
			             NULL,
		              GPmudMon_Applet_menu_verbs,
		              pd);

	g_signal_connect(pd->app, "change_size", (GCallback) change_size_cb, pd);
	g_signal_connect(pd->app, "change_orient", (GCallback) change_orient_cb, pd);

	gtk_widget_show_all(GTK_WIDGET(pd->app));

	pd->tooltip=gtk_tooltips_new();
	gtk_tooltips_set_tip(GTK_TOOLTIPS(pd->tooltip), GTK_WIDGET(applet), "PMU Monitor", NULL);

#ifdef DEBUG
	logfile = fopen("/tmp/gpmudmon.log", "w");
#endif
	count_timer(pd);
	pd->timeout = gtk_timeout_add(pd->updatetime, (GtkFunction)count_timer, pd);
	return(TRUE);
}
/* }}} */

/* gpmudmon_applet_factory() {{{
 *
 */
static gboolean
gpmudmon_applet_factory(PanelApplet *applet,
                     const gchar *iid,
							       gpointer data)
{
	gboolean retval = FALSE;

  if (!strcmp (iid, "OAFIID:GNOME_GPmudMon_Applet"))
		retval = gpmudmon_applet_fill (applet);

	return retval;
}
/* }}} */

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:GNOME_GPmudMon_Applet_Factory",
		                         PANEL_TYPE_APPLET,
		                         "Gnome PMU Monitor",
		                         "0",
		                         gpmudmon_applet_factory,
		                         NULL)

