// Options.cs
// 
// Copyright (C) 2008, James P. Michels III <james.p.michels@gmail.com>
// Copyright © 2008 David Paleino <d.paleino@gmail.com>
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//

using System;
using System.Data;
using Mono.Data.SqliteClient;
using Mono.Unix;
using Gtk;

namespace GnomeRDP
{
    /// <summary>
    /// Class to handle application options (user preferences, ...)
    /// </summary>
	public class Options
	{
        private const string optionTableName = "appOptions";
		
		private const string optionUseKeyring = "useKeyring";
        
        public Options() {
            CheckDatabase();
        }
        
#region Methods
  		/// <summary>
		/// Get an application option from database, without throwing exceptions.
		/// </summary>
		/// <param name="name">
		/// A <see cref="System.String"/>, the option name.
		/// </param>
		/// <param name="defaultValue">
		/// A <see cref="System.String"/>, the default value if none is found.
		/// </param>
		/// <returns>
		/// A <see cref="System.String"/>, the final value.
		/// </returns>
		private static string Get(string name, string defaultValue) {
            Sqlite db = new Sqlite(null);
            try {
                db.Connect(Defines.DatabasePath);
                db.Query(string.Format("SELECT * FROM {0} WHERE name = '{1}'",
                                       optionTableName, name));
                IDataReader result;
                while (db.FetchRow(out result)) {
                    return result["value"].ToString();
                }
                return defaultValue;
            } catch (Exception ex) {
                // we had some problem during the connection...
                
				Gdk.Threads.Enter();
                string message = string.Format(Catalog.GetString(
                                               "An exception has been caught "+
                                               "during database querying. "+
                                               "Please provide the following "+
                                               "information to the project "+
                                               "developers:\n\n<b>Source</b>: {0}\n"+
                                               "<b>Message</b>:{1}"), ex.Source,
                                               ex.Message);
                // TODO: check if this could be a SIGSEGV.
                MessageDialog md = new MessageDialog(null,
                                                     DialogFlags.DestroyWithParent,
                                                     MessageType.Error,
                                                     ButtonsType.Ok,
                                                     true,
                                                     message);
                md.Run();
                md.Destroy();
                Gdk.Threads.Leave();
				
				// FIXME: we shouldn't really return anything...
				// Yes, we should. Thats the point of the default parameter.
				return defaultValue;
            } finally {
                if (db != null)
                    db.Close();
            }
		}
		
        /// <summary>
        /// Set an application option.
        /// </summary>
        /// <param name="name">
        /// A <see cref="System.String"/>, the option name.
        /// </param>
        /// <param name="value">
        /// A <see cref="System.String"/>, the option value.
        /// </param>
        /// <returns>
        /// A <see cref="System.Boolean"/>, true for success.
        /// </returns>
        private static bool Set(string name, string value) {
            Sqlite db = new Sqlite(null);
            try {
                db.Connect(Defines.DatabasePath);
                db.Query(string.Format("INSERT OR REPLACE INTO {0} (name, value) "+
                                       "VALUES ('{1}', '{2}')", optionTableName,
                                       name, value));
            } catch (Exception ex) {
                // we had some problem during the insert...
                Gdk.Threads.Enter();
                string message = string.Format(Catalog.GetString(
                                               "An exception has been caught "+
                                               "during database INSERT. "+
                                               "Please provide the following "+
                                               "information to the project "+
                                               "developers:\n\n<b>Source</b>: {0}\n"+
                                               "<b>Message</b>:{1}"), ex.Source,
                                               ex.Message);
                // TODO: check if this could be a SIGSEGV.
                MessageDialog md = new MessageDialog(null,
                                                     DialogFlags.DestroyWithParent,
                                                     MessageType.Error,
                                                     ButtonsType.Ok,
                                                     true,
                                                     message);
                md.Run();
                md.Destroy();
                Gdk.Threads.Leave();
                return false;
            } finally {
                if (db != null)
                    db.Close();
            }
			return true;
        }
        
        /// <summary>
        /// Creates the options table if it doesn't exist.
        /// </summary>
        public static void CheckDatabase() {
            Sqlite db = new Sqlite(null);
            try {
                db.Connect(Defines.DatabasePath);
                db.Query(string.Format("CREATE TABLE IF NOT EXISTS {0} "+
                                       "(name VARCHAR(300) PRIMARY KEY, "+
                                       "value VARCHAR(300))", optionTableName));
            } catch (Exception ex) {
                // we had some problem during the connection...
                Gdk.Threads.Enter();
                string message = string.Format(Catalog.GetString(
                                               "An exception has been caught "+
                                               "during database initialization. "+
                                               "Please provide the following "+
                                               "information to the project "+
                                               "developers:\n\n<b>Source</b>: {0}\n"+
                                               "<b>Message</b>:{1}"), ex.Source,
                                               ex.Message);
                MessageDialog md = new MessageDialog(null,
                                                     DialogFlags.DestroyWithParent,
                                                     MessageType.Error,
                                                     ButtonsType.Ok,
                                                     true,
                                                     message);
                md.Run();
                md.Destroy();
                Gdk.Threads.Leave();
            } finally {
                if (db != null)
                    db.Close();
            }
        }
#endregion
		
#region Properties
        /// <value>
        /// Whether to use GNOME's Keyring or not.
        /// </value>
        public static bool UseKeyring {
            get {
                return bool.Parse(Get("UseKeyring", bool.TrueString));
            }
            set {
                Set("UseKeyring", value.ToString());
            }
        }
#endregion
	}
}
