------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2006 Dmitriy Anisimkov.                                   --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: dec_conv_test.adb,v 1.3 2008/05/05 12:05:40 vagul Exp $

--  Example of Decimal<->OCINumber convertions.

with Ada.Text_IO;

with OCI.Thick.Gen_OCINumber;
with OCI.Thick.Number_Functions;
with GNAT.Float_Control;

with System;

with OCI.Thick.Number_Functions;

procedure Dec_Conv_Test is
   use OCI.Thick.Number_Functions;

   type Decimal_Type is delta 0.001 digits System.Max_Digits;

   Num : Decimal_Type := Decimal_Type'First + Decimal_Type'Small;
   Num_Range : Decimal_Type := 1000.0;

   procedure Test;

   procedure Test is
      function To_Decimal is
        new OCI.Thick.Gen_OCINumber.Num_To_Decimal (Decimal_Type);

      function To_Number is
        new OCI.Thick.Gen_OCINumber.Decimal_To_Number (Decimal_Type);

      Numb : OCI.Thick.OCINumber := To_Number (Num);

      Fmt : constant String := "FM999999999999900.999";

      Image : constant String := To_String (Numb, Fmt);

      Half_Up : constant OCI.Thick.OCINumber :=
        To_Number (Long_Float (Decimal_Type'Small) / 1.99999999999999);
      Half_Down : constant OCI.Thick.OCINumber :=
        To_Number (Long_Float (Decimal_Type'Small) / 2.00000000000001);
      Half : constant OCI.Thick.OCINumber :=
        To_Number (Long_Float (Decimal_Type'Small) / 2.0);

   begin
      if Image (Image'Last - 2 .. Image'Last) = "00." then
         Ada.Text_IO.Put_Line (Image & Num'Img);
      end if;

      if To_Decimal (Numb) /= Num then
         raise Constraint_Error with "Error in " & Image & Num'Img;

      elsif To_Decimal (Numb + Half_Down) /= Num then
         raise Constraint_Error with "Error in down round.";

      elsif To_Decimal (Numb + Half_Up) /= Num + Decimal_Type'Small then
         raise Constraint_Error with
           "Error in up round " & To_String (Numb + Half_Up)
           & Decimal_Type'Image (Num + Decimal_Type'Small);

      elsif Sign (Numb) < 0
        and then To_Decimal (Numb - Half) /= Num - Decimal_Type'Small
      then
         raise Constraint_Error with
           "Error in negative half round " & To_String (Numb - Half);

      elsif Sign (Numb) > 0
        and then To_Decimal (Numb + Half) /= Num + Decimal_Type'Small
      then
         raise Constraint_Error with
           "Error in positive half round " & To_String (Numb + Half);

      elsif Sign (Numb) = 0 and then not Is_Zero (Numb) then
         raise Constraint_Error with
           "Error in zero detection " & To_String (Numb);
      end if;

      if To_String (Round (Numb, Decimal_Type'Aft), Fmt) /= Image then
         Ada.Text_IO.Put_Line ("Error in " & Image);
         raise Constraint_Error;
      end if;

   exception
      when others =>
         Ada.Text_IO.Put_Line (To_String (Numb));
         raise;
   end Test;

begin
   loop
      Test;

      if Num = Num_Range then
         Num := Decimal_Type'Last - Num_Range;

      elsif Num = Decimal_Type'First + Num_Range then
         Num := -Num_Range;

      elsif Num = Decimal_Type'Last then
         exit;

      else
         Num := Num + Decimal_Type'Small;
      end if;

   end loop;
end Dec_Conv_Test;