with Ada.Characters.Handling;
with OCI.Thick.DB;

package body OCI.Thick.Containers.Strings is

   function To_Bind_Name (Item : String) return Bind_Names.Bounded_String;
   pragma Inline (To_Bind_Name);

   -----------
   -- Clear --
   -----------

   procedure Clear
     (Container : in out Container_Type;
      Name      : in     String;
      Iteration : in     Positive := 1) is
   begin
      Set
        (Container => Container,
         Item      => Null_Data,
         Name      => Name,
         Iteration => Iteration);
   end Clear;

   ---------
   -- Get --
   ---------

   procedure Get
     (Container : in     Container_Type;
      Item      :    out Data_Holder;
      Name      : in     String;
      Iteration : in     Positive := 1) is
   begin
      Get
        (Container => Container,
         Item      => Item,
         Position  => Container.Index.Element (To_Bind_Name (Name)),
         Iteration => Iteration);
   end Get;

   function Get
     (Container : in Container_Type;
      Name      : in String;
      Iteration : in Positive := 1) return Integer
   is
      Result : Data_Holder;

   begin
      Get
        (Container => Container,
         Item      => Result,
         Name      => Name,
         Iteration => Iteration);

      return Result.Int;
   end Get;

   function Get
     (Container : in Container_Type;
      Name      : in String;
      Iteration : in Positive := 1) return Long_Float
   is
      Result : Data_Holder;

   begin
      Get
        (Container => Container,
         Item      => Result,
         Name      => Name,
         Iteration => Iteration);

      return Result.Flt;
   end Get;

   function Get
     (Container : in Container_Type;
      Name      : in String;
      Iteration : in Positive := 1) return String
   is
      Result : Data_Holder;

   begin
      Get
        (Container => Container,
         Item      => Result,
         Name      => Name,
         Iteration => Iteration);

      return Ada.Strings.Unbounded.To_String (Result.Str);
   end Get;

   function Get
     (Container : in Container_Type;
      Name      : in String;
      Iteration : in Positive := 1) return OCINumber
   is
      Result : Data_Holder;

   begin
      Get
        (Container => Container,
         Item      => Result,
         Name      => Name,
         Iteration => Iteration);

      return Result.Numb;
   end Get;

   function Get
     (Container : in Container_Type;
      Name      : in String;
      Iteration : in Positive := 1) return Ada.Calendar.Time
   is
      Result : Data_Holder;

   begin
      Get
        (Container => Container,
         Item      => Result,
         Name      => Name,
         Iteration => Iteration);

      return Result.Dat;
   end Get;

   function Get
     (Container : in Container_Type;
      Name      : in String;
      Iteration : in Positive := 1) return Lobs.Locator
   is
      Result : Data_Holder;

   begin
      Get
        (Container => Container,
         Item      => Result,
         Name      => Name,
         Iteration => Iteration);

      return Result.Loc;
   end Get;

   function Get
     (Container : in Container_Type;
      Name      : in String;
      Iteration : in Positive := 1) return Data_Holder
   is
      Result : Data_Holder;
   begin
      Get
        (Container => Container,
         Item      => Result,
         Name      => Name,
         Iteration => Iteration);

      return Result;
   end Get;

   -----------
   -- Index --
   -----------

   function Index
     (Container : in Container_Type; Name : String) return Natural
   is
      C : constant Name_Indexes.Cursor
        := Container.Index.Find (To_Bind_Name (Name));

   begin
      if Name_Indexes.Has_Element (C) then
         return Name_Indexes.Element (C);

      else
         return 0;
      end if;
   end Index;

   -------------
   -- Is_Null --
   -------------

   function Is_Null
     (Container : in Container_Type;
      Name      : in String;
      Iteration : in Positive := 1) return Boolean is
   begin
      return Is_Null
               (Container => Container,
                Position  => Container.Index.Element (To_Bind_Name (Name)),
                Iteration => Iteration);
   end Is_Null;

   -------------------------
   -- Name_Bind_Positions --
   -------------------------

   procedure Name_Bind_Positions
     (Container : in out Container_Type; Statement : in DB.Statement) is
   begin
      Container.Index.Clear;

      for J in 1 .. DB.Bind_Count (Statement) loop
         Container.Index.Insert
           (To_Bind_Name (DB.Bind_Name (Statement, J)), J);
      end loop;
   end Name_Bind_Positions;

   ---------------------------
   -- Name_Define_Positions --
   ---------------------------

   procedure Name_Define_Positions
     (Container : in out Container_Type; Statement : in DB.Statement) is
   begin
      Container.Index.Clear;

      for J in 1 .. DB.Bind_Count (Statement) loop
         Container.Index.Insert
           (To_Bind_Name (DB.Column_Name (Statement, J)), J);
      end loop;
   end Name_Define_Positions;

   ---------
   -- Set --
   ---------

   procedure Set
     (Container : in out Container_Type;
      Item      : in     Data_Holder;
      Name      : in     String;
      Iteration : in     Positive := 1) is
   begin
      Set
        (Container => Container,
         Item      => Item,
         Position  => Container.Index.Element (To_Bind_Name (Name)),
         Iteration => Iteration);
   end Set;

   procedure Set
     (Container : in out Container_Type;
      Item      : in     Integer;
      Name      : in     String;
      Iteration : in     Positive := 1) is
   begin
      Set
        (Container => Container,
         Item      => Data_Holder'(Kind => Type_Integer, Int => Item),
         Name      => Name,
         Iteration => Iteration);
   end Set;

   procedure Set
     (Container : in out Container_Type;
      Item      : in     Long_Float;
      Name      : in     String;
      Iteration : in     Positive := 1) is
   begin
      Set
        (Container => Container,
         Item      => Data_Holder'(Kind => Type_Long_Float, Flt => Item),
         Name      => Name,
         Iteration => Iteration);
   end Set;

   procedure Set
     (Container : in out Container_Type;
      Item      : in     String;
      Name      : in     String;
      Iteration : in     Positive := 1) is
   begin
      Set
        (Container => Container,
         Item      => Data_Holder'
                        (Kind => Type_String,
                         Str  =>
                           Ada.Strings.Unbounded.To_Unbounded_String (Item)),
         Name      => Name,
         Iteration => Iteration);
   end Set;

   procedure Set
     (Container : in out Container_Type;
      Item      : in     OCINumber;
      Name      : in     String;
      Iteration : in     Positive := 1) is
   begin
      Set
        (Container => Container,
         Item      => Data_Holder'(Kind => Type_Number, Numb => Item),
         Name      => Name,
         Iteration => Iteration);
   end Set;

   procedure Set
     (Container : in out Container_Type;
      Item      : in     Ada.Calendar.Time;
      Name      : in     String;
      Iteration : in     Positive := 1) is
   begin
      Set
        (Container => Container,
         Item      => Data_Holder'(Kind => Type_Date, Dat => Item),
         Name      => Name,
         Iteration => Iteration);
   end Set;

   procedure Set
     (Container : in out Container_Type;
      Item      : in     Lobs.Locator;
      Name      : in     String;
      Iteration : in     Positive := 1) is
   begin
      Set
        (Container => Container,
         Item      => Data_Holder'(Kind => Type_Lob, Loc => Item),
         Name      => Name,
         Iteration => Iteration);
   end Set;

   ------------------
   -- To_Bind_Name --
   ------------------

   function To_Bind_Name (Item : String) return Bind_Names.Bounded_String is
   begin
      return Bind_Names.To_Bounded_String
               (Ada.Characters.Handling.To_Upper (Item));
   end To_Bind_Name;

end OCI.Thick.Containers.Strings;
