-------------------------------------------------------------------------------
--
--  Filename        : $Source: /cvsroot/gnade/gnade/contrib/gsql/gui_table.adb,v $
--  Description     : Table Editor
--  Author          : Michael Erdmann
--  Created         : 8.8.2001
--  Last Modified By: $Author: me $
--  Last Modified On: $Date: 2001/12/21 10:11:41 $
--  Status          : $State: Exp $
--
--  Copyright (C) 2000 Michael Erdmann
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  Author: Michael Erdmann <michael.erdmann@snafu.de>                       --
--                                                                           --
--  GNADE is implemented to work with GNAT, the GNU Ada compiler.            --
--                                                                           --
-------------------------------------------------------------------------------
with Ada.Strings.Unbounded;    use Ada.Strings.Unbounded;

with Ada.Text_Io;              use Ada.Text_IO;
with Ada.Exceptions;           use Ada.Exceptions;
with Ada.Strings;              use Ada.Strings;
with Ada.Strings.Fixed;        use Ada.Strings.Fixed;
with Unchecked_Deallocation;

with Glib;                     use Glib;
-- Gdk
with Gdk;
with Gdk.Drawable;             use Gdk.Drawable;
with Gdk.Bitmap;               use Gdk.Bitmap;
with Gdk.Event;                use Gdk.Event;
with Gdk.Font;                 use Gdk.Font;
with Gdk.Color;                use Gdk.Color;
with Gdk.GC;                   use Gdk.GC;
with Gdk.Pixmap;               use Gdk.Pixmap;
with Gdk.Rectangle;            use Gdk.Rectangle;
with Gdk.Types;                use Gdk.Types;
with Gdk.Window;               use Gdk.Window;
-- Gtk
with Gtk.Rc;
with Gtk;                      use Gtk;
with Gtk.Arguments;            use Gtk.Arguments;
with Gtk.Box;                  use Gtk.Box;
with Gtk.Clist;                use Gtk.Clist;
with Gtk.Editable;             use Gtk.Editable;
with Gtk.Event_Box;            use Gtk.Event_Box;
with Gtk.Enums;                use Gtk.Enums;
with Gtk.Frame;                use Gtk.Frame;
with Gtk.Handlers;             use Gtk.Handlers;
with Gtkada.Handlers;          use Gtkada.Handlers;
with Gtk.Scrolled_Window;      use Gtk.Scrolled_Window;
with Gtk.Text;                 use Gtk.Text;
with Gtk.Window;               use Gtk.Window;
with Gtk.Widget;               use Gtk.Widget;
with Gtk.GEntry;               use Gtk.GEntry;

with GUI_Common;               use GUI_Common;
with GUI_Logging;              use GUI_Logging;
with GUI_Addrow;               use GUI_Addrow;
with GUI_Dialog;               use GUI_Dialog;

with Contents_Handler;         use Contents_Handler;
with Text_Block;               use Text_Block;
with Import_Export;            use Import_Export;

package body GUI_Table is

   Version : constant String := "$Id: gui_table.adb,v 1.17 2001/12/21 10:11:41 me Exp $";

   --- ************************************************************************** ---
   --  ***                       C L A S S    D A T A                         *** ---
   --- ************************************************************************** ---

   Dismiss_CMD     : constant Natural := 1;
   Add_Row_CMD     : constant Natural := 2;
   Remove_Row_CMD  : constant Natural := 3;
   Edit_Row_CMD    : constant Natural := 4;
   Import_CMD      : constant Natural := 5;
   Export_CMD      : constant Natural := 6;

   --- ************************************************************************** ---
   --- ***                 I N S T A N C E     D A T A                        *** ---
   --- ************************************************************************** ---

   BLANK          : constant Unbounded_String := To_Unbounded_String(" ");
   DML_DEFAULT    : constant Unbounded_String := To_Unbounded_String(" DEFAULT");
   DML_CHECK      : constant Unbounded_String := To_Unbounded_String(" CHECK");
   DML_PRIMARY_KEY: constant Unbounded_String := To_Unbounded_String(" PRIMARY KEY");
   DML_UNIQUE     : constant Unbounded_String := To_Unbounded_String(" UNIQUE");
   DML_NOT_NULL   : constant Unbounded_String := To_Unbounded_String(" NOT NULL");
   DML_NULL       : constant Unbounded_String := To_Unbounded_String(" NULL");

   ----------------------------
   -- DML_Type_Specification --
   ----------------------------
   function DML_Type_Specification(
      Def    : in Row_Information_Record ) return Unbounded_String is
      Result : Unbounded_String := Null_Unbounded_String;
   begin
      Result := Def.Sql_Type;
      if Def.Default /= Null_Unbounded_String then
         Result := Result & DML_DEFAULT & BLANK & Def.Default;
      end if;
      if Def.Is_Null then
         Result := Result & DML_NULL;
      elsif Def.Is_Not_Null then
         Result := Result & DML_NOT_NULL;
      end if;
      if Def.Unique then
         Result := Result & DML_UNIQUE;
      end if;
      if Def.Primary_Key then
         Result := Result & DML_PRIMARY_KEY;
      end if;

      return Result;
   end DML_Type_Specification ;

   --------------------
   -- Fetch_Row_Info --
   --------------------
   procedure Fetch_Row_Info(
      Text : in Text_Buffer;
      Last : in out Natural;
      Info : out Row_Information_Record ) is
      -- fetch next row information from the text buffer

      function Value(
         S :in Unbounded_String ) return Boolean is
      begin
         if S /= Null_Unbounded_String then
            return Boolean'Value(To_String(S));
         else
            return False;
         end if;
      end Value;

      Result   : Text_Buffer( 1..Max_Section_Length );
   begin
      Get_Text_Block( Result, Text, Last );

      Info.Name            := Result(1);
      Info.Sql_Type        := Result(2);
      Info.Default         := Result(3);
      Info.Constraint      := Result(4);
      Info.Constraint_Name := Result(5);
      Info.Comment         := Result(6);
      Info.Primary_Key     := Value( Result(7) );
      Info.Unique          := Value( Result(8) );
      Info.Is_Null         := Value( Result(9) );
      Info.Is_Not_Null     := Value( Result(10) );
   end Fetch_Row_Info;

   ----------------------
   -- DML_Create_Table --
   ----------------------
   function DML_Create_Table_Command(
      Text     : in Text_Buffer ) return String is
      -- This procedure builds an SQL DML command in oder to create
      -- the specified table.
      Last     : Positive := Text'First;
      Line     : Unbounded_String := Null_Unbounded_String;
      Result   : Unbounded_String := Null_Unbounded_String;

      Comment  : constant Unbounded_String := To_Unbounded_String("--");
      Info     : Row_Information_Record;
   begin
      Fetch_Next( Line, Text, Last );
      Result := To_Unbounded_String( "CREATE TABLE " & To_String(Line) & " (");

      loop
         Fetch_Next( Line, Text, Last );
         exit when Is_End_Of_Block(Line);

--         Result := Result & Comment & ( Line & Ascii.LF );
      end loop;

      Fetch_Next( Line, Text, Last );
      if Line /= Null_Unbounded_String then
         declare
            Nbr_Of_Rows : Positive := Positive'Value( To_String( Line ) );
         begin
            -- copy the domain information
            for I in 1..Nbr_Of_Rows loop
               Fetch_Row_Info( Text, Last, Info );
               Result := Result & Info.Name & " " ;
               Result := Result & DML_Type_Specification( Info );
               if I < Nbr_Of_Rows then
                  Result := Result & "," & ASCII.LF;
               end if;
            end loop;
         end ;
      end if;

      Result := Result & ")";

      return To_String(Result);
   end DML_Create_Table_Command;

   -------------------
   -- To_Row_Format --
   -------------------
   procedure Display_Row_Info(
      This    : in out Object ;
      Def     : in Row_Information_Record;
      Row     : in Natural := 0) is
      -- create a displayable for of the row information. The displayed
      -- information represents the information inserted later as
      -- sql (dml) statements.
      Display : Text_Buffer( 1..4 );
      Result  : Unbounded_String := Null_Unbounded_String;
   begin
      Display(1) := Def.Name ;
      Display(2) := DML_Type_Specification( Def );
      Display(3) := Def.Comment;
      Insert_Row( This.Definition, Display, Row );
   end Display_Row_Info;

   ----------------
   -- Get_Result --
   ----------------
   procedure Get_Result(
      This        : in  Object;
      Text        : out Text_Buffer;
      Last        : out Positive ) is
      -- copy the current editor contents into the buffer handled by the
      -- contents manager/handler.
      Name        : Gtk_Entry renames This.Name;
      Description : Gtk_Text  renames This.Description;
      Definition  : Gtk_Clist renames This.Definition;

      procedure Add(
         S : in Unbounded_String ) is
      begin
         Text(Last) := S;
         Last := Last + 1;
      end Add;

      procedure Add(
         S : in Boolean ) is
      begin
         Add( To_Unbounded_String(Boolean'Image(S)) );
      end Add;

   begin
      Last := Text'First;
      Add( To_Unbounded_String(Get_Text( Name )) );


      Put_Text_Block(
         Text, To_Text_Buffer(Get_Chars(Gtk_Editable(This.Description))), Last);

      Add( To_Unbounded_String(Positive'Image( This.Nbr_Of_Rows )) );

      for I in 1..This.Nbr_Of_Rows loop
         Add( This.Row_Info(I).Name );
         Add( This.Row_Info(I).Sql_Type );
         Add( This.Row_Info(I).Default );
         Add( This.Row_Info(I).Constraint );
         Add( This.Row_Info(I).Constraint_Name );
         Add( This.Row_Info(I).Comment );
         Add( This.Row_Info(I).Primary_Key );
         Add( This.Row_Info(I).Unique );
         Add( This.Row_Info(I).Is_Null );
         Add( This.Row_Info(I).Is_Not_Null );

         Insert_End_Of_Block( Text, Last );
      end loop;
   end Get_Result;

   ---------------
   -- Set_Input --
   ---------------
   procedure Set_Input(
      This        : in out Object;
      Text        : in Text_Buffer ) is
      --
      Name        : Gtk_Entry renames This.Name;
      Description : Gtk_Text  renames This.Description;
      Definition  : Gtk_Clist :=      This.Definition;
      Import_Name : Gtk_Entry renames This.Import_Name;
      Export_Name : Gtk_Entry renames This.Export_Name;

      Last        : Positive := Text'First;
      Line        : Unbounded_String := Null_Unbounded_String;

      function Value(
         S :in Unbounded_String ) return Boolean is
      begin
         if S /= Null_Unbounded_String then
            return Boolean'Value(To_String(S));
         else
            return False;
         end if;
      end Value;

      procedure Fetch_Row_Info(
         Info     : out Row_Information_Record ) is
         -- fetch next row information
         Result   : Text_Buffer( 1..Max_Section_Length );
      begin
         Get_Text_Block( Result, Text, Last );

         Info.Name            := Result(1);
         Info.Sql_Type        := Result(2);
         Info.Default         := Result(3);
         Info.Constraint      := Result(4);
         Info.Constraint_Name := Result(5);
         Info.Comment         := Result(6);
         Info.Primary_Key     := Value( Result(7) );
         Info.Unique          := Value( Result(8) );
         Info.Is_Null         := Value( Result(9) );
         Info.Is_Not_Null     := Value( Result(10) );

      end Fetch_Row_Info;

   begin
      Set_Text( Name, Key( Contents(This).all ) );
      Last := Last + 1;

      -- copy the description
      Freeze( Description);
      loop
         Fetch_Next( Line, Text, Last );
         exit when Is_End_Of_Block(Line);

         Insert( Description,
                 Null_Font,
                 Null_Color,
                 Null_Color,
                 To_String( Line ) & Ascii.LF,
                 Gint (To_String( Line )'Length) + 1 );
      end loop;
      Thaw( Description );

      Fetch_Next( Line, Text, Last );
      if Line /= Null_Unbounded_String then
         This.Nbr_Of_Rows := Positive'Value( To_String( Line ) );
         -- copy the domain information
         for I in 1..This.Nbr_Of_Rows loop
            Fetch_Row_Info( This.Row_Info(I) );
            Display_Row_Info( This, This.Row_Info(I) );
         end loop;
      end if;

      Set_Text( Import_Name, Key( Contents(This).all ) & ".import" );
      Set_Text( Export_Name, Key( Contents(This).all ) & ".export" );

   exception
      when End_Of_Block_Reached =>
         GUI_Logging.Log("end of block" );
   end Set_Input;


   ----------------
   -- Insert_Row --
   ----------------
   procedure Insert_Row(
      This        : in out Object;
      Def         : in Row_Information_Record ) is
      -- Insert a row into the table data.
      --
      -- R.1 If a row of the same name already exists the entry
      --     is deleted and newly inserted.
      -- R.2 if the number of rows allowes still a new entry, the
      --     it is inserted. Else an error is issued. This should
      --     never happen, but has been added for security.
      --
      Display     : Text_Buffer(1..4);
      use Gint_List;
   begin
      for I in 1..This.Nbr_Of_Rows loop
         if This.Row_Info(I).Name = Def.Name then           -- R.1
            This.Row_Info(I) := Def ;
            Remove( This.Definition, Gint( I - 1 ) );
            Display_Row_Info(This, Def, Row => I );
            return;
         end if;
      end loop;

      if This.Nbr_Of_Rows < This.Row_Info'Last then         -- R.2
         This.Nbr_Of_Rows := This.Nbr_Of_Rows + 1;
         This.Row_Info(This.Nbr_Of_Rows) := Def ;
         Display_Row_Info( This, Def);
      else
         GUI_Logging.Error("The number of rows per table has been exceeded");
      end if;

   end Insert_Row;

   -------------
   -- Display --
   -------------
   procedure Display(
      This        : in out Object;
      Box         : in out Gtk_Box ) is
      -- add the table information
      Name        : Gtk_Entry renames This.Name;
      Import_Name : Gtk_Entry renames This.Import_Name;
      Export_Name : Gtk_Entry renames This.Export_Name;

      Description : Gtk_Text  renames This.Description;
      Definition  : Gtk_Clist renames This.Definition;

      Vbox        : Gtk_Box;
      Text        : Text_Buffer( 1..Max_Section_Length );
      Last        : Positive := 1;

      V1          : Gtk_Box;
   begin
      Gtk_New_VBox( Vbox );
      Gtk_New_VBox( V1 );
      ----------------
      -- Table Name --
      ----------------
      Entry_Field( V1, Name,        "Table Name"  );
      Entry_Field( V1, Export_Name, "Import File" );
      Entry_Field( V1, Import_Name, "Export File" );

      Pack_Start( Vbox, V1, Expand => False );
      ----------------------
      -- Decription Field --
      ----------------------
      GUI_Common.Add_Editor( Vbox, Description, "Description" );

      ----------------------
      -- Table components --
      ----------------------
      GUI_Common.Add_Table( Vbox, Definition, 3 );
      Set_Column_Title( Definition, 0, "Domain        " );
      Set_Column_Title( Definition, 1, "Type                        " );
      Set_Column_Title( Definition, 2, "Comment                     " );

      ----------------------
      Load( Contents(This).all, Text, Last );
      Set_Input( This, Text(1..Last) );

      ----------------------
      Pack_Start( Box, Vbox );
      Show_All( Box );

      --------------
      -- COMMANDS --
      --------------
      Add_Command( This, "New Column",    Add_Row_CMD    );
      Add_Command( This, "Edit Column",   Edit_Row_CMD   );
      Add_Command( This, "Remove Column", Remove_Row_CMD );
      Add_Command( This, "Dismiss",       Dismiss_CMD    );

      Add_Command( This, "Import Data",   Import_CMD     );
      Add_Command( This, "Export Data",   Export_CMD     );
   end Display;

   -------------
   -- Command --
   -------------
   procedure Command(
      This        : in out Object;
      Cmd         : in Natural ) is
      -- execute a command
      Name        : Gtk_Entry renames This.Name;
      Import_Name : Gtk_Entry renames This.Import_Name;
      Export_Name : Gtk_Entry renames This.Export_Name;

      Description : Gtk_Text  renames This.Description;
      Definition  : Gtk_Clist renames This.Definition;
      Row         : Row_Information_Record;
      use Gint_List;
      I           : Gint;
      J           : Positive;
   begin
      case Cmd is
         when Dismiss_CMD =>
            -- GUI_AddRow.Finalize( This.Row_Dialog );
            Remove_Sheet( This );

         when Add_Row_CMD =>
            if This.Nbr_Of_Rows = This.Row_Info'Last then
               GUI_Logging.Error("The number of rows per table is exceeded");
            else
               GUI_AddRow.Initialize( This.Row_Dialog, Row, This );
               GUI_Dialog.Add_Dialog( This.Row_Dialog, "New column", 400, 500 );
            end if;

         when Remove_Row_Cmd =>
            Freeze (Definition);
            loop
                exit when Length (Get_Selection (Definition)) = 0;
                I := Get_Data (First (Get_Selection (Definition)));
                J := Positive( I + 1 );

                for K in J..This.Row_Info'Last-1 loop
                   This.Row_Info(K)        := This.Row_Info(K+1);
                   This.Row_Info(K+1).Name := Null_Unbounded_String;
                end loop;

                This.Nbr_Of_Rows := This.Nbr_Of_Rows - 1;

                Remove (Definition, I);
            end loop;
            Thaw (Definition);

         when Edit_Row_Cmd =>
            J := Positive( Get_Data (First (Get_Selection (Definition))) + 1 );
            Put_Line( Positive'Image(J) );
            GUI_AddRow.Initialize( This.Row_Dialog, This.Row_Info(J), This );
            GUI_Dialog.Add_Dialog(
               This.Row_Dialog,
               " Row : " & To_String(This.Row_Info(J).Name) ,
               400, 500 );

         when Import_CMD =>
            Import_Export.Export( Get_Text( Name ), Get_Text( Import_Name ));

         when Export_CMD =>
            Import_Export.Export( Get_Text( Name ), Get_Text( Export_Name ));

         when others =>
            GUI_Logging.Log("Unimplemented dialog command: " & Natural'Image( Cmd ) );
      end case;

   end Command;

end GUI_Table;

