#include <glib-object.h>

#include "common.h"

#include <gmyth/gmyth_uri.h>
#include <gmyth/gmyth_backendinfo.h>
#include <gmyth/gmyth_livetv.h>
#include <gmyth/gmyth_monitor_handler.h>
#include <gmyth/gmyth_common.h>

#define URI_DEFAULT	"myth://192.168.3.165:6543/livetv?channel=9"

/**
 * The GObject signal handler function, from which all status messages 
 * from the Monitor Handler will be advertized, all time it receives
 * LiveTV status messages from the MythTV backend
 * 
 * @param monitor a GMythMonitorHandler instance
 * @param msg_code the MythTV's server numeric status code
 * @param message the message's string description
 * @param user_data pointer to the GMythLiveTV instance (NULL in the tests)
 */
static void
gmyth_livetv_monitor_signal_handler(GMythMonitorHandler * monitor,
                                    gint msg_code, gchar * message,
                                    gpointer user_data)
{
    g_debug
        ("LIVETV Signal handler ( msg = %s, code = %d, live_tv param = %s, user_data = %s )\n",
         message, msg_code, "NULL", user_data != NULL ? "" : "NULL");

}

/**
 * Starts the Monitor Handler to this GMythLiveTV session, in order
 * to receive the status messages from the MythTV's backend server 
 * 
 * @param live_tv the GMythLiveTV instance
 * 
 * @return <code>true</code> if the Monitor Handler start-up process
 * 	   had been concluded succcesfully 
 */
static          gboolean
gmyth_test_monitor_handler_start(GMythBackendInfo * backend_info,
                                 GMythMonitorHandler * *monitor)
{

    gboolean        res = TRUE;

    if (*monitor != NULL) {
        g_object_unref(*monitor);
        *monitor = NULL;
    }

    *monitor = gmyth_monitor_handler_new();

    res =
        gmyth_monitor_handler_open(*monitor,
                                   backend_info->hostname,
                                   backend_info->port);

    if (res == TRUE) {
        g_debug
            ("Connect MythTV Monitor event socket! Trying to start the message handler...");

        res = gmyth_monitor_handler_start(*monitor);

        if (res) {
            g_debug
                ("MythTV Monitor event socket connected and listening!");
            g_signal_connect(G_OBJECT(*monitor), "backend-events-handler",
                             (GCallback)
                             gmyth_livetv_monitor_signal_handler, NULL);
        } else {
            g_debug
                ("Problems when trying to start MythTV Monitor event socket!");
            goto error;
        }
    }

  error:
    return res;

}

static          gboolean
test_monitor_handler_setup(gchar * uri)
{
    gboolean        ret = TRUE;
    GMythMonitorHandler *monitor;

    if (NULL == uri)
        uri = g_strdup(URI_DEFAULT);

    GMythURI       *gmyth_uri = gmyth_uri_new_with_value(uri);

    GMythBackendInfo *backend_info = gmyth_backend_info_new_with_uri(uri);
    g_debug("uri = %s", uri);

    ret = gmyth_test_monitor_handler_start(backend_info, &monitor);

    if (ret == FALSE) {
        g_debug("MonitorHandler couldn't start!\n");
        goto init_failed;
    }

  init_failed:
    if (monitor != NULL)
        g_object_unref(monitor);

    if (gmyth_uri != NULL)
        g_object_unref(gmyth_uri);

    if (backend_info != NULL)
        g_object_unref(backend_info);

    return ret;

}

gint
main(gint args, const gchar ** argv)
{
    gboolean        ret;

    g_type_init();

    g_thread_init(NULL);

    fprintf(stdout, SYNC_STRING);
    fflush(NULL);
    getchar();

    if (args > 1)
        ret = test_monitor_handler_setup(argv[1]);
    else
        ret = test_monitor_handler_setup(NULL);

    if (!ret)
        g_debug("Error when running Monitor Handler setup test script!");
    else
        g_debug("MonitorHandler setup test script setup with success.");

    return (0);
}
