/*
 *  Copyright 2004-2006 Michael Terry
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "GtkMultTable.h"
#include "MultPuzzle.h"
#include "CharBox.h"
#include "DigitChooser.h"
#include "config.h"
#include "gettext.h"
#include <ctype.h>

GtkMultTable::GtkMultTable (MultPuzzle *p)
	: Gtk::Table (1, 1, TRUE), puzzle (p)
{
	this->restart (p);
	
	this->set_col_spacings (3);
	this->set_row_spacings (3);
	
	this->set_border_width (12);
	
	puzzle->signal_changed ().connect (sigc::mem_fun (*this, &GtkMultTable::redraw));
}

GtkMultTable::~GtkMultTable (void)
{
}

void GtkMultTable::restart (MultPuzzle *p)
{
	char *y, *z;
	
	this->puzzle = p;
	
	y = this->puzzle->getMultiplier ();
	z = this->puzzle->getAnswer ();
	
	this->resize (5 + strlen (y), 1 + strlen (z));
	
	free (y);
	free (z);
	
	this->redraw ();
}


void GtkMultTable::redraw (void)
{
	Gtk::HSeparator *sep;
	CharBox *b;
	char *x, *y, *z;
	std::list <Widget *> children;
	std::list <Widget *>::iterator i;
	
	/* Clear out old children */
	children = this->get_children ();
	for (i = children.begin (); i != children.end (); i++)
	{
		this->remove (**i);
	}
	
	b = manage (new CharBox ("×", false));
	this->attach (*b, 0, 1, 1, 2);
	
	b = manage (new CharBox ("+", false));
	this->attach (*b, 0, 1, this->property_n_rows () - 3, 
		this->property_n_rows () - 2);
	
	sep = manage (new Gtk::HSeparator ());
	this->attach (*sep, 0, this->property_n_columns (),
	this->property_n_rows () - 2, this->property_n_rows () - 1);
	
	sep = manage (new Gtk::HSeparator ());
	this->attach (*sep, 0, this->property_n_columns (), 2, 3);
	
	x = this->puzzle->getMultiplicand ();
	y = this->puzzle->getMultiplier ();
	z = this->puzzle->getAnswer ();
	
	this->addString (x, 0, 0);
	this->addString (y, 1, 0);
	this->addString (z, this->property_n_rows () - 1, 0);
	
	for (int i = 0; i < this->puzzle->getNumAddends (); i++)
	{
		char *n = this->puzzle->getAddend (i);
		this->addString (n, 3 + i, i);
		free (n);
	}
	
	free (x);
	free (y);
	free (z);
	
	this->show_all ();
}

void GtkMultTable::addString (const char *str, int row, int offset)
{
	int size = strlen (str);
	int startCol = this->property_n_columns () - size - offset;
	
	for (int i = 0; i < size; i++)
	{
		CharBox *b;
		
		if (isdigit (str[i]))
			b = manage (new CharBox (str[i], false));
		else
			b = manage (new TableBox (str[i], this->puzzle));
		
		b->show_all ();
		b->set_size_request (36, 24);
		this->attach (*b, startCol + i, startCol + i + 1, row, row + 1,
			Gtk::FILL | Gtk::EXPAND, Gtk::FILL | Gtk::EXPAND, 0, 0);
	}
}


/* table box stuff */
GtkMultTable::TableBox::TableBox (char c, MultPuzzle *p)
	: CharBox (c, true), puzzle (p), character (c)
{
	this->add_events (Gdk::BUTTON_RELEASE_MASK);
}

bool GtkMultTable::TableBox::on_button_release_event (GdkEventButton *e)
{
	char chosen;
	DigitChooser dc (*((Gtk::Window *) this->get_toplevel ()), *puzzle, this->character);
	
	chosen = dc.run ();
	
	if (chosen)
		puzzle->guess (chosen, this->character);
	
	return TRUE;
}
