/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.enterprise.deployment.backend;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import javax.enterprise.deploy.shared.ModuleType;

import com.sun.enterprise.deployment.Application;
import com.sun.enterprise.deployment.archivist.ApplicationArchivist;
import com.sun.enterprise.deployment.archivist.Archivist;
import com.sun.enterprise.deployment.archivist.ArchivistFactory;
import com.sun.enterprise.deployment.deploy.shared.AbstractArchive;
import com.sun.enterprise.deployment.interfaces.ClientJarMaker;
import com.sun.enterprise.deployment.RootDeploymentDescriptor;
import com.sun.enterprise.deployment.util.ModuleDescriptor;
import com.sun.enterprise.util.zip.ZipItem;

/**
 * This class is responsible for creating an appclient jar file that
 * will be used by the appclient container to run the appclients for
 * the deployed application.
 *
 * @author Jerome Dochez
 */
class ApplicationClientJarMaker implements ClientJarMaker {

    protected Properties props;

    /**
     * Default constructor for this stateless object
     * @param props are the implementation properties (if any)
     */
    public ApplicationClientJarMaker(Properties props) {
        this.props = props;
    }
    
    /**
     * creates the appclient container jar file
     * @param descriptor is the loaded module's deployment descriptor
     * @param source is the abstract archive for the source module deployed
     * @param target is the abstract archive for the desired appclient container jar file
     * @param stubs are the stubs generated by the deployment codegen
     * @param props is a properties collection to pass implementation parameters
     *
     * @throws IOException when the jar file creation fail
     */
    public void create(RootDeploymentDescriptor descriptor, AbstractArchive source,
        AbstractArchive target,ZipItem[] stubs, Properties props)
        throws IOException {
        create(descriptor, source, null, target, stubs, props);
    }

    /**
     * creates the appclient container jar file
     * @param descriptor is the loaded module's deployment descriptor
     * @param source is the abstract archive for the source module deployed
     * @param source is the abstract archive for the generated xml directory
     * @param target is the abstract archive for the desired appclient container jar file
     * @param stubs are the stubs generated by the deployment codegen
     * @param props is a properties collection to pass implementation parameters
     *
     * @throws IOException when the jar file creation fail
     */
    public void create(RootDeploymentDescriptor descriptor, AbstractArchive source,
        AbstractArchive source2, AbstractArchive target,ZipItem[] stubs, 
        Properties props) throws IOException {

        // in all cases we copy the stubs file in the target archive
        ClientJarMakerUtils.populateStubs(target, stubs);

        //@@@ this block will not be called if we use ModuleClientMaker
        if (!descriptor.isApplication()) {
            //copy over all content of the appclient
            ClientJarMakerUtils.populateModuleJar(source, source2, target);
            return;
        }
        
        /* libraries holds URIs (relative to the application's root level) for
         * directories and JARs that need to be added to the 
         * client JAR because they are in the <library-directory> (spec requirement),
         * they are at the top-level of the EAR (legacy feature compatibility), 
         * or they appear in the manifest Class-Path entry of a JAR otherwise
         * to be included in the app client JAR
         */
        ArrayList<String> libraries = new ArrayList<String>();
        URI appURI = new File(source.getArchiveUri()).toURI();
        
        Application app = Application.class.cast(descriptor);
        for (Iterator modules = app.getModules(); modules.hasNext();) {
            ModuleDescriptor md = ModuleDescriptor.class.cast(modules.next());

            //ignore the war and rar modules, include both appclient and ejb
            if ( ! (md.getModuleType().equals(ModuleType.WAR) 
                || md.getModuleType().equals(ModuleType.RAR)) ){

                AbstractArchive subSource = source.getEmbeddedArchive(md.getArchiveUri());
                AbstractArchive subSource2 = null;
                if (source2 != null) {
                    subSource2 = source2.getEmbeddedArchive(md.getArchiveUri());
                }
                AbstractArchive subTarget = target.getEmbeddedArchive(md.getArchiveUri());
                
                /*
                 * populateModuleJar will add URIs for any JARs mentioned in
                 * the module JAR's manifest Class-Path, but to do so it needs
                 * to know the URI of the app and the URI of the parent of the module
                 * since Class-Path entries are relative to the parent of the 
                 * referring JAR.  So compute the parentURI.
                 */
                String archiveUri = md.getArchiveUri().replace('\\','/');
                URI parentURI = null;
                try {
                    parentURI = ClientJarMakerUtils.getParent(archiveUri);
                } catch (URISyntaxException ex) {
                    IOException ioe = new IOException();
                    ioe.initCause(ex);
                    throw ioe;
                }
                
                //copy over all content of the appclient - also updates 'libraries' as needed
                ClientJarMakerUtils.populateModuleJar(subSource, subSource2, subTarget, libraries, appURI, parentURI);

                target.closeEntry(subTarget);
                source.closeEntry(subSource);
                if (source2 != null) {
                    source2.closeEntry(subSource2);
                }
            }

            //copy over the alternative deployment descriptors
            //Do this even for web or resource adapter submodules because
            //the application.xml which will be copied into the client jar
            //will still refer to those alt. DD files.  So even though the
            //web or RA submodules will not be included, include the alt. DDs
            //so the generated app client jar archive will load correctly
            //on the client.
            if (md.getAlternateDescriptor() != null) {
                String ddPath = md.getAlternateDescriptor();
                String runtimeDDPath = "sun-" + ddPath;
                if (source2 != null) {
                    ClientJarMakerUtils.copy(source2, target, ddPath);
                    ClientJarMakerUtils.copy(source2, target, runtimeDDPath);
                } else {
                    ClientJarMakerUtils.copy(source, target, ddPath);
                    ClientJarMakerUtils.copy(source, target, runtimeDDPath);
                }
            }
        }

        // The libraries list already contains URIs for JARs or directories
        // that appeared in the module's manifest Class-Path entries.  Now
        // add to that list URIs for other JARs that need to be included.
        List<String> additionalLibraries = 
            ClientJarMakerUtils.getLibraryEntries(app, source);
        libraries.addAll(additionalLibraries);
        
        // Go through the JARs to be added to the classpath.  For each one check
        // to see if it has a Manifest Class-Path entry.  If so, add each
        // part of the Class-Path entry to the list of libraries.  Note that the
        // entries can be either JARs or directories.  Either will work.
        ClientJarMakerUtils.addClassPathElementsFromManifestClassPaths(new File(appURI), libraries);
        
        // Copy all the library JARs and directories into the app client archive
        ClientJarMakerUtils.copyLibraries(source, target, libraries);

        // deployment descriptors for the application
        ClientJarMakerUtils.copyDeploymentDescriptors(
            new ApplicationArchivist(), source, source2, target);
    }
}
