#ifndef ALIGNMENT_H
#define ALIGNMENT_H

/**
 * Logiciel Gassst (Global Alignment Short Sequence Search Tool)
 * \file Alignment.h
 * \brief Class Alignment, dfinissant un alignement entre deux squences
 * \author Damien Fleury
 * \version 5.2
 * \date 28/08/2008
 */


/**
 * \class Alignment, Un alignement entre deux squences
 * \brief Un alignement contient toutes les informations identifiant l'alignement et permettant de l'enregistrer
 * Cette classe est principalement utilise pour le cas des alignements avec gap
 */


class Alignment
{
private:
	/**
	 * La premire squence de l'alignement, c'est--dire celle de la base de donnes
	 * La chane est termine par un \0
	 */
	char* sequence1;
	
	/**
	 * La seconde squence de l'alignement, c'est--dire celle de la base de requtes
	 * La chane est termine par un \0
	 */
	char* sequence2;

	/**
	 * La longueur de l'alignement en nombre de caractres
	 */
	int length;
	
	/**
	 * Le nombre de msappariements dans l'alignement
	 */
	int nb_mismatches;
	
	/**
	 * Le nombre de gaps dans l'alignement
	 */
	int nb_gaps;
	
	/**
	 * L'indice de dpart de l'alignement dans la premire squence
	 */
	int start1;
	
	/**
	 * L'indice de dpart de l'alignement dans la seconde squence
	 */
	int start2;
	
	/**
	 * L'indice de fin de l'alignement dans la premire squence
	 */
	int end1;
	
	/**
	 * L'indice de fin de l'alignement dans la seconde squence
	 */
	int end2;
	

	//2variables utiles pour le calcul du cigar
	/**
	 * operation cigar en cours : 'M'  'I' ou 'D'
	 */
	char etat_cigar;

	/**
	 *  longueur de l'operation en cours du cigar
	 */
	unsigned int cpt_cigar; 

	/**
	 *  nombre de char deja imprimes du cigar
	 */
	unsigned int cigar_index;

public:
	
	/**
	 * indique si alignement sur strand -, dans ce cas il faudra reverse la sortie
	 */
	char rev_comp;

	//score evalue de l alignement
	double e_value ;

	//index des commentaires des sequences
	char * j1;

	char * j2;

	//le numero des sequences,relatif a la partition
	unsigned int  n1,n2;

	//description de l'align en cigar, (pour format sam)
	char * cigar;


	void finish_cigar();


	/**
	 * Le consructeur par dfaut d'Alignment
	 */
	Alignment();
	
	/**
	 * Un consructeur d'Alignment
	 * \param size la taille de l'alignement
	 */
	Alignment(int size);
	
	/**
	 * La fonction d'initialisation d'un Alignment
	 * \param size la taille de l'alignement
	 */
	void init();
	
	/**
	 * La fonction d'initialisation d'un Alignment
	 * \param rever si align sur strand -
	 */
	void init(char rever);
	/**
	 * Le consructeur par recopie d'Alignment
	 * \param al un objet Alignment
	 */
	Alignment(const Alignment& al);
	
	/**
	 * L'oprateur d'affectation d'Alignment
	 * \param al un objet Alignment
	 * \return l'objet Alignment affect
	 */
	Alignment& operator=(const Alignment& al);
	
	/**
	 * Le destructeur par dfaut d'Alignment
	 */
	virtual ~Alignment();
	
	/**
	 * Mthode d'obtention de la longueur de l'alignement
	 * \return la longueur de l'objet Alignment appelant
	 */
	inline int getLength();
	
	/**
	 * Mthode d'obtention du nombre de msappariements
	 * \return le nombre de msappariements de l'objet Alignment appelant
	 */
	inline int getMis();
	
	/**
	 * Mthode d'obtention du nombre de gaps
	 * \return le nombre de gaps de l'objet Alignment appelant
	 */
	inline int getGaps();
	
	/**
	 * Mthode d'incrmentation du nombre de msappariements
	 */
	inline void addMis();
	
	/**
	 * Mthode d'incrmentation du nombre de gaps
	 */
	inline void addGap();

	/**
	 * Mthode permettant d'initialiser les adresses de dbut et de fin de l'alignement dans les squences
	 * \param deb1 l'indice de dpart du fragment de la premire squence
	 * \param deb2 l'indice de dpart du fragment de la seconde squence
	 * \param len la longueur de l'alignement
	 * \param sizeSeq1 la longueur de la plus longue squence
	 * \rev_comp un boolen indiquant si la premire squence est inverse et complmente ou non
	 */
	void setOffsets(int deb1, int deb2, int len, int sizeSeq1, bool rev_comp);
	
	//complement the start position if rev _comp
	void adjust_rev_comp( int sizeSeq1, bool rev_comp);



	//complement the alignment for correct output
	void apply_rev_comp(int output_format);
	/**
	 * Fonction utilise pour incrmenter la valeur de l'indice de dpart dans la premire squence
	 * On peut ainsi tenir compte des gaps utiliss
	 */
	inline void incAlign();
	
	/**
	 * Fonction utilise pour dcrmenter la valeur de l'indice de dpart dans la premire squence
	 * On peut ainsi tenir compte des gaps utiliss
	 */
	inline void decAlign();
	
	inline void incEnd1();
	inline void decStart1();

	inline void decEnd1();
	inline void incStart1();
	/**
	 * Mthode d'accs  l'attribut start1
	 * \return la valeur de start1
	 */
	inline int getStart1();
	
	/**
	 * Mthode d'accs  l'attribut end1
	 * \return la valeur de end1
	 */
	inline int getEnd1();
	
	/**
	 * Mthode d'accs  l'attribut start2
	 * \return la valeur de start2
	 */
	inline int getStart2();
	
	/**
	 * Mthode d'accs  l'attribut end2
	 * \return la valeur de end2
	 */
	inline int getEnd2();
	
	/**
	 * Mthode d'ajout d'un couple d'un caractre aux squences
	 * \param c1 le caractre  ajouter  la premire squence
	 * \param c2 le caractre  ajouter  la seconde squence
	 */
	 void addPair(char c1, char c2);

	/**
	 * Mthode permettant de rcuprer la premire squence
	 * \return la premire squence de l'Alignment
	 */
	inline char* getSeq1();
	
	/**
	 * Mthode permettant de rcuprer la seconde squence
	 * \return la seconde squence de l'Alignment
	 */
	inline char* getSeq2();
	 
	/**
	 * Mthode permettant de rcuprer le string cigar
	 * \return le string cigar
	 */
	inline char* getCigar();
	/**
	 * Mthode permettant d'obtenir le caractre  l'indice en paramtre dans la premire squence
	 * \param indice l'indice du caractre dans le tableau squence1
	 * \return le caractre  l'indice voulu dans la premire squence
	 */
	char getChar1(int indice);
	
	/**
	 * Mthode permettant d'obtenir le caractre  l'indice en paramtre dans la seconde squence
	 * \param indice l'indice du caractre dans le tableau squence2
	 * \return le caractre  l'indice voulu dans la seconde squence
	 */
	char getChar2(int indice);	
	
	/**
	 * Mthode d'affichage d'un Aligment
	 */
	void affichage();
};


/**
 * Mthode d'obtention de la longueur de l'alignement
 * \return la longueur de l'objet Alignment apelant
 */
int Alignment::getLength()
{
	return length;
}

/**
 * Mthode d'obtention du nombre de msappariements
 * \return le nombre de msappariements de l'objet Alignment appelant
 */
int Alignment::getMis()
{
	return nb_mismatches;
}

/**
 * Mthode d'obtention du nombre de gaps
 * \return le nombre de gaps de l'objet Alignment appelant
 */
int Alignment::getGaps()
{
	return nb_gaps;
}

/**
 * Mthode d'incrmentation du nombre de msappariements
 */
void Alignment::addMis()
{
	++nb_mismatches;
}

/**
 * Mthode d'incrmentation du nombre de gaps
 */
void Alignment::addGap()
{
	++nb_gaps;
}

/**
 * Fonction utilise pour incrmenter la valeur de l'indice de dpart dans la premire squence
 * On peut ainsi tenir compte des gaps utiliss
*/
void Alignment::incAlign()
{
	++start1;
}

   void Alignment::incEnd1()
{
	++end1;

}
   void Alignment::decStart1()
{
	--start1;

}

 void Alignment::decEnd1()
{
	--end1;

}
   void Alignment::incStart1()
{
	++start1;

}
/**
 * Fonction utilise pour dcrmenter la valeur de l'indice de dpart dans la premire squence
 * On peut ainsi tenir compte des gaps utiliss
 */
void Alignment::decAlign()
{
	--start1;
}

/**
 * Mthode d'accs  l'attribut start1
 * \return la valeur de start1
 */
inline int Alignment::getStart1()
{
	return start1;
}

/**
 * Mthode d'accs  l'attribut end1
 * \return la valeur de end1
 */
inline int Alignment::getEnd1()
{
	return end1;
}

/**
 * Mthode d'accs  l'attribut start2
 * \return la valeur de start2
 */
inline int Alignment::getStart2()
{
	return start2;
}
	
/**
 * Mthode d'accs  l'attribut end2
* \return la valeur de end2
 */
inline int Alignment::getEnd2()
{
	return end2;
}

/**
 * Mthode permettant de rcuprer la premire squence
 * \return la premire squence de l'Alignment
 */
inline char* Alignment::getSeq1()
{
	return sequence1;
}

/**
 * Mthode permettant de rcuprer la seconde squence
 * \return la seconde squence de l'Alignment
 */
inline char* Alignment::getSeq2()
{
	return sequence2;
}


/**
 * Mthode permettant de rcuprer le string cigar
 * \return le string cigar
 */
inline char* Alignment::getCigar()
{
  return cigar;
}


#endif
