/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */

/* gul-bonobo-extensions.c - implementation of new functions that conceptually
                             belong in bonobo. Perhaps some of these will be
                             actually rolled into bonobo someday.

            This file is based on nautilus-bonobo-extensions.c from
            libnautilus-private.

   Copyright (C) 2000, 2001 Eazel, Inc.

   The Gnome Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The Gnome Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the Gnome Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Authors: John Sullivan <sullivan@eazel.com>
            Darin Adler <darin@bentspoon.com>
*/

#include <config.h>

#include "gul-bonobo-extensions.h"
#include <string.h>

#include <bonobo/bonobo-ui-util.h>
#include <gtk/gtkmain.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <bonobo/bonobo-control.h>

typedef enum {
	NUMBERED_MENU_ITEM_PLAIN,
	NUMBERED_MENU_ITEM_TOGGLE,
	NUMBERED_MENU_ITEM_RADIO
} NumberedMenuItemType;

void
gul_bonobo_set_accelerator (BonoboUIComponent *ui,
			    const char *path,
			    const char *accelerator)
{
	if (bonobo_ui_component_get_container (ui)) /* should not do this here... */
	{
		bonobo_ui_component_set_prop (ui, path, "accel", accelerator, NULL);
	}
}

void
gul_bonobo_set_label (BonoboUIComponent *ui,
		      const char *path,
		      const char *label)
{
	if (bonobo_ui_component_get_container (ui)) /* should not do this here... */
	{
		bonobo_ui_component_set_prop (ui, path, "label", label, NULL);
	}
}

void
gul_bonobo_set_tip (BonoboUIComponent *ui,
		    const char *path,
		    const char *tip)
{
	if (bonobo_ui_component_get_container (ui)) /* should not do this here... */
	{
		bonobo_ui_component_set_prop (ui, path, "tip", tip, NULL);
	}
}

void
gul_bonobo_set_hidden (BonoboUIComponent *ui,
		       const char *path,
		       gboolean hidden)
{
	if (bonobo_ui_component_get_container (ui)) /* should not do this here... */
	{
		bonobo_ui_component_set_prop (ui, path, "hidden", hidden ? "1" : "0", NULL);
	}
}

static char *
get_numbered_menu_item_name (guint index)
{
	return g_strdup_printf ("%u", index);
}			      

char *
gul_bonobo_get_numbered_menu_item_path (BonoboUIComponent *ui,
					const char *container_path, 
					guint index)
{
	char *item_name;
	char *item_path;
	
	g_return_val_if_fail (BONOBO_IS_UI_COMPONENT (ui), NULL); 
	g_return_val_if_fail (container_path != NULL, NULL);
	
	item_name = get_numbered_menu_item_name (index);
	item_path = g_strconcat (container_path, "/", item_name, NULL);
	g_free (item_name);
	
	return item_path;
}

char *
gul_bonobo_get_numbered_menu_item_command (BonoboUIComponent *ui,
					   const char *container_path, 
					   guint index)
{
	char *command_name;
	char *path;
	
	g_return_val_if_fail (BONOBO_IS_UI_COMPONENT (ui), NULL); 
	g_return_val_if_fail (container_path != NULL, NULL);
	
	path = gul_bonobo_get_numbered_menu_item_path (ui, container_path, index);
	command_name = gnome_vfs_escape_string (path);
	g_free (path);
	
	return command_name;
}

static char *
gul_bonobo_add_numbered_menu_item_internal (BonoboUIComponent *ui,
					    const char *container_path,
					    guint index,
					    const char *label,
					    NumberedMenuItemType type,
					    GdkPixbuf *pixbuf,
					    const char *radio_group_name)
{
	char *xml_item, *xml_command; 
	char *command_name;
	char *item_name, *pixbuf_data;
	char *path;
	
	g_assert (BONOBO_IS_UI_COMPONENT (ui)); 
	g_assert (container_path != NULL);
	g_assert (label != NULL);
	g_assert (type == NUMBERED_MENU_ITEM_PLAIN || pixbuf == NULL);
	g_assert (type == NUMBERED_MENU_ITEM_RADIO || radio_group_name == NULL);
	g_assert (type != NUMBERED_MENU_ITEM_RADIO || radio_group_name != NULL);
	
	item_name = get_numbered_menu_item_name (index);
	command_name = gul_bonobo_get_numbered_menu_item_command 
		(ui, container_path, index);
	
	switch (type) {
	case NUMBERED_MENU_ITEM_TOGGLE:
		xml_item = g_strdup_printf ("<menuitem name=\"%s\" id=\"%s\" type=\"toggle\"/>\n", 
					    item_name, command_name);
		break;
	case NUMBERED_MENU_ITEM_RADIO:
		xml_item = g_strdup_printf ("<menuitem name=\"%s\" id=\"%s\" "
					    "type=\"radio\" group=\"%s\"/>\n", 
					    item_name, command_name, radio_group_name);
		break;
	case NUMBERED_MENU_ITEM_PLAIN:
		if (pixbuf != NULL) {
			pixbuf_data = bonobo_ui_util_pixbuf_to_xml (pixbuf);			
			xml_item = g_strdup_printf ("<menuitem name=\"%s\" verb=\"%s\" "
						    "pixtype=\"pixbuf\" pixname=\"%s\"/>\n", 
						    item_name, command_name, pixbuf_data);
			g_free (pixbuf_data);
		} else {
			xml_item = g_strdup_printf ("<menuitem name=\"%s\" verb=\"%s\"/>\n", 
						    item_name, command_name);
		}
		break;
	default:
		g_assert_not_reached ();
		xml_item = NULL;	/* keep compiler happy */
	}
	
	g_free (item_name);
	
	bonobo_ui_component_set (ui, container_path, xml_item, NULL);
	
	g_free (xml_item);
	
	path = gul_bonobo_get_numbered_menu_item_path (ui, container_path, index);
	gul_bonobo_set_label (ui, path, label);
	g_free (path);
	
	/* Make the command node here too, so callers can immediately set
	 * properties on it (otherwise it doesn't get created until some
	 * time later).
	 */
	xml_command = g_strdup_printf ("<cmd name=\"%s\"/>\n", command_name);
	bonobo_ui_component_set (ui, "/commands", xml_command, NULL);
	g_free (xml_command);
	
	g_free (command_name);

	return item_name;
}

/* Add a menu item specified by number into a given path. Used for
 * dynamically creating a related series of menu items. Each index
 * must be unique (normal use is to call this in a loop, and
 * increment the index for each item).
 */
void
gul_bonobo_add_numbered_menu_item (BonoboUIComponent *ui, 
				   const char *container_path, 
				   guint index,
				   const char *label, 
				   GdkPixbuf *pixbuf)
{
	g_return_if_fail (BONOBO_IS_UI_COMPONENT (ui)); 
	g_return_if_fail (container_path != NULL);
	g_return_if_fail (label != NULL);
	
	gul_bonobo_add_numbered_menu_item_internal (ui, container_path, index, label,
						    NUMBERED_MENU_ITEM_PLAIN, pixbuf, NULL);
}

void 
gul_bonobo_replace_path (BonoboUIComponent *uic, const gchar *path_src, 
			 const char *path_dst)
{
	BonoboUINode *node;
	const char *name;
	char *path_dst_folder;

	name = strrchr (path_dst, '/');
	g_return_if_fail (name != NULL);
       	path_dst_folder = g_strndup (path_dst, name - path_dst);
	name++;

	node = bonobo_ui_component_get_tree (uic, path_src, TRUE, NULL);
	bonobo_ui_node_set_attr (node, "name", name);

	gul_bonobo_clear_path (uic, path_dst);

	bonobo_ui_component_set_tree (uic, path_dst_folder, node, NULL);

	g_free (path_dst_folder);
	bonobo_ui_node_free (node);
}

void
gul_bonobo_clear_path (BonoboUIComponent *uic,
		       const gchar *path)
{
	if (bonobo_ui_component_path_exists  (uic, path, NULL))
	{
		char *remove_wildcard = g_strdup_printf ("%s/*", path);
		bonobo_ui_component_rm (uic, remove_wildcard, NULL);
		g_free (remove_wildcard);
	}
}
