/* ethos-ui-manager-widget.c
 *
 * Copyright (C) 2009 Christian Hergert <chris@dronelabs.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston MA 
 * 02110-1301 USA
 */

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib/gi18n.h>

#include "ethos-manager.h"
#include "ethos-ui-configurable.h"
#include "ethos-ui-manager-widget.h"

/**
 * SECTION:ethos-ui-manager-widget
 * @title: EthosUIManagerWidget
 * @short_description: gtk widget for managing ethos during runtime
 *
 *  The #EthosUIManagerWidget can be used to manage an #EthosManager during runtime.  It provides
 *  features to activate and deactivate plugins as well as display information about them.  If the
 *  plugin implements #EthosUIConfigurable, then the plugin can also be configured from the dialog.
 */

struct _EthosUIManagerWidgetPrivate
{
	EthosManager *manager;

	GtkListStore *list_store;
	GtkWidget    *tree_view;
	GtkWidget    *about;
	GtkWidget    *configure;

	guint         initialized_handler;
	guint         loaded_handler;
	guint         unloaded_handler;
};

G_DEFINE_TYPE (EthosUIManagerWidget, ethos_ui_manager_widget, GTK_TYPE_VBOX)

static void
ethos_ui_manager_widget_finalize (GObject *object)
{
	G_OBJECT_CLASS (ethos_ui_manager_widget_parent_class)->finalize (object);
}

static void
ethos_ui_manager_widget_class_init (EthosUIManagerWidgetClass *klass)
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS (klass);
	object_class->finalize = ethos_ui_manager_widget_finalize;
	g_type_class_add_private (object_class, sizeof(EthosUIManagerWidgetPrivate));
}

static void
active_cell_data_func (GtkTreeViewColumn *column,
                       GtkCellRenderer   *cell,
                       GtkTreeModel      *tree_model,
                       GtkTreeIter       *iter,
                       gpointer           data)
{
	EthosPluginInfo *plugin_info;

	gtk_tree_model_get (tree_model, iter, 0, &plugin_info, -1);

	g_object_set (cell,
	              "active", ethos_plugin_info_get_active (plugin_info),
	              "sensitive", !ethos_plugin_info_has_errors (plugin_info),
	              "activatable", !ethos_plugin_info_has_errors (plugin_info),
	              NULL);
}

static void
pixbuf_cell_data_func (GtkTreeViewColumn *column,
                       GtkCellRenderer   *cell,
                       GtkTreeModel      *tree_model,
                       GtkTreeIter       *iter,
                       gpointer           data)
{
	EthosPluginInfo *plugin_info;
	const gchar     *icon;

	gtk_tree_model_get (tree_model, iter, 0, &plugin_info, -1);
	icon = ethos_plugin_info_get_icon_name (plugin_info);

	if (!icon || g_strcmp0 (icon, "") == 0)
		icon = "ethos-plugin";

	g_object_set (cell,
	              "icon-name", icon,
	              "sensitive", !ethos_plugin_info_has_errors (plugin_info),
	              NULL);
}

static void
e_pixbuf_cell_data_func (GtkTreeViewColumn *column,
                         GtkCellRenderer   *cell,
                         GtkTreeModel      *tree_model,
                         GtkTreeIter       *iter,
                         gpointer           data)
{
	EthosPluginInfo *plugin_info;
	const gchar     *icon = NULL;

	gtk_tree_model_get (tree_model, iter, 0, &plugin_info, -1);

	if (ethos_plugin_info_has_errors (plugin_info))
		icon = GTK_STOCK_DIALOG_WARNING;

	g_object_set (cell,
	              "icon-name", icon,
	              NULL);
}

static void
selection_changed_cb (GtkTreeSelection *tree_selection,
                      gpointer          data)
{
	EthosUIManagerWidgetPrivate *priv;
	EthosPluginInfo             *plugin_info;
	EthosPlugin                 *plugin;
	GtkTreeModel                *model;
	GtkTreeIter                  iter;
	gboolean                     sensitive;

	g_return_if_fail (ETHOS_UI_IS_MANAGER_WIDGET (data));

	priv = ETHOS_UI_MANAGER_WIDGET (data)->priv;

	sensitive = gtk_tree_selection_get_selected (tree_selection, &model, &iter);
	gtk_widget_set_sensitive (priv->about, sensitive);
	gtk_widget_set_sensitive (priv->configure, FALSE);

	if (sensitive) {
		gtk_tree_model_get (model, &iter, 0, &plugin_info, -1);
		plugin = ethos_manager_get_plugin (priv->manager, plugin_info);
		if (plugin && ETHOS_UI_IS_CONFIGURABLE (plugin))
			gtk_widget_set_sensitive (priv->configure, TRUE);
	}
}

static void
text_cell_data_func (GtkTreeViewColumn *column,
                     GtkCellRenderer   *cell,
                     GtkTreeModel      *tree_model,
                     GtkTreeIter       *iter,
                     gpointer           data)
{
	EthosPluginInfo *plugin_info;
	gchar           *name_esc,
			*desc_esc,
			*str;

	gtk_tree_model_get (tree_model, iter, 0, &plugin_info, -1);
	name_esc = g_markup_escape_text (ethos_plugin_info_get_name (plugin_info), -1);
	desc_esc = g_markup_escape_text (ethos_plugin_info_get_description (plugin_info), -1);
	str = g_strconcat ("<b>", name_esc, "</b>\n", desc_esc, NULL);

	g_object_set (cell,
	              "markup", str,
	              "sensitive", !ethos_plugin_info_has_errors (plugin_info),
	              NULL);

	g_free (str);
	g_free (name_esc);
	g_free (desc_esc);
}

static void
about_dialog_response_cb (GtkDialog *dialog,
                          gint       response_id,
                          gpointer   user_data)
{
	if (response_id == GTK_RESPONSE_CANCEL) {
		gtk_widget_hide (GTK_WIDGET (dialog));
		gtk_widget_destroy (GTK_WIDGET (dialog));
	}
}

static void
about_clicked_cb (GtkWidget *about,
                  gpointer   data)
{
	EthosUIManagerWidgetPrivate *priv;
	EthosPluginInfo             *plugin_info;
	GtkTreeSelection            *selection;
	GtkTreeModel                *model;
	GtkTreeIter                  iter;
	GtkWidget                   *about_dialog;

	g_return_if_fail (ETHOS_UI_IS_MANAGER_WIDGET (data));

	priv = ETHOS_UI_MANAGER_WIDGET (data)->priv;
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->tree_view));

	if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
		gtk_tree_model_get (model, &iter, 0, &plugin_info, -1);
		about_dialog = gtk_about_dialog_new ();
		const gchar **authors = NULL;

		g_object_get (plugin_info, "authors", &authors, NULL);

		gtk_about_dialog_set_authors (GTK_ABOUT_DIALOG (about_dialog), authors);
		gtk_about_dialog_set_copyright (GTK_ABOUT_DIALOG (about_dialog),
		                                ethos_plugin_info_get_copyright (plugin_info));
		gtk_about_dialog_set_logo_icon_name (GTK_ABOUT_DIALOG (about_dialog),
		                                     ethos_plugin_info_get_icon_name (plugin_info));
		gtk_about_dialog_set_version (GTK_ABOUT_DIALOG (about_dialog),
		                              ethos_plugin_info_get_version (plugin_info));
		gtk_about_dialog_set_program_name (GTK_ABOUT_DIALOG (about_dialog),
		                                   ethos_plugin_info_get_name (plugin_info));
		gtk_about_dialog_set_comments (GTK_ABOUT_DIALOG (about_dialog),
		                               ethos_plugin_info_get_description (plugin_info));
		gtk_about_dialog_set_website (GTK_ABOUT_DIALOG (about_dialog),
		                              ethos_plugin_info_get_website (plugin_info));

		g_signal_connect (G_OBJECT (about_dialog),
		                  "response",
		                  G_CALLBACK (about_dialog_response_cb),
		                  NULL);
		gtk_window_set_transient_for (
				GTK_WINDOW (about_dialog),
				GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (data))));

		gtk_widget_show (about_dialog);
	}
}

static void
url_hook_cb (GtkAboutDialog *about_dialog,
             const gchar    *link_,
             gpointer        data)
{
/*
	// Simply having this method here is making gtk automatically
	// create the link setup for us.
	gtk_show_uri (gtk_widget_get_screen (GTK_WIDGET (about_dialog)),
	              link_,
	              GDK_CURRENT_TIME,
	              NULL);
*/
}

static void
toggled_cb (GtkCellRendererToggle *cell,
            gchar                 *path,
            gpointer               user_data)
{
	EthosUIManagerWidgetPrivate *priv;
	EthosPluginInfo             *plugin_info;
	GtkTreeIter                  iter;
	GtkWidget                   *dialog,
				    *hbox,
				    *vbox,
				    *image,
				    *label,
				    *label2,
				    *content_area;
	GError                      *error = NULL;
	gboolean                     is_unload = FALSE;
	gchar                       *markup;

	g_return_if_fail (ETHOS_UI_IS_MANAGER_WIDGET (user_data));

	priv = ETHOS_UI_MANAGER_WIDGET (user_data)->priv;

	if (gtk_tree_model_get_iter_from_string (GTK_TREE_MODEL (priv->list_store), &iter, path)) {
		gtk_tree_model_get (GTK_TREE_MODEL (priv->list_store),
		                    &iter, 0, &plugin_info, -1);
		if (ethos_plugin_info_get_active (plugin_info)) {
			is_unload = TRUE;
			if (!ethos_manager_unload_plugin (priv->manager,
			                                  plugin_info,
			                                  &error))
			        goto show_error_dialog;
		}
		else {
			if (!ethos_manager_load_plugin (priv->manager,
			                                plugin_info,
			                                &error))
			        goto show_error_dialog;
		}
	}

	return;

show_error_dialog:
	dialog = gtk_dialog_new_with_buttons (
			"",
			GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (user_data))),
			GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_STOCK_CLOSE,
			GTK_STOCK_CANCEL,
			NULL);
	hbox = gtk_hbox_new (FALSE, 6);
	vbox = gtk_vbox_new (FALSE, 6);
	image = gtk_image_new_from_stock (GTK_STOCK_DIALOG_ERROR,
	                                  GTK_ICON_SIZE_DIALOG);
	label = gtk_label_new (NULL);
	label2 = gtk_label_new (NULL);

	gtk_window_set_title (GTK_WINDOW (dialog), "");
	gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);
	gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
	gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
	content_area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
	gtk_container_set_border_width (GTK_CONTAINER (content_area), 12);

	gtk_container_set_border_width (GTK_CONTAINER (hbox), 12);
	gtk_box_set_spacing (GTK_BOX (hbox), 12);
	gtk_box_pack_start (GTK_BOX (content_area), hbox, TRUE, TRUE, 0);
	gtk_widget_show (hbox);

	g_object_set (image,
	              "yalign", 0.0,
	              NULL);
	gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, TRUE, 0);
	gtk_widget_show (image);

	gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
	gtk_widget_show (vbox);

	markup = g_strdup_printf ("<span size=\"larger\" weight=\"bold\">"
	                          "There was an error %s the %s plugin."
	                          "</span>\n",
	                          is_unload ? "unloading" : "loading",
	                          ethos_plugin_info_get_name (plugin_info));
	gtk_label_set_markup (GTK_LABEL (label), markup);
	g_free (markup);

	g_object_set (label,
	              "use-markup", TRUE,
	              "wrap", TRUE,
	              "yalign", 0.0,
	              "xalign", 0.0,
	              NULL);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, TRUE, 0);
	gtk_widget_show (label);

	if (error) {
		markup = g_markup_printf_escaped ("%s", error->message);
		gtk_label_set_markup (GTK_LABEL (label2), markup);
		g_free (markup);
	}

	g_object_set (label2,
	              "use-markup", TRUE,
	              "wrap", TRUE,
	              "yalign", 0.0,
	              "xalign", 0.0,
	              NULL);
	gtk_box_pack_start (GTK_BOX (vbox), label2, FALSE, TRUE, 0);
	gtk_widget_show (label2);

	gtk_widget_show (dialog);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);

	if (error)
		g_error_free (error);
}

static void
configure_clicked_cb (GtkWidget *configure,
                      gpointer   data)
{
	EthosUIManagerWidgetPrivate *priv;
	EthosPluginInfo             *plugin_info;
	EthosPlugin                 *plugin;
	GtkTreeSelection            *selection;
	GtkTreeModel                *model;
	GtkTreeIter                  iter;

	g_return_if_fail (ETHOS_UI_IS_MANAGER_WIDGET (data));

	priv = ETHOS_UI_MANAGER_WIDGET (data)->priv;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->tree_view));
	g_assert (selection);

	if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
		gtk_tree_model_get (model, &iter, 0, &plugin_info, -1);
		plugin = ethos_manager_get_plugin (priv->manager, plugin_info);
		if (plugin && ETHOS_UI_IS_CONFIGURABLE (plugin))
			ethos_ui_configurable_configure (ETHOS_UI_CONFIGURABLE (plugin),
			                                 gtk_widget_get_toplevel (configure));
	}
}

static void
ethos_ui_manager_widget_init (EthosUIManagerWidget *manager_widget)
{
	GtkWidget         *vbox,
	                  *scroller,
	                  *tree_view,
	                  *hbbox,
	                  *configure,
	                  *configure_hbox,
	                  *configure_image,
	                  *configure_label,
	                  *about,
		          *about_hbox,
		          *about_image,
		          *about_label;
	GtkListStore      *list_store;
	gchar             *icon_dir;
	GtkTreeViewColumn *active_column,
			  *info_column,
			  *error_column;
	GtkCellRenderer   *toggle_cell,
			  *text_cell,
			  *pixbuf_cell,
			  *e_pixbuf_cell;

	manager_widget->priv = G_TYPE_INSTANCE_GET_PRIVATE (manager_widget,
	                                                    ETHOS_UI_TYPE_MANAGER_WIDGET,
	                                                    EthosUIManagerWidgetPrivate);

	icon_dir = g_build_filename (DATADIR, "ethos", "icons", NULL);
	gtk_icon_theme_append_search_path (gtk_icon_theme_get_default (), icon_dir);
	g_free (icon_dir);

	gtk_box_set_spacing (GTK_BOX (manager_widget), 6);

	vbox = gtk_vbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (manager_widget), vbox, TRUE, TRUE, 0);
	gtk_widget_show (vbox);

	scroller = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scroller),
	                                     GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroller),
	                                GTK_POLICY_NEVER,
	                                GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start (GTK_BOX (vbox), scroller, TRUE, TRUE, 0);
	gtk_widget_show (scroller);

	tree_view = gtk_tree_view_new ();
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (tree_view), FALSE);
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_view), TRUE);
	gtk_container_add (GTK_CONTAINER (scroller), tree_view);
	gtk_widget_show (tree_view);

	hbbox = gtk_hbutton_box_new ();
	gtk_box_set_spacing (GTK_BOX (hbbox), 8);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (hbbox), GTK_BUTTONBOX_END);
	gtk_box_pack_start (GTK_BOX (vbox), hbbox, FALSE, TRUE, 0);
	gtk_widget_show (hbbox);

	about_hbox = gtk_hbox_new (FALSE, 2);
	about_image = gtk_image_new_from_stock (GTK_STOCK_ABOUT, GTK_ICON_SIZE_BUTTON);
	about_label = gtk_label_new (NULL);
	gtk_label_set_markup_with_mnemonic (GTK_LABEL (about_label), g_dgettext (GETTEXT_PACKAGE, "_About Plugin"));
	gtk_box_pack_start (GTK_BOX (about_hbox), about_image, FALSE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (about_hbox), about_label, TRUE, TRUE, 0);
	gtk_widget_show_all (about_hbox);

	configure_hbox = gtk_hbox_new (FALSE, 2);
	configure_image = gtk_image_new_from_stock (GTK_STOCK_PREFERENCES, GTK_ICON_SIZE_BUTTON);
	configure_label = gtk_label_new (NULL);
	gtk_label_set_markup_with_mnemonic (GTK_LABEL (configure_label), g_dgettext (GETTEXT_PACKAGE, "_Configure Plugin"));
	gtk_box_pack_start (GTK_BOX (configure_hbox), configure_image, FALSE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (configure_hbox), configure_label, TRUE, TRUE, 0);
	gtk_widget_show_all (configure_hbox);

	about = gtk_button_new ();
	gtk_widget_set_sensitive (about, FALSE);
	gtk_container_add (GTK_CONTAINER (about), about_hbox);
	gtk_box_pack_start (GTK_BOX (hbbox), about, FALSE, TRUE, 0);
	gtk_widget_show (about);

	configure = gtk_button_new ();
	gtk_widget_set_sensitive (configure, FALSE);
	gtk_container_add (GTK_CONTAINER (configure), configure_hbox);
	gtk_box_pack_start (GTK_BOX (hbbox), configure, FALSE, TRUE, 0);
	gtk_widget_show (configure);

	list_store = gtk_list_store_new (1, ETHOS_TYPE_PLUGIN_INFO);
	gtk_tree_view_set_model (GTK_TREE_VIEW (tree_view),
	                         GTK_TREE_MODEL (list_store));

	active_column = gtk_tree_view_column_new ();
	toggle_cell = gtk_cell_renderer_toggle_new ();
	g_object_set (toggle_cell, "xpad", 6, NULL);
	gtk_tree_view_column_pack_start (active_column, toggle_cell, TRUE);
	gtk_tree_view_column_set_cell_data_func (
			active_column, toggle_cell, active_cell_data_func,
			g_object_ref (manager_widget), g_object_unref);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), active_column);

	info_column = gtk_tree_view_column_new ();
	pixbuf_cell = gtk_cell_renderer_pixbuf_new ();
	g_object_set (pixbuf_cell, "stock-size", GTK_ICON_SIZE_SMALL_TOOLBAR, NULL);
	g_object_set (pixbuf_cell, "xpad", 6, NULL);
	text_cell = gtk_cell_renderer_text_new ();
	g_object_set (text_cell, "ellipsize", PANGO_ELLIPSIZE_END, NULL);
	gtk_tree_view_column_pack_start (info_column, pixbuf_cell, FALSE);
	gtk_tree_view_column_pack_start (info_column, text_cell, TRUE);
	gtk_tree_view_column_set_cell_data_func (
			info_column, pixbuf_cell, pixbuf_cell_data_func,
			g_object_ref (manager_widget), g_object_unref);
	gtk_tree_view_column_set_cell_data_func (
			info_column, text_cell, text_cell_data_func,
			g_object_ref (manager_widget), g_object_unref);
	gtk_tree_view_column_set_expand (info_column, TRUE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), info_column);

	error_column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_resizable (error_column, FALSE);
	e_pixbuf_cell = gtk_cell_renderer_pixbuf_new ();
	g_object_set (e_pixbuf_cell,
	              "stock-size", GTK_ICON_SIZE_SMALL_TOOLBAR,
	              "xpad", 6,
	              NULL);
	gtk_tree_view_column_pack_start (error_column, e_pixbuf_cell, FALSE);
	gtk_tree_view_column_set_cell_data_func (
			error_column, e_pixbuf_cell, e_pixbuf_cell_data_func,
			g_object_ref (manager_widget), g_object_unref);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), error_column);

	g_signal_connect (gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view)),
	                  "changed",
	                  G_CALLBACK (selection_changed_cb),
	                  manager_widget);

	g_signal_connect (about,
	                  "clicked",
	                  G_CALLBACK (about_clicked_cb),
	                  manager_widget);

	g_signal_connect (configure,
	                  "clicked",
	                  G_CALLBACK (configure_clicked_cb),
	                  manager_widget);

	g_signal_connect (toggle_cell,
	                  "toggled",
	                  G_CALLBACK (toggled_cb),
	                  manager_widget);

	manager_widget->priv->tree_view = tree_view;
	manager_widget->priv->configure = configure;
	manager_widget->priv->about = about;
	manager_widget->priv->list_store = list_store;

	if (gtk_about_dialog_set_url_hook (url_hook_cb, NULL, NULL) != NULL) {
		g_warning ("Ethos overwrote a previous GtkAboutDialog url hook"
		           ".  This may cause a bug in your application.");
	}
}

/**
 * ethos_ui_manager_widget_new:
 *
 * Creates a new instance of #EthosUIManagerWidget.  This widget can be added to a Gtk
 * application to manage ethos plugins at runtime.
 *
 * Return value: the newly created #EthosUIManagerWidget instance
 */
GtkWidget*
ethos_ui_manager_widget_new ()
{
	return g_object_new (ETHOS_UI_TYPE_MANAGER_WIDGET, NULL);
}

static void
build_list_store (EthosUIManagerWidget *widget)
{
	EthosUIManagerWidgetPrivate *priv;
	GList                       *list,
	                            *info_iter;
	GtkTreeIter                  iter;

	g_return_if_fail (ETHOS_UI_IS_MANAGER_WIDGET (widget));

	priv = widget->priv;
	list = ethos_manager_get_plugin_info (priv->manager);

	for (info_iter = list; info_iter; info_iter = info_iter->next) {
		gtk_list_store_append (priv->list_store, &iter);
		gtk_list_store_set (priv->list_store, &iter, 0, info_iter->data, -1);
	}

	g_list_free (list);
}

static void
plugin_loaded_cb (EthosManager         *manager,
                  EthosPluginInfo      *plugin_info,
                  EthosUIManagerWidget *manager_widget)
{
	// just queue draw so we get the collateral damage from dependencies
	gtk_widget_queue_draw (GTK_WIDGET (manager_widget));
}

static void
plugin_unloaded_cb (EthosManager         *manager,
                    EthosPluginInfo      *plugin_info,
                    EthosUIManagerWidget *manager_widget)
{
	// might as well here too.
	gtk_widget_queue_draw (GTK_WIDGET (manager_widget));
}

static void
initialized_cb (EthosManager         *manager,
                EthosUIManagerWidget *widget)
{
	build_list_store (widget);
}

/**
 * ethos_ui_manager_widget_set_manager:
 * @widget: an #EthosUIManagerWidget
 * @manager: an #EthosManager
 *
 * Sets the #EthosManager instance that the widget should observe for state changes.  The
 * manager is also used to activate/deactivate plugins during runtime.
 */
void
ethos_ui_manager_widget_set_manager (EthosUIManagerWidget *widget,
                                     EthosManager         *manager)
{
	EthosUIManagerWidgetPrivate *priv;

	g_return_if_fail (ETHOS_UI_IS_MANAGER_WIDGET (widget));
	g_return_if_fail (!manager || ETHOS_IS_MANAGER (manager));

	priv = widget->priv;

	gtk_list_store_clear (priv->list_store);

	if (priv->manager) {
		g_signal_handler_disconnect (priv->manager, priv->initialized_handler);
		g_signal_handler_disconnect (priv->manager, priv->loaded_handler);
		g_signal_handler_disconnect (priv->manager, priv->unloaded_handler);
		g_object_unref (priv->manager);
	}

	priv->manager = NULL;

	if (!manager)
		return;

	priv->manager = g_object_ref (manager);
	priv->initialized_handler =
		g_signal_connect (priv->manager,
		                  "initialized",
		                  G_CALLBACK (initialized_cb),
		                  widget);
	priv->loaded_handler =
		g_signal_connect (priv->manager,
		                  "plugin-loaded",
		                  G_CALLBACK (plugin_loaded_cb),
		                  widget);
	priv->unloaded_handler =
		g_signal_connect (priv->manager,
		                  "plugin-unloaded",
		                  G_CALLBACK (plugin_unloaded_cb),
		                  widget);

	build_list_store (widget);
}
