/*
***************************************************************************
*
* Author: Teunis van Beelen
*
* Copyright (C) 2010 Teunis van Beelen
*
* teuniz@gmail.com
*
***************************************************************************
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation version 2 of the License.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*
***************************************************************************
*
* This version of GPL is at http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt
*
***************************************************************************
*/



#include "import_annotations.h"


#if defined(__APPLE__) || defined(__MACH__) || defined(__APPLE_CC__)

#define fseeko fseek
#define ftello ftell
#define fopeno fopen

#else

#define fseeko fseeko64
#define ftello ftello64
#define fopeno fopen64

#endif



UI_ImportAnnotationswindow::UI_ImportAnnotationswindow(QWidget *parent)
{
  mainwindow = (UI_Mainwindow *)parent;

  if(mainwindow->files_open < 1)
  {
    UI_Messagewindow popuperror("Failure", "Can not import annotations without opening an EDF/BDF-file first.");
    return;
  }

  ImportAnnotsDialog = new QDialog;

  ImportAnnotsDialog->setMinimumSize(490, 485);
  ImportAnnotsDialog->setMaximumSize(490, 485);
  ImportAnnotsDialog->setWindowTitle("Import annotations/events");
  ImportAnnotsDialog->setModal(TRUE);
  ImportAnnotsDialog->setAttribute(Qt::WA_DeleteOnClose, TRUE);

  SeparatorLabel = new QLabel(ImportAnnotsDialog);
  SeparatorLabel->setText("Column separator");

  OnsetColumnLabel = new QLabel(ImportAnnotsDialog);
  OnsetColumnLabel->setText("Onset column");

  DescriptionColumnLabel = new QLabel(ImportAnnotsDialog);
  DescriptionColumnLabel->setText("Description column");

  DatastartLabel = new QLabel(ImportAnnotsDialog);
  DatastartLabel->setText("Data starts at line");

  OnsetTimeLabel = new QLabel(ImportAnnotsDialog);
  OnsetTimeLabel->setText("Onset time coding is");

  SeparatorLineEdit = new QLineEdit(ImportAnnotsDialog);
  SeparatorLineEdit->setMaxLength(3);
  SeparatorLineEdit->setText("tab");

  OnsetColumnSpinBox = new QSpinBox(ImportAnnotsDialog);
  OnsetColumnSpinBox->setRange(1,256);
  OnsetColumnSpinBox->setValue(1);

  DescriptionColumnSpinBox = new QSpinBox(ImportAnnotsDialog);
  DescriptionColumnSpinBox->setRange(1,256);
  DescriptionColumnSpinBox->setValue(2);

  DatastartSpinbox = new QSpinBox(ImportAnnotsDialog);
  DatastartSpinbox->setRange(1,100);
  DatastartSpinbox->setValue(1);

  RelativeTimeComboBox = new QComboBox(ImportAnnotsDialog);
  RelativeTimeComboBox->addItem("in seconds, relative to start of file");
  RelativeTimeComboBox->addItem("hh:mm:ss");
  RelativeTimeComboBox->addItem("hh:mm:ss.xxx");
  RelativeTimeComboBox->addItem("yyyy-mm-ddThh:mm:ss");
  RelativeTimeComboBox->addItem("yyyy-mm-ddThh:mm:ss.xxx");

  CSVRadioButton = new QRadioButton("ASCII/CSV");
  CSVRadioButton->setChecked(TRUE);

  XMLRadioButton = new QRadioButton("XML");

  formatGroupBox = new QGroupBox("input format", ImportAnnotsDialog);
  formatGroupBox->setGeometry(20, 20, 450, 90);

  formatVBoxLayout = new QVBoxLayout;
  formatVBoxLayout->addWidget(CSVRadioButton);
  formatVBoxLayout->addWidget(XMLRadioButton);
  formatGroupBox->setLayout(formatVBoxLayout);

  asciiSettingsGroupBox = new QGroupBox("ASCII settings", ImportAnnotsDialog);
  asciiSettingsGroupBox->setGeometry(20, 130, 450, 230);

  asciiSettingsHBoxLayout1 = new QHBoxLayout;
  asciiSettingsHBoxLayout1->addWidget(SeparatorLabel, 2);
  asciiSettingsHBoxLayout1->addWidget(SeparatorLineEdit, 1);
  asciiSettingsHBoxLayout1->addStretch(2);

  asciiSettingsHBoxLayout2 = new QHBoxLayout;
  asciiSettingsHBoxLayout2->addWidget(OnsetColumnLabel, 2);
  asciiSettingsHBoxLayout2->addWidget(OnsetColumnSpinBox, 1);
  asciiSettingsHBoxLayout2->addStretch(2);

  asciiSettingsHBoxLayout3 = new QHBoxLayout;
  asciiSettingsHBoxLayout3->addWidget(DescriptionColumnLabel, 2);
  asciiSettingsHBoxLayout3->addWidget(DescriptionColumnSpinBox, 1);
  asciiSettingsHBoxLayout3->addStretch(2);

  asciiSettingsHBoxLayout4 = new QHBoxLayout;
  asciiSettingsHBoxLayout4->addWidget(DatastartLabel, 2);
  asciiSettingsHBoxLayout4->addWidget(DatastartSpinbox, 1);
  asciiSettingsHBoxLayout4->addStretch(2);

  asciiSettingsHBoxLayout5 = new QHBoxLayout;
  asciiSettingsHBoxLayout5->addWidget(OnsetTimeLabel, 2);
  asciiSettingsHBoxLayout5->addWidget(RelativeTimeComboBox, 3);

  asciiSettingsVBoxLayout = new QVBoxLayout;
  asciiSettingsVBoxLayout->addLayout(asciiSettingsHBoxLayout1);
  asciiSettingsVBoxLayout->addLayout(asciiSettingsHBoxLayout2);
  asciiSettingsVBoxLayout->addLayout(asciiSettingsHBoxLayout3);
  asciiSettingsVBoxLayout->addLayout(asciiSettingsHBoxLayout4);
  asciiSettingsVBoxLayout->addLayout(asciiSettingsHBoxLayout5);
  asciiSettingsGroupBox->setLayout(asciiSettingsVBoxLayout);

  IgnoreConsecutiveCheckBox = new QCheckBox(" Ignore consecutive events with the\n same description", ImportAnnotsDialog);
  IgnoreConsecutiveCheckBox->setGeometry(25, 380, 300, 40);
  IgnoreConsecutiveCheckBox->setTristate(FALSE);
  IgnoreConsecutiveCheckBox->setCheckState(Qt::Unchecked);

  ImportButton = new QPushButton(ImportAnnotsDialog);
  ImportButton->setGeometry(20, 440, 80, 25);
  ImportButton->setText("Import");

  CloseButton = new QPushButton(ImportAnnotsDialog);
  CloseButton->setGeometry(390, 440, 80, 25);
  CloseButton->setText("Close");

  SeparatorLineEdit->setText(mainwindow->import_annotations_var->separator);
  OnsetColumnSpinBox->setValue(mainwindow->import_annotations_var->onsetcolumn);
  DescriptionColumnSpinBox->setValue(mainwindow->import_annotations_var->descriptioncolumn);
  DatastartSpinbox->setValue(mainwindow->import_annotations_var->datastartline);
  RelativeTimeComboBox->setCurrentIndex(mainwindow->import_annotations_var->onsettimeformat);
  if(mainwindow->import_annotations_var->format == 1)
  {
    CSVRadioButton->setChecked(TRUE);
    asciiSettingsGroupBox->setEnabled(TRUE);
  }
  else
  {
    XMLRadioButton->setChecked(TRUE);
    asciiSettingsGroupBox->setEnabled(FALSE);
  }

  QObject::connect(CloseButton,    SIGNAL(clicked()),     ImportAnnotsDialog, SLOT(close()));
  QObject::connect(ImportButton,   SIGNAL(clicked()),     this,               SLOT(ImportButtonClicked()));
  QObject::connect(CSVRadioButton, SIGNAL(toggled(bool)), this,               SLOT(outputformatRadioButtonClicked(bool)));

  ImportAnnotsDialog->exec();
}



void UI_ImportAnnotationswindow::outputformatRadioButtonClicked(bool checked)
{
  if(checked == TRUE)
  {
    asciiSettingsGroupBox->setEnabled(TRUE);
  }
  else
  {
    asciiSettingsGroupBox->setEnabled(FALSE);
  }
}



void UI_ImportAnnotationswindow::ImportButtonClicked()
{
  int i, j,
      temp,
      column,
      column_end,
      str_start,
      line_nr,
      startline=1,
      descr_column=2,
      onset_column=1,
      onset_is_set,
      descr_is_set,
      max_descr_length,
      onsettime_coding=0,
      digits,
      days=0,
      ignore_consecutive=0,
      csv_format=0;

  char path[1024],
       line[2048],
       scratchpad[256],
       str[256],
       separator=',',
       description[256],
       last_description[256],
       *result;

  long long onset=0LL,
            l_temp,
            last_onset=0LL,
            utc_time=0LL;

  FILE *inputfile=NULL;

  struct annotationblock *annotation;

  struct date_time_struct date_time;

  struct xml_handle *xml_hdl;


  max_descr_length = 20;

  description[0] = 0;

  last_description[0] = 0;

  ImportAnnotsDialog->setEnabled(FALSE);

  if(CSVRadioButton->isChecked() == TRUE)
  {
    csv_format = 1;

    strcpy(str, SeparatorLineEdit->text().toLatin1().data());

    if(!strcmp(str, "tab"))
    {
      separator = '\t';
    }
    else
    {
      if(strlen(str)!=1)
      {
        UI_Messagewindow popuperror("Invalid input", "Separator must be one character or \"tab\".");
        ImportAnnotsDialog->setEnabled(TRUE);
        return;
      }

      if((str[0]<32)||(str[0]>126))
      {
        UI_Messagewindow popuperror("Invalid input", "Separator character is not a valid ASCII character.");
        ImportAnnotsDialog->setEnabled(TRUE);
        return;
      }

      if(str[0]=='.')
      {
        UI_Messagewindow popuperror("Invalid input", "Separator character can not be a dot.");
        ImportAnnotsDialog->setEnabled(TRUE);
        return;
      }

      if((str[0]>47)&&(str[0]<58))
      {
        UI_Messagewindow popuperror("Invalid input", "Separator character can not be a number.");
        ImportAnnotsDialog->setEnabled(TRUE);
        return;
      }

      separator = str[0];
    }

    strcpy(mainwindow->import_annotations_var->separator, str);

    startline = DatastartSpinbox->value();

    descr_column = DescriptionColumnSpinBox->value() - 1;

    onset_column = OnsetColumnSpinBox->value() - 1;

    onsettime_coding = RelativeTimeComboBox->currentIndex();

    if(descr_column == onset_column)
    {
      UI_Messagewindow popuperror("Invalid input", "Onset and Description can not be in the same column.");
      ImportAnnotsDialog->setEnabled(TRUE);
      return;
    }

    mainwindow->import_annotations_var->onsettimeformat = onsettime_coding;
    mainwindow->import_annotations_var->onsetcolumn = onset_column + 1;
    mainwindow->import_annotations_var->descriptioncolumn = descr_column + 1;
    mainwindow->import_annotations_var->datastartline = startline;
  }
  else
  {
    csv_format = 0;
  }

  if(IgnoreConsecutiveCheckBox->checkState() == Qt::Checked)
  {
    ignore_consecutive = 1;
  }
  else
  {
    ignore_consecutive = 0;
  }

  mainwindow->import_annotations_var->format = csv_format;

  if(csv_format)
  {
    strcpy(path, QFileDialog::getOpenFileName(0, "Open ASCII file", mainwindow->recent_opendir, "ASCII files (*.txt *.TXT *.csv *.CSV)").toLatin1().data());
  }
  else
  {
    strcpy(path, QFileDialog::getOpenFileName(0, "Open XML file", mainwindow->recent_opendir, "XML files (*.xml *.XML)").toLatin1().data());
  }

  if(!strcmp(path, ""))
  {
    ImportAnnotsDialog->setEnabled(TRUE);
    return;
  }

  get_directory_from_path(mainwindow->recent_opendir, path, 1024);

//////////////////////////////////////////// XML format //////////////////////////////////////////

  if(!csv_format)
  {
    xml_hdl = xml_get_handle(path);
    if(xml_hdl==NULL)
    {
      UI_Messagewindow popuperror("Error", "Can not open file for reading.");
      ImportAnnotsDialog->setEnabled(TRUE);
      return;
    }

    if((xml_hdl->encoding != 1) && (xml_hdl->encoding != 2))
    {
      UI_Messagewindow popuperror("Error", "Encoding of XML-file must be UTF-8 or ISO-8859-1.");
      xml_close(xml_hdl);
      ImportAnnotsDialog->setEnabled(TRUE);
      return;
    }

    if(strcmp(xml_hdl->elementname, "annotationlist"))
    {
      UI_Messagewindow popuperror("Error", "Can not find root element \"annotationlist\".");
      xml_close(xml_hdl);
      ImportAnnotsDialog->setEnabled(TRUE);
      return;
    }

    QApplication::setOverrideCursor(Qt::WaitCursor);

    for(j=0; j<10; j++)  qApp->processEvents();

    for(i=0; i<10000; i++)
    {
      if(xml_goto_nth_element_inside(xml_hdl, "annotation", i))
      {
        if(i == 0)
        {
          QApplication::restoreOverrideCursor();
          UI_Messagewindow popuperror("Error", "Can not find child element \"annotation\".");
          xml_close(xml_hdl);
          ImportAnnotsDialog->setEnabled(TRUE);
          return;
        }

        break;
      }

      if(xml_goto_nth_element_inside(xml_hdl, "onset", 0))
      {
        xml_go_up(xml_hdl);
        continue;
      }

      result = xml_get_content_of_element(xml_hdl);

      if(strlen(result) > 17)
      {
        if((result[4] == '-') && (result[7] == '-') && (result[10] == 'T') && (result[13] == ':') && (result[16] == ':'))
        {
          date_time.year = atoi(result);
          date_time.month = atoi(result + 5);
          date_time.day = atoi(result + 8);
          date_time.hour = atoi(result + 11);
          date_time.minute = atoi(result + 14);
          date_time.second = atoi(result + 17);
          date_time_to_utc(&utc_time, date_time);
          onset = utc_time - mainwindow->edfheaderlist[0]->utc_starttime;
          onset *= TIME_DIMENSION;

          if(strlen(result) > 19)
          {
            if(result[19] == '.')
            {
              for(digits=0; digits<32; digits++)
              {
                if((result[20 + digits] < '0') || (result[20 + digits] > '9'))
                {
                  break;
                }
              }
              result[20 + digits] = 0;
              if(digits)
              {
                l_temp = (atoi(result + 20) * TIME_DIMENSION);
                for(; digits>0; digits--)
                {
                  l_temp /= 10LL;
                }
                onset += l_temp;
              }
            }
            else
            {
              xml_go_up(xml_hdl);
              continue;
            }
          }
        }
      }
      else
      {
        xml_go_up(xml_hdl);
        continue;
      }

      xml_go_up(xml_hdl);

      if(xml_goto_nth_element_inside(xml_hdl, "description", 0))
      {
        xml_go_up(xml_hdl);
        continue;
      }

      result = xml_get_content_of_element(xml_hdl);

      if(strcmp(result, last_description) || (!ignore_consecutive))
      {
        annotation = (struct annotationblock *)calloc(1, sizeof(struct annotationblock));
        if(annotation == NULL)
        {
          QApplication::restoreOverrideCursor();
          UI_Messagewindow popuperror("Error", "Malloc error (annotation).");
          free(result);
          xml_close(xml_hdl);
          ImportAnnotsDialog->setEnabled(TRUE);
          return;
        }
        annotation->onset = onset;
        strncpy(annotation->annotation, result, MAX_ANNOTATION_LEN);
        if(xml_hdl->encoding == 1)
        {
          latin1_to_utf8(annotation->annotation, MAX_ANNOTATION_LEN);
        }
        annotation->annotation[MAX_ANNOTATION_LEN] = 0;
        edfplus_annotation_add_item(&mainwindow->annotationlist[0], annotation);

        strcpy(last_description, result);
      }

      free(result);

      xml_go_up(xml_hdl);

      xml_go_up(xml_hdl);
    }

    xml_close(xml_hdl);
  }

//////////////////////////////////////////// CSV format //////////////////////////////////////////

  if(csv_format)
  {
    inputfile = fopeno(path, "rb");
    if(inputfile==NULL)
    {
      UI_Messagewindow popuperror("Error", "Can not open file for reading.");
      ImportAnnotsDialog->setEnabled(TRUE);
      return;
    }

    rewind(inputfile);

    QApplication::setOverrideCursor(Qt::WaitCursor);

    for(j=0; j<10; j++)  qApp->processEvents();

    for(i=0; i<(startline-1);)
    {
      temp = fgetc(inputfile);

      if(temp==EOF)
      {
        QApplication::restoreOverrideCursor();
        UI_Messagewindow popuperror("Error", "File does not contain enough lines.");
        fclose(inputfile);
        ImportAnnotsDialog->setEnabled(TRUE);
        return;
      }

      if(temp=='\n')
      {
        i++;
      }
    }

    i = 0;

    column = 0;

    column_end = 1;

    str_start = 0;

    line_nr = startline;

    onset_is_set = 0;

    descr_is_set = 0;

    while(1)
    {
      temp = fgetc(inputfile);

      if(temp==EOF)
      {
        break;
      }

      line[i] = temp;

      if(line[i]=='\r')
      {
        continue;
      }

      if(separator!=',')
      {
        if(line[i]==',')
        {
          line[i] = '.';
        }
      }

      if(line[i]==separator)
      {
        line[i] = 0;

        if(!column_end)
        {
          if(column == onset_column)
          {
            onset = 0LL;
            strncpy(scratchpad, line + str_start, 30);
            scratchpad[30] = 0;

            if(onsettime_coding == 0)
            {
              onset = atoll_x(scratchpad, TIME_DIMENSION);
            }

            if(onsettime_coding == 1)
            {
              if(strlen(scratchpad) > 6)
              {
                if((scratchpad[2] == ':') && (scratchpad[5] == ':'))
                {
                  scratchpad[8] = 0;
                  onset = atoi(scratchpad) * 3600LL;
                  onset += (atoi(scratchpad + 3) * 60LL);
                  onset += (long long)(atoi(scratchpad + 6));
                  onset *= TIME_DIMENSION;
                  onset -= mainwindow->edfheaderlist[0]->l_starttime;
                  if(onset < last_onset)
                  {
                    last_onset = onset;
                    days++;
                  }

                  onset_is_set = 1;
                }
              }
              if(strlen(scratchpad) > 5)
              {
                if((scratchpad[1] == ':') && (scratchpad[4] == ':'))
                {
                  scratchpad[7] = 0;
                  onset = atoi(scratchpad) * 3600LL;
                  onset += (atoi(scratchpad + 2) * 60LL);
                  onset += (long long)(atoi(scratchpad + 5));
                  onset *= TIME_DIMENSION;
                  onset -= mainwindow->edfheaderlist[0]->l_starttime;
                  if(onset < last_onset)
                  {
                    last_onset = onset;
                    days++;
                  }

                  onset_is_set = 1;
                }
              }
            }

            if(onsettime_coding == 2)
            {
              if(strlen(scratchpad) > 8)
              {
                if((scratchpad[2] == ':') && (scratchpad[5] == ':') && ((scratchpad[8] == '.') || (scratchpad[8] == ',')))
                {
                  for(digits=0; digits<32; digits++)
                  {
                    if((scratchpad[9 + digits] < '0') || (scratchpad[9 + digits] > '9'))
                    {
                      break;
                    }
                  }
                  scratchpad[9 + digits] = 0;
                  onset = atoi(scratchpad) * 3600LL;
                  onset += (atoi(scratchpad + 3) * 60LL);
                  onset += (long long)(atoi(scratchpad + 6));
                  onset *= TIME_DIMENSION;
                  if(digits)
                  {
                    l_temp = (atoi(scratchpad + 9) * TIME_DIMENSION);
                    for(; digits>0; digits--)
                    {
                      l_temp /= 10LL;
                    }
                    onset += l_temp;
                  }
                  onset -= mainwindow->edfheaderlist[0]->l_starttime;
                  if(onset < last_onset)
                  {
                    last_onset = onset;
                    days++;
                  }

                  onset_is_set = 1;
                }
              }
              if(strlen(scratchpad) > 7)
              {
                if((scratchpad[1] == ':') && (scratchpad[4] == ':') && ((scratchpad[7] == '.') || (scratchpad[7] == ',')))
                {
                  for(digits=0; digits<32; digits++)
                  {
                    if((scratchpad[8 + digits] < '0') || (scratchpad[8 + digits] > '9'))
                    {
                      break;
                    }
                  }
                  scratchpad[8 + digits] = 0;
                  onset = atoi(scratchpad) * 3600LL;
                  onset += (atoi(scratchpad + 2) * 60LL);
                  onset += (long long)(atoi(scratchpad + 5));
                  onset *= TIME_DIMENSION;
                  if(digits)
                  {
                    l_temp = (atoi(scratchpad + 8) * TIME_DIMENSION);
                    for(; digits>0; digits--)
                    {
                      l_temp /= 10LL;
                    }
                    onset += l_temp;
                  }
                  onset -= mainwindow->edfheaderlist[0]->l_starttime;
                  if(onset < last_onset)
                  {
                    last_onset = onset;
                    days++;
                  }

                  onset_is_set = 1;
                }
              }
            }

            if(onsettime_coding == 3)
            {
              if(strlen(scratchpad) > 17)
              {
                if((scratchpad[4] == '-') && (scratchpad[7] == '-') && (scratchpad[13] == ':') && (scratchpad[16] == ':'))
                {
                  scratchpad[19] = 0;
                  date_time.year = atoi(scratchpad);
                  date_time.month = atoi(scratchpad + 5);
                  date_time.day = atoi(scratchpad + 8);
                  date_time.hour = atoi(scratchpad + 11);
                  date_time.minute = atoi(scratchpad + 14);
                  date_time.second = atoi(scratchpad + 17);
                  date_time_to_utc(&utc_time, date_time);
                  onset = utc_time - mainwindow->edfheaderlist[0]->utc_starttime;
                  onset *= TIME_DIMENSION;

                  onset_is_set = 1;
                }
              }
            }

            if(onsettime_coding == 4)
            {
              if(strlen(scratchpad) > 19)
              {
                if((scratchpad[4] == '-') && (scratchpad[7] == '-') && (scratchpad[13] == ':') && (scratchpad[16] == ':') && ((scratchpad[19] == ',') || (scratchpad[19] == '.')))
                {
                  for(digits=0; digits<32; digits++)
                  {
                    if((scratchpad[20 + digits] < '0') || (scratchpad[20 + digits] > '9'))
                    {
                      break;
                    }
                  }
                  scratchpad[20 + digits] = 0;
                  date_time.year = atoi(scratchpad);
                  date_time.month = atoi(scratchpad + 5);
                  date_time.day = atoi(scratchpad + 8);
                  date_time.hour = atoi(scratchpad + 11);
                  date_time.minute = atoi(scratchpad + 14);
                  date_time.second = atoi(scratchpad + 17);
                  date_time_to_utc(&utc_time, date_time);
                  onset = utc_time - mainwindow->edfheaderlist[0]->utc_starttime;
                  onset *= TIME_DIMENSION;
                  if(digits)
                  {
                    l_temp = (atoi(scratchpad + 20) * TIME_DIMENSION);
                    for(; digits>0; digits--)
                    {
                      l_temp /= 10LL;
                    }
                    onset += l_temp;
                  }

                  onset_is_set = 1;
                }
              }
            }
          }

          if(column == descr_column)
          {
            strncpy(description, line + str_start, max_descr_length);
            description[max_descr_length] = 0;
            latin1_to_utf8(description, max_descr_length);
            descr_is_set = 1;
          }

          column_end = 1;

          column++;
        }
      }
      else
      {
        if(line[i]!='\n')
        {
          if(column_end)
          {
            str_start = i;

            column_end = 0;
          }
        }
      }

      if(line[i]=='\n')
      {
        line[i] = 0;

        if(!column_end)
        {
          if(column == onset_column)
          {
            if(descr_is_set)
            {
              onset = 0LL;
              strncpy(scratchpad, line + str_start, 30);
              scratchpad[30] = 0;

              if(onsettime_coding == 0)
              {
                onset = atoll_x(scratchpad, TIME_DIMENSION);
              }

              if(onsettime_coding == 1)
              {
                if(strlen(scratchpad) > 6)
                {
                  if((scratchpad[2] == ':') && (scratchpad[5] == ':'))
                  {
                    scratchpad[8] = 0;
                    onset = atoi(scratchpad) * 3600LL;
                    onset += (atoi(scratchpad + 3) * 60LL);
                    onset += (long long)(atoi(scratchpad + 6));
                    onset *= TIME_DIMENSION;
                    onset -= mainwindow->edfheaderlist[0]->l_starttime;
                    if(onset < last_onset)
                    {
                      last_onset = onset;
                      days++;
                    }

                    onset_is_set = 1;
                  }
                }
                if(strlen(scratchpad) > 5)
                {
                  if((scratchpad[1] == ':') && (scratchpad[4] == ':'))
                  {
                    scratchpad[7] = 0;
                    onset = atoi(scratchpad) * 3600LL;
                    onset += (atoi(scratchpad + 2) * 60LL);
                    onset += (long long)(atoi(scratchpad + 5));
                    onset *= TIME_DIMENSION;
                    onset -= mainwindow->edfheaderlist[0]->l_starttime;
                    if(onset < last_onset)
                    {
                      last_onset = onset;
                      days++;
                    }

                    onset_is_set = 1;
                  }
                }
              }

              if(onsettime_coding == 2)
              {
                if(strlen(scratchpad) > 8)
                {
                  if((scratchpad[2] == ':') && (scratchpad[5] == ':') && ((scratchpad[8] == '.') || (scratchpad[8] == ',')))
                  {
                    for(digits=0; digits<32; digits++)
                    {
                      if((scratchpad[9 + digits] < '0') || (scratchpad[9 + digits] > '9'))
                      {
                        break;
                      }
                    }
                    scratchpad[9 + digits] = 0;
                    onset = atoi(scratchpad) * 3600LL;
                    onset += (atoi(scratchpad + 3) * 60LL);
                    onset += (long long)(atoi(scratchpad + 6));
                    onset *= TIME_DIMENSION;
                    if(digits)
                    {
                      l_temp = (atoi(scratchpad + 9) * TIME_DIMENSION);
                      for(; digits>0; digits--)
                      {
                        l_temp /= 10LL;
                      }
                      onset += l_temp;
                    }
                    onset -= mainwindow->edfheaderlist[0]->l_starttime;
                    if(onset < last_onset)
                    {
                      last_onset = onset;
                      days++;
                    }

                    onset_is_set = 1;
                  }
                }
                if(strlen(scratchpad) > 7)
                {
                  if((scratchpad[1] == ':') && (scratchpad[4] == ':') && ((scratchpad[7] == '.') || (scratchpad[7] == ',')))
                  {
                    for(digits=0; digits<32; digits++)
                    {
                      if((scratchpad[8 + digits] < '0') || (scratchpad[8 + digits] > '9'))
                      {
                        break;
                      }
                    }
                    scratchpad[8 + digits] = 0;
                    onset = atoi(scratchpad) * 3600LL;
                    onset += (atoi(scratchpad + 2) * 60LL);
                    onset += (long long)(atoi(scratchpad + 5));
                    onset *= TIME_DIMENSION;
                    if(digits)
                    {
                      l_temp = (atoi(scratchpad + 8) * TIME_DIMENSION);
                      for(; digits>0; digits--)
                      {
                        l_temp /= 10LL;
                      }
                      onset += l_temp;
                    }
                    onset -= mainwindow->edfheaderlist[0]->l_starttime;
                    if(onset < last_onset)
                    {
                      last_onset = onset;
                      days++;
                    }

                    onset_is_set = 1;
                  }
                }
              }

              if(onsettime_coding == 3)
              {
                if(strlen(scratchpad) > 17)
                {
                  if((scratchpad[4] == '-') && (scratchpad[7] == '-') && (scratchpad[13] == ':') && (scratchpad[16] == ':'))
                  {
                    scratchpad[19] = 0;
                    date_time.year = atoi(scratchpad);
                    date_time.month = atoi(scratchpad + 5);
                    date_time.day = atoi(scratchpad + 8);
                    date_time.hour = atoi(scratchpad + 11);
                    date_time.minute = atoi(scratchpad + 14);
                    date_time.second = atoi(scratchpad + 17);
                    date_time_to_utc(&utc_time, date_time);
                    onset = utc_time - mainwindow->edfheaderlist[0]->utc_starttime;
                    onset *= TIME_DIMENSION;

                    onset_is_set = 1;
                  }
                }
              }

              if(onsettime_coding == 4)
              {
                if(strlen(scratchpad) > 19)
                {
                  if((scratchpad[4] == '-') && (scratchpad[7] == '-') && (scratchpad[13] == ':') && (scratchpad[16] == ':') && ((scratchpad[19] == ',') || (scratchpad[19] == '.')))
                  {
                    for(digits=0; digits<32; digits++)
                    {
                      if((scratchpad[20 + digits] < '0') || (scratchpad[20 + digits] > '9'))
                      {
                        break;
                      }
                    }
                    scratchpad[20 + digits] = 0;
                    date_time.year = atoi(scratchpad);
                    date_time.month = atoi(scratchpad + 5);
                    date_time.day = atoi(scratchpad + 8);
                    date_time.hour = atoi(scratchpad + 11);
                    date_time.minute = atoi(scratchpad + 14);
                    date_time.second = atoi(scratchpad + 17);
                    date_time_to_utc(&utc_time, date_time);
                    onset = utc_time - mainwindow->edfheaderlist[0]->utc_starttime;
                    onset *= TIME_DIMENSION;
                    if(digits)
                    {
                      l_temp = (atoi(scratchpad + 20) * TIME_DIMENSION);
                      for(; digits>0; digits--)
                      {
                        l_temp /= 10LL;
                      }
                      onset += l_temp;
                    }

                    onset_is_set = 1;
                  }
                }
              }
            }
          }

          if(column == descr_column)
          {
            if(onset_is_set)
            {
              strncpy(description, line + str_start, max_descr_length);
              description[max_descr_length] = 0;
              latin1_to_utf8(description, max_descr_length);
              descr_is_set = 1;
            }
          }
        }

        if(onset_is_set && descr_is_set)
        {
          if(strcmp(description, last_description) || (!ignore_consecutive))
          {
            annotation = (struct annotationblock *)calloc(1, sizeof(struct annotationblock));
            if(annotation == NULL)
            {
              QApplication::restoreOverrideCursor();
              UI_Messagewindow popuperror("Error", "Malloc error (annotation).");
              fclose(inputfile);
              ImportAnnotsDialog->setEnabled(TRUE);
              return;
            }
            annotation->onset = onset + (86400LL * TIME_DIMENSION * days);
            strncpy(annotation->annotation, description, MAX_ANNOTATION_LEN);
            annotation->annotation[MAX_ANNOTATION_LEN] = 0;
            edfplus_annotation_add_item(&mainwindow->annotationlist[0], annotation);

            strcpy(last_description, description);
          }
        }

        line_nr++;

        str_start = 0;

        i = 0;

        column = 0;

        column_end = 1;

        onset_is_set = 0;

        descr_is_set = 0;

        qApp->processEvents();

        continue;
      }

      i++;

      if(i>2046)
      {
        QApplication::restoreOverrideCursor();
        snprintf(scratchpad, 256, "Error, line %i is too long.\n", line_nr);
        UI_Messagewindow popuperror("Error", scratchpad);
        fclose(inputfile);
        ImportAnnotsDialog->setEnabled(TRUE);
        return;
      }
    }
  }

  QApplication::restoreOverrideCursor();

  if(mainwindow->annotations_dock[0] == NULL)
  {
    mainwindow->annotations_dock[0] = new UI_Annotationswindow(0, mainwindow);

    mainwindow->addDockWidget(Qt::RightDockWidgetArea, mainwindow->annotations_dock[0]->docklist, Qt::Vertical);

    if(!mainwindow->annotationlist[0])
    {
      mainwindow->annotations_dock[0]->docklist->hide();
    }
  }

  mainwindow->annotations_edited = 1;

  mainwindow->annotations_dock[0]->updateList(0);

  mainwindow->save_act->setEnabled(TRUE);

  if(csv_format)
  {
    if(fclose(inputfile))
    {
      UI_Messagewindow popuperror("Error", "An error occurred when closing inputfile.");
      ImportAnnotsDialog->setEnabled(TRUE);
      return;
    }
  }

  UI_Messagewindow popupmessage("Ready", "Done");

  ImportAnnotsDialog->setEnabled(TRUE);

  ImportAnnotsDialog->close();
}















