"""This module defines a jit function that wraps the jit function for
the chosen form compiler."""

# Copyright (C) 2008-2009 Johan Hake
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Anders Logg, 2008-2009.
# Modified by Garth N. Wells, 2011.
#
# First added:  2008-12-04
# Last changed: 2011-09-10

__all__ = ["jit"]

from functools import wraps

# Import SWIG-generated extension module (DOLFIN C++)
import dolfin.cpp as cpp
from dolfin.cpp import MPI, info, parameters, warning, info

# Set UFL log level
import ufl
ufl.set_level(ufl.INFO)

# Import check for correct swig version from instant
from instant import check_swig_version, get_swig_version, check_and_set_swig_binary

# Cached swig check
_swig_version_ok = False

def mpi_jit_decorator(local_jit, *args, **kwargs):
    """A decorator for jit compilation

    Use this function as a decorator to any jit compiler function.  In
    a paralell run, this function will first call the jit compilation
    function on the first process. When this is done, and the module
    is in the cache, it will call the jit compiler on the remaining
    processes, which will then use the cached module.

    *Example*
        .. code-block:: python

            def jit_something(something):
                ....

    """
    @wraps(local_jit)
    def mpi_jit(*args, **kwargs):

        # Just call JIT compiler when running in serial
        if MPI.num_processes() == 1:
            return local_jit(*args, **kwargs)

        # Compile first on process 0
        if MPI.process_number() == 0:
            output = local_jit(*args, **kwargs)

        # Wait for the first process to finish
        MPI.barrier()

        # Then compile on all other processes (which may then just read the cache)
        if not MPI.process_number() == 0:
            output = local_jit(*args,**kwargs)

        return output

    # Return the decorated jit function
    return mpi_jit

@mpi_jit_decorator
def jit(form, form_compiler_parameters=None, common_cell=None):
    """Just-in-time compile any provided form.

    It uses the jit function from the form compiler registered by
    parameters["form_compiler"]["name"].
    """

    # Check that form is not empty
    if isinstance(form, ufl.Form):
        if form.integrals() == ():
            raise RuntimeError, "Form is empty. Cannot pass to JIT compiler."

    global _swig_version_ok

    # Check and set swig binary
    if not _swig_version_ok and not check_and_set_swig_binary(\
        parameters["swig_binary"], parameters["swig_path"]):
        raise OSError, "Could not find swig installation. Pass an existing "\
              "swig binary or install SWIG version 2.0 or higher.\n"

    # Check that the form compiler will use the same swig version
    # that PyDOLFIN was compiled with
    _swig_version_ok = _swig_version_ok or \
                       check_swig_version(cpp.__swigversion__, same=True)

    if not _swig_version_ok:
        raise OSError, """\
PyDOLFIN was not compiled with the present version of swig.
Install swig version %s or recompiled PyDOLFIN with present swig
"""%cpp.__swigversion__

    # Cache swig version test
    _swig_version_ok = True

    # Import form compiler
    form_compiler_name = cpp.parameters["form_compiler"]["name"]
    try:
        form_compiler = __import__(form_compiler_name)
    except ImportError, message:
        print message
        warning("Could not import %s form compiler, falling back to FFC." % form_compiler_name)
        try:
            form_compiler = __import__("ffc")
        except:
            cpp.dolfin_error("jit.py",
                             "perform just-in-time compilation of form",
                             "Could not import FFC form compiler")

    # Prepare form compiler parameters
    p = form_compiler.default_parameters()

    # Set parameters from global DOLFIN parameter set
    for key, value in parameters["form_compiler"].iteritems():
        p[key] = value

    # Override with local parameters if any
    if form_compiler_parameters:
        p.update(form_compiler_parameters)

    # Set the swig binary and path
    p["swig_binary"] = parameters["swig_binary"]
    p["swig_path"] = parameters["swig_path"]

    # Get jit function
    try:
        jit_compile = form_compiler.jit
    except AttributeError:
        raise RuntimeError, "Form compiler must implement the jit function."

    return jit_compile(form, parameters=p, common_cell=common_cell)
