/******************************************************************************
 *
 * Beats-per-minute (BPM) detection routine
 *
 * Author        : Copyright (c) Olli Parviainen
 * Author e-mail : oparviai @ iki.fi
 * File created  : 11-Jan-2003
 *
 * Last changed  : $Date: 2003/10/13 00:35:13 $
 * File revision : $Revision: 1.1 $
 *
 * $Id: BPMDetect.h,v 1.1 2003/10/13 00:35:13 melanie_t Exp $
 *
 * License :
 *
 *  SoundTouch sound processing library
 *  Copyright (c) Olli Parviainen
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *****************************************************************************/

#ifndef _BPMDetect_H_
#define _BPMDetect_H_

#include <qthread.h>

/// Minimum allowed BPM rate
#define MIN_BPM 30

/// Maximum allowed BPM rate
#define MAX_BPM 230


class StreamSource;
class QWidget;

class QBPMEvent : public QEvent
{
public:
    enum Type {LoaderEvent = 1002};
    QBPMEvent(Type t=LoaderEvent);
    ~QBPMEvent();

	float bpm();
	void setBpm(float i_bpm);
protected:
	float s_bpm;
};

/// Internal class prototype 
class FIFOSampleBuffer;

/// Class for calculating BPM rate for audio data.
class BPMDetect : public QThread
{
protected:
    /// Auto-correlation accumulator bins.
    float *xcorr;
    
    /// Amplitude envelope sliding average approximation level accumulator
    float envelopeAccu;

    /// RMS volume sliding average approximation level accumulator
    float RMSVolumeAccu;

    /// Sample average counter.
    int decimateCount;

    /// Sample average accumulator for FIFO-like decimation.
    int decimateSum;

    /// Decimate sound by this coefficient to reach approx. 500 Hz.
    int decimateBy;

    /// Auto-correlation window length
    int windowLen;

    /// Beginning of auto-correlation window: Autocorrelation isn't being updated for
    /// the first these many correlation bins.
    int windowStart;
 
    /// FIFO-buffer for decimated processing samples.
    FIFOSampleBuffer *buffer;

    /// Initialize the class for WAV file.
    void init(StreamSource *source);

    /// Updates auto-correlation function for given number of decimated samples that 
    /// are read from the internal 'buffer' pipe (samples aren't removed from the pipe 
    /// though).
    void updateXCorr(const int process_samples      /// How many samples are processed.
                     );

    /// Decimates samples to approx. 500 Hz.
    ///
    /// \return Number of output samples.
    int decimate(short *dest,       ///< Destination buffer
                 const short *src,  ///< Source sample buffer
                 int numsamples     ///< Number of source samples.
                 );

    /// Reads and prepares a block of samples for processing. Stereo sound is converted 
    /// to mono and sound data is then decimated down to approx. 500 Hz.
    ///
    /// \return Number of samples moved to outBuffer.
    int inputSamples(StreamSource *source,  ///< WAV file where data is read from
                     short *outBuffer       ///< Buffer for samples.
                     );

    /// Processes block of samples: Envelopes the samples and then
    /// updates the autocorrelation estimation.
    void processBlock(short *samples,   ///< Pointer to input/working data buffer
                      int numSamples    ///< Number of samples in buffer
                      );

    /// Calculates amplitude envelope for the buffer of samples.
    /// Result is outputted in 'samples.
    void calcEnvelope(short *samples,   ///< Pointer to input/output data buffer
                      int numsamples    ///< Number of samples in buffer
                      );

	StreamSource *s_source;

    /// Analyzes the WAV file data and returns BPM rate. Notice that the
    /// WAV audio data has to be in 16bit sample format.
    ///
    /// \return Beats-per-minute rate, or zero if detection failed.
	float detectBpm(StreamSource *source);

	float s_res;

	bool s_abort;
	QWidget *s_parent;
public:
    /// Constructor.
	BPMDetect(QWidget *parent, StreamSource *source);

    /// Destructor.
	virtual ~BPMDetect();

	void run(void);
	float result();
	void abort();
};

#endif // _BPMDetect_H_
