# Written by John Hoffman
# Modified by Cameron Dale
# see LICENSE.txt for license information

# $Id: inifile.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Functions to read/write a Windows-style INI file

format::

    aa = "bb"
    cc = 11
    
    [eee]
    ff = "gg"

decodes to::

    d = { '': {'aa':'bb','cc':'11'}, 'eee': {'ff':'gg'} }

the encoder can also take this as input::

    d = { 'aa': 'bb, 'cc': 11, 'eee': {'ff':'gg'} }

though it will only decode in the above format.  Keywords must be strings.
Values that are strings are written surrounded by quotes, and the decoding
routine automatically strips any quotes from the values.
Booleans are written as integers.  Anything other than strings, integers, 
and floats may have unpredictable results.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module

"""

from cStringIO import StringIO
from types import DictType, StringType
import logging
from types import BooleanType

logger = logging.getLogger('DebTorrent.inifile')

def ini_write(f, d, comment=''):
    """Write the ini file.
    
    @type f: C{string}
    @param f: the file name to write
    @type d: C{dictionary}
    @param d: the data to write to the ini file
    @type comment: C{string}
    @param comment: a comment to write at the top of the file, hash marks
        will be prefixed (optional, default is no comment)
    @rtype: C{boolean}
    @return: whether the write succeeded
    
    """
    
    try:
        a = {'':{}}
        for k,v in d.items():
            assert type(k) == StringType
            k = k.lower()
            if type(v) == DictType:
                logger.debug('new section:' +k)
                if k:
                    assert not a.has_key(k)
                    a[k] = {}
                aa = a[k]
                for kk,vv in v:
                    assert type(kk) == StringType
                    kk = kk.lower()
                    assert not aa.has_key(kk)
                    if type(vv) == BooleanType:
                        vv = int(vv)
                    if type(vv) == StringType:
                        vv = '"'+vv+'"'
                    aa[kk] = str(vv)
                    logger.debug('a['+k+']['+kk+'] = '+str(vv))
            else:
                aa = a['']
                assert not aa.has_key(k)
                if type(v) == BooleanType:
                    v = int(v)
                if type(v) == StringType:
                    v = '"'+v+'"'
                aa[k] = str(v)
                logger.debug('a[\'\']['+k+'] = '+str(v))
        r = open(f,'w')
        if comment:
            for c in comment.split('\n'):
                r.write('# '+c+'\n')
            r.write('\n')
        l = a.keys()
        l.sort()
        for k in l:
            if k:
                r.write('\n['+k+']\n')
            aa = a[k]
            ll = aa.keys()
            ll.sort()
            for kk in ll:
                r.write(kk+' = '+aa[kk]+'\n')
        success = True
    except:
        logger.exception('Error writing config file: '+f)
        success = False
    try:
        r.close()
    except:
        pass
    return success


def errfunc(lineno, line, err):
    """Display an error message when reading the ini file fails.
    
    @type lineno: C{int}
    @param lineno: the line number that caused the error 
    @type line: C{string}
    @param line: the line that caused the error
    @type err: C{string}
    @param err: the error that was generated
    
    """
    
    logger.error('('+str(lineno)+') '+err+': '+line)

def ini_read(f):
    """Read the ini file.
    
    @type f: C{string}
    @param f: the file name to read
    @rtype: C{dictionary}
    @return: the data read from the ini file
    
    """
    try:
        r = open(f,'r')
        ll = r.readlines()
        d = {}
        dd = {'':d}
        for i in xrange(len(ll)):
            l = ll[i]
            l = l.strip()
            if not l:
                continue
            if l[0] == '#':
                continue
            if l[0] == '[':
                if l[-1] != ']':
                    errfunc(i,l,'syntax error')
                    continue
                l1 = l[1:-1].strip().lower()
                if not l1:
                    errfunc(i,l,'syntax error')
                    continue
                if dd.has_key(l1):
                    errfunc(i,l,'duplicate section')
                    d = dd[l1]
                    continue
                d = {}
                dd[l1] = d
                continue
            try:
                k,v = l.split('=',1)
            except:
                try:
                    k,v = l.split(':',1)
                except:
                    errfunc(i,l,'syntax error')
                    continue
            k = k.strip().lower()
            v = v.strip()
            if len(v) > 1 and ( (v[0] == '"' and v[-1] == '"') or
                                (v[0] == "'" and v[-1] == "'") ):
                v = v[1:-1]
            if not k:
                errfunc(i,l,'syntax error')
                continue
            if d.has_key(k):
                errfunc(i,l,'duplicate entry')
                continue
            d[k] = v
        logger.debug(str(dd))
    except:
        logger.exception('Error reading config file: '+f)
        dd = None
    try:
        r.close()
    except:
        pass
    return dd
