/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * md.c: Copyright (C) Eric Prevoteau <www@a2pb.gotdns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: md.c,v 1.4 2003/12/28 08:12:38 uid68112 Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <netinet/in.h>
#include <glib.h>

#include "display.h"
#include "md.h"
/* the following function compute 2 MD5SUM-like values for the given filename */
/* The following code is a simplification of "md5sum" program code */

/* The following is from gnupg-1.0.2's cipher/bithelp.h.  */
/* Rotate a 32 bit integer by n bytes */
#if defined __GNUC__ && defined __i386__
static inline guint32
rol(guint32 x, int n)
{  
  __asm__("roll %%cl,%0"
     :"=r" (x)
     :"0" (x),"c" (n));
  return x;
}
#else
# define rol(x,n) ( ((x) << (n)) | ((x) >> (32-(n))) )
#endif


/* These are the four functions used in the four steps of the MD5 algorithm
   and defined in the RFC 1321.  The first function is a little bit optimized
   (as found in Colin Plumbs public domain implementation).  */
#if 1
#define FF(b, c, d) ((b & c) | (~b & d))
#else
#define FF(b, c, d) (d ^ (b & (c ^ d)))
#endif
#define FG(b, c, d) FF (d, b, c)
#define FH(b, c, d) (b ^ c ^ d)
#define FI(b, c, d) (c ^ (b | ~d))

/*      b&c  ~b&d   FF1      c^d   &b  ^d=FF2
b c d   
0 0 0    0     0     0        0     0   0
0 0 1    0     1     1        1     0   1
0 1 0    0     0     0        1     0   0
0 1 1    0     1     1        0     0   1
1 0 0    0     0     0        0     0   0
1 0 1    0     0     0        1     1   0
1 1 0    1     0     1        1     1   1
1 1 1    1     0     1        0     0   1
*/


/* Process LEN bytes of BUFFER.
   It is assumed that LEN % 64 == 0.  */
void md5_process_block (const void *buffer, unsigned char md5sum[MD5SUMLEN])
{
	guint32 correct_words[16];
	const guint32 *words = buffer;
	const guint32 *endp = words + (BLOCKSIZE / sizeof (guint32));
	guint32 A = 0x67452301;
	guint32 B = 0xefcdab89;
	guint32 C = 0x98badcfe;
	guint32 D = 0x10325476;

	/* First increment the byte count.	RFC 1321 specifies the possible
		 length of the file up to 2^64 bits.	Here we only compute the
		 number of bytes.	Do a double word increment.	*/

	/* Process all bytes in the buffer with 64 bytes in each round of
		 the loop.	*/
	while (words < endp)
	{
		guint32 *cwp = correct_words;
		guint32 A_save = A;
		guint32 B_save = B;
		guint32 C_save = C;
		guint32 D_save = D;

		/* First round: using the given function, the context and a constant
		the next context is computed.	Because the algorithms processing
		unit is a 32-bit word and it is determined to work on words in
		little endian byte order we perhaps have to change the byte order
		before the computation.	To reduce the work for the next steps
		we store the swapped words in the array CORRECT_WORDS.	*/

		/* I have used htonl because it swaps if necessary into network byte order (big endian) */

#define OP(a, b, c, d, s, T)								 				\
		{																		\
	 		a = (a + T) + (FF (b, c, d) + (*cwp++ = htonl (*words)));	\
	 		++words;															\
	 		a = rol (a, s);												\
			a += b;															\
		}																		

		/* Before we start, one word to the strange constants.
		They are defined in RFC 1321 as

		T[i] = (int) (4294967296.0 * fabs (sin (i))), i=1..64, or
		perl -e 'foreach(1..64){printf "0x%08x\n", int (4294967296 * abs (sin $_))}'
		*/

		/* Round 1.	*/
		OP (A, B, C, D,	7, 0xd76aa478);
		OP (D, A, B, C, 12, 0xe8c7b756);
		OP (C, D, A, B, 17, 0x242070db);
		OP (B, C, D, A, 22, 0xc1bdceee);

		OP (A, B, C, D,	7, 0xf57c0faf);
		OP (D, A, B, C, 12, 0x4787c62a);
		OP (C, D, A, B, 17, 0xa8304613);
		OP (B, C, D, A, 22, 0xfd469501);

		OP (A, B, C, D,	7, 0x698098d8);
		OP (D, A, B, C, 12, 0x8b44f7af);
		OP (C, D, A, B, 17, 0xffff5bb1);
		OP (B, C, D, A, 22, 0x895cd7be);

		OP (A, B, C, D,	7, 0x6b901122);
		OP (D, A, B, C, 12, 0xfd987193);
		OP (C, D, A, B, 17, 0xa679438e);
		OP (B, C, D, A, 22, 0x49b40821);

		/* For the second to fourth round we have the possibly swapped words
		in CORRECT_WORDS.	Redefine the macro to take an additional first
		argument specifying the function to use.	*/
#undef OP
#define OP(f, a, b, c, d, k, s, T)							\
 		{											 				\
	 		a = (a + T) + (f (b, c, d) + correct_words[k]);	\
	 		a = rol (a, s);									\
	 		a += b;										 		\
 		}											 				

		/* Round 2.	*/
		OP (FG, A, B, C, D,	1,	5, 0xf61e2562);
		OP (FG, D, A, B, C,	6,	9, 0xc040b340);
		OP (FG, C, D, A, B, 11, 14, 0x265e5a51);
		OP (FG, B, C, D, A,	0, 20, 0xe9b6c7aa);

		OP (FG, A, B, C, D,	5,	5, 0xd62f105d);
		OP (FG, D, A, B, C, 10,	9, 0x02441453);
		OP (FG, C, D, A, B, 15, 14, 0xd8a1e681);
		OP (FG, B, C, D, A,	4, 20, 0xe7d3fbc8);

		OP (FG, A, B, C, D,	9,	5, 0x21e1cde6);
		OP (FG, D, A, B, C, 14,	9, 0xc33707d6);
		OP (FG, C, D, A, B,	3, 14, 0xf4d50d87);
		OP (FG, B, C, D, A,	8, 20, 0x455a14ed);

		OP (FG, A, B, C, D, 13,	5, 0xa9e3e905);
		OP (FG, D, A, B, C,	2,	9, 0xfcefa3f8);
		OP (FG, C, D, A, B,	7, 14, 0x676f02d9);
		OP (FG, B, C, D, A, 12, 20, 0x8d2a4c8a);

		/* Round 3.	*/
		OP (FH, A, B, C, D,	5,	4, 0xfffa3942);
		OP (FH, D, A, B, C,	8, 11, 0x8771f681);
		OP (FH, C, D, A, B, 11, 16, 0x6d9d6122);
		OP (FH, B, C, D, A, 14, 23, 0xfde5380c);

		OP (FH, A, B, C, D,	1,	4, 0xa4beea44);
		OP (FH, D, A, B, C,	4, 11, 0x4bdecfa9);
		OP (FH, C, D, A, B,	7, 16, 0xf6bb4b60);
		OP (FH, B, C, D, A, 10, 23, 0xbebfbc70);

		OP (FH, A, B, C, D, 13,	4, 0x289b7ec6);
		OP (FH, D, A, B, C,	0, 11, 0xeaa127fa);
		OP (FH, C, D, A, B,	3, 16, 0xd4ef3085);
		OP (FH, B, C, D, A,	6, 23, 0x04881d05);

		OP (FH, A, B, C, D,	9,	4, 0xd9d4d039);
		OP (FH, D, A, B, C, 12, 11, 0xe6db99e5);
		OP (FH, C, D, A, B, 15, 16, 0x1fa27cf8);
		OP (FH, B, C, D, A,	2, 23, 0xc4ac5665);

		/* Round 4.	*/
		OP (FI, A, B, C, D,	0,	6, 0xf4292244);
		OP (FI, D, A, B, C,	7, 10, 0x432aff97);
		OP (FI, C, D, A, B, 14, 15, 0xab9423a7);
		OP (FI, B, C, D, A,	5, 21, 0xfc93a039);

		OP (FI, A, B, C, D, 12,	6, 0x655b59c3);
		OP (FI, D, A, B, C,	3, 10, 0x8f0ccc92);
		OP (FI, C, D, A, B, 10, 15, 0xffeff47d);
		OP (FI, B, C, D, A,	1, 21, 0x85845dd1);

		OP (FI, A, B, C, D,	8,	6, 0x6fa87e4f);
		OP (FI, D, A, B, C, 15, 10, 0xfe2ce6e0);
		OP (FI, C, D, A, B,	6, 15, 0xa3014314);
		OP (FI, B, C, D, A, 13, 21, 0x4e0811a1);

		OP (FI, A, B, C, D,	4,	6, 0xf7537e82);
		OP (FI, D, A, B, C, 11, 10, 0xbd3af235);
		OP (FI, C, D, A, B,	2, 15, 0x2ad7d2bb);
		OP (FI, B, C, D, A,	9, 21, 0xeb86d391);

		/* Add the starting values of the context.	*/
		A += A_save;
		B += B_save;
		C += C_save;
		D += D_save;
	}

	/* Put checksum in context given as argument.	*/
	((unsigned long*)md5sum)[0]=htonl(A);
	((unsigned long*)md5sum)[1]=htonl(B);
	((unsigned long*)md5sum)[2]=htonl(C);
	((unsigned long*)md5sum)[3]=htonl(D);
}

/****************************************************/
/* convert the given md5sum into a printable string */
/****************************************************/
void md5tostr(const unsigned char md5sum[MD5SUMLEN],char str[3*MD5SUMLEN+1])
{
	int i;
	for(i=0;i<MD5SUMLEN;i++)
	{
		unsigned int v=md5sum[i];
		sprintf(str+3*i,"%1u%1u%1u",v/100,(v%100)/10,v%10);
	}
}

/****************************************/
/* convert the given string into md5sum */
/****************************************/
/* output: 1= ok, 0=invalid string */
/***********************************/
int strtomd5(const char str[3*MD5SUMLEN+1],unsigned char md5sum[MD5SUMLEN])
{
	int i;
	for(i=0;i<MD5SUMLEN;i++)
	{
		unsigned int v1,v2,v3;

		v1=str[3*i]-'0';
		v2=str[3*i+1]-'0';
		v3=str[3*i+2]-'0';

		/* valid digit ? (due to unsigned, value cannot be <0 */
		if((v1>9) || (v2>9) || (v3>9))
			return 0;

		v3+=(v1*100)+(v2*10);

		if(v3>255)
			return 0;
		md5sum[i]=v3;
	}
	return 1;
}

/****************************************/
/* compute the MD5SUM of the given file */
/****************************************/
/* output: 0=ok, !=0=error */
/***************************/
int md5sum_of_file(const char *filename, unsigned char md5sum[MD5SUMLEN])
{
	FILE *f;
	int a;

	char buf[BLOCKSIZE];

	f=fopen(filename,"rb");
	if(f==NULL)
	{
		disp_msg(ERR_MSG,"md5sum_of_file","fail to open",filename,NULL);
		return -1;
	}

	a=fread(buf,1,BLOCKSIZE,f);
	fclose(f);
	if(a!=BLOCKSIZE)
	{
		disp_msg(ERR_MSG,"md5sum_of_file",filename,"is too small",NULL);
		return -2;
	}

	md5_process_block (buf, md5sum);

	return 0;
}

