/* main.cxx
     $Id: main.cxx,v 1.56 2003/08/21 04:40:23 elf Exp $

   written by Marc Singer
   12 August 1996

   This file is part of the project CurVeS.  See the file README for
   more information.

   Copyright (C) 1996 Marc Singer

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   with your Debian GNU/Linux system, in
   /usr/share/common-licenses/GPL, or with the Debian GNU/Linux hello
   source package as the file COPYING. If not, write to the Free
   Software Foundation, Inc., 59 Temple Place -Suite 330, MA
   02111-1307, USA.

   -----------
   DESCRIPTION
   -----------

   Application main () for CurVeS, a CVS frontend.

   -- Signals

   The ncurses library defines some signal handlers to restore the
   screen and do cleanup.  Since their signal handlers don't chain
   ours, we need to make sure that theirs are loaded first because we
   DO chain to preexisting handlers.

   -- UseStatusHash

   If the UseStatusHash preference is true, we will try to use the -H
   switch for cvs status which can improve the performance of the call
   dramatically.

*/


#include "std.h"
#include "preferences.h"	// Everyone MUST get into the pref scene

#define DECLARE_VERSION
#include "version.h"

#include "curves.h"
#undef umode_t			// *** See NOTES
#include "signal.h"
#include "options.h"
#include "file.h"
#include "mouse.h"
#include "termcap.h"

static char *g_rgszMenu[] = {
  "CurVeS",
  "CurVeS: C)vs F)ile M)ark S)ort V)ersion Q)uit",
  "   CVS: A)dd B)inaryToggle C)ommit D)iff L)og U)pdate",
  "  File: C)hange dir O)ptions V)iew",
  "  Sort: A)lpha C)lass N)ewest O)ldest L)argest S)mallest", // T)ag",
  "  Mark: A)ddedNotYet C)lear E)dited U)ncommitted",
};

typedef enum {
  menuNul     = 0,
  menuMain    = 1,
  menuCVS     = 2,
  menuFile    = 3,
  menuSort    = 4,
  menuMark    = 5,
} eMenu;

char* g_szProgram;

extern int g_maskTrace;
int do_usage (OPTION*, const char*);
int do_version (OPTION*, const char*);

OPTION g_rgOption[] = {
  { "debug",	OPTION_F_SET_INT | OPTION_F_ARGUMENT,
    (void*) &g_maskTrace, NULL },
  { "version",	0, NULL, do_version },
  { "V",	0, NULL, do_version },
  { "help",	0, NULL, do_usage },
  { "h",	0, NULL, do_usage },
  { NULL },
};


void signal_suspend (LSignalHandle handle, void*)
{
  if (LSignal::signal_of (handle) == SIGTSTP) {
    LSignal::release (SIGTSTP, signal_suspend, NULL);
    //    fprintf (stderr, "Suspending\n");
    raise (LSignal::signal_of (handle));
  }
  else
;
//    LSignal::accept (SIGTSTP, signal_suspend, NULL, 0, 0);
}

int read_command (LCurves* pCurves)
{
  int result = pCurves->getch ();
  if (result < 128 && isalpha (result))
    result = tolower (result);
  return result;
}

int do_usage (OPTION*, const char*)
{
  printf (
"usage: %s [options]\n"
"  --debug <value>      Decimal or hexadecimal debug mask\n"
"  --version, -V        Display version and copyright\n"
"  --help, -h           Show this usage message\n"
,
	  g_szProgram);
  return 1;
}  /* do_usage */

int do_version (OPTION*, const char*)
{
  printf ("CurVeS version %s -- %s\n", g_szVersion, g_szCopyright);
  return 1;
}

int main (int argc, char** argv)
{
  int argc_used = 0;
  g_szProgram = parse_application (*argv);

	// -- Options from ~/.curves first
  // FIXME: we should probably use a document for this data and 'read'
  // from it, memory, instead of this clumsy call construct.  Plus, if
  // there is a document, we can use it to create the user's default
  // preferences file.
  g_preferences.associate ("ChangeLogFileName", "./ChangeLog");
  g_preferences.associate ("ChangeLogFormat", "$date $author\n\t$comment");
  g_preferences.associate ("Debug", "0");
  g_preferences.associate ("DebugOutput", "local");
  g_preferences.associate ("DebugTimestamps", "none");
  g_preferences.associate ("MaintainChangeLog", "yes");
  g_preferences.associate ("SenseBinaryFiles", "yes");
  g_preferences.associate ("Sort", "ca");
  //  g_preferences.associate ("CommitCommentEditor", "internal");
  g_preferences.associate ("ColorScheme", "1");
  g_preferences.associate ("InhibitAltCharset", "no");
  g_preferences.associate ("UseStatusHash", "no");
  g_preferences.associate ("cvs", "cvs");
  g_preferences.associate ("cvs_server", "");
  {
    const char* szPager = getenv ("PAGER");
    if (szPager == NULL)
      szPager = "more";
    g_preferences.associate ("more", szPager);
  }
  g_preferences.transfer_defaults ();
  g_preferences.read (SZ_PATH_PREFERENCES);

  g_maskTrace = g_preferences.as_int ("Debug");

  TRACE((T_INFO, "Starting Curves"));


  // FIXME: fetch debug flag from preferences

	// -- Options from command line after
  if (parse_options (argc - 1, argv + 1, g_rgOption, &argc_used))
    return 0;
  
  g_preferences.associate ("Debug", g_maskTrace);

  argc -= argc_used;
  argv += argc_used;

  LCurves curves;
  LMouse mouse;
  mouse.init ();

  LSignal::accept (SIGTSTP, signal_suspend, NULL, 0, 0);
  LSignal::accept (SIGCONT, signal_suspend, NULL, 0, 0);

  curves.init_menus (g_rgszMenu, sizeof (g_rgszMenu)/sizeof (char*));

  char szPath[256];
  if (argc > 1) {
    strncpy (szPath, argv[1], sizeof (szPath) - 1);
    szPath[sizeof (szPath) - 1] = 0;
  }
  else
    getcwd (szPath, sizeof (szPath));
  curves.set_path (szPath);
  
  curves.refresh ();
  curves.explore ();
  curves.refresh ();

  int command;
  bool fQuit = false;
  int result;

  char szSearch[100];		// Last search

#define MENU_ID(menu,ch) (((menu) << 16) + (ch)) // Requires 32 bit integers

  curves.set_menu (menuMain);
  curves.refresh ();

  while (!fQuit) {
    char sz[256];		// For command input
    switch (MENU_ID (curves.menu (), command = read_command (&curves))) {
    case MENU_ID (menuFile, 27):
    case MENU_ID (menuFile, 7):
    case MENU_ID (menuSort, 27):
    case MENU_ID (menuSort, 7):
    case MENU_ID (menuMark, 27):
    case MENU_ID (menuMark, 7):
    case MENU_ID (menuCVS, 27):
    case MENU_ID (menuCVS, 7):
      curves.set_menu (menuMain);
      curves.refresh ();
      break;

    case MENU_ID (menuMain, '\t'):
      {
	int iPane = curves.pane () ? 0 : 1;
	if (curves.can_select_pane (iPane))
	  curves.select_pane (iPane);
      }
      break;

    case MENU_ID (menuMain, '\f'):
      curves.refresh (true);
      break;

    case MENU_ID (menuMain, 'c'):
      curves.set_menu (menuCVS);
      curves.refresh ();
      break;

    case MENU_ID (menuMain, 'f'):
      curves.set_menu (menuFile);
      curves.refresh ();
      break;

    case MENU_ID (menuMain, 'q'):
      fQuit = true;
      curves.show_cursor (false);
      curves.win_command ()->erase ();
      //      curves.win_command ()->defrefresh ();
      curves.update ();
      break;

    case MENU_ID (menuMain, KEYCODE (termcapKeyDown)):
    case MENU_ID (menuMain, 'j'):
      if (!curves.pane ())
	curves.do_directory_move (1);
      else
	curves.do_file_move (1);
      break;

    case MENU_ID (menuMain, KEYCODE (termcapKeyUp)):
    case MENU_ID (menuMain, 'k'):
      if (!curves.pane ())
	curves.do_directory_move (-1);
      else
	curves.do_file_move (-1);
      break;

    case MENU_ID (menuMain, KEYCODE (termcapKeyLeft)):
      if (curves.pane ())
	curves.do_file_move (-2);
      break;
    case MENU_ID (menuMain, KEYCODE (termcapKeyRight)):
      if (curves.pane ())
	curves.do_file_move (2);
      break;

    case MENU_ID (menuMain, '\n'):
    case MENU_ID (menuMain, '\r'):
    case MENU_ID (menuMain, ' '):
      if (curves.pane ())
	curves.do_file_tag ();
      else if (command == '\n' || command == '\r') {
	curves.set_menu (menuNul);
	curves.do_directory_change ();
	curves.win_command ()->erase ();
	curves.refresh ();
	curves.explore ();
	curves.set_menu (menuMain);
	curves.refresh ();
      }
      break;

    case MENU_ID (menuMain, 's'):
      curves.set_menu (menuSort);
      curves.refresh ();
      {
	const char* szSort = g_preferences.fetch ("Sort");
	assert_ (szSort);
	strcpy (sz, szSort);
      }
      result = curves.win_command ()->prompt (Position (0, 0),
					      "Sort Criteria: ", 
					      sz, sizeof (sz), 
					      sz);
      curves.win_command ()->erase ();
      if (!result) {
	curves.show_cursor (false);
	g_preferences.associate ("Sort", sz);
	//	curves.set_sort (sz);
	curves.sort ();
	curves.dirty_path ();
      }
      curves.set_menu (menuMain);
      curves.refresh ();
      break;

    case MENU_ID (menuMain, 'v'):
      curves.win_command ()->draw_text (Position (0, 0),
					"CurVeS version %s -- %s",
					g_szVersion, g_szCopyright);
      curves.refresh ();
      break;

    case MENU_ID (menuMain, 'm'):
      curves.set_menu (menuMark);
      curves.refresh ();
      break;

    case MENU_ID (menuMain, '/'):
      result = curves.win_command ()->prompt (Position (0, 0), "Search: ",
					      szSearch, sizeof (szSearch), 
					      szSearch);
      curves.do_search (szSearch, false);
      curves.win_command ()->erase ();
      curves.refresh ();
      break;

    case MENU_ID (menuMain, '?'):
      result = curves.win_command ()->prompt (Position (0, 0),
					      "Reverse Search: ",
					      szSearch, sizeof (szSearch), 
					      szSearch);
      curves.do_search (szSearch, true);
      curves.win_command ()->erase ();
      curves.refresh ();
      break;

    case MENU_ID (menuFile, 'c'):
      curves.set_menu (menuNul);
      curves.refresh ();
      result = curves.win_command ()->prompt (Position (0, 0),
					      "Change Directory: ",
					      sz, sizeof (sz),
					      curves.path_abbreviate
					      (curves.win_command ()->dx ()
					       - 19));

      curves.win_command ()->erase ();
      if (!result) {
	curves.set_path (sz);
	curves.refresh ();
	curves.explore ();
      }
      curves.set_menu (menuMain);
      curves.refresh ();
      break;

    case MENU_ID (menuFile, 'o'):
      curves.win_command ()->draw_text (Position (0, 0), "Editing options");
      curves.refresh ();
      curves.save ();
      curves.show_cursor (false);
      {
	const char* szFile = file_edit (SZ_PATH_PREFERENCES);
	if (szFile) {
	  file_copy (SZ_PATH_PREFERENCES, szFile);
	  unlink (szFile);
	  g_preferences.release_settings ();
	  g_preferences.read (SZ_PATH_PREFERENCES); // Reread preferences
	}
      }
      curves.set_menu (menuMain);
      curves.sort ();
      curves.dirty_path ();
      curves.restore (true);
      curves.show_cursor (true);
      curves.refresh ();
      break;

    case MENU_ID (menuFile, 'v'):
      if (curves.pane ()) {
	curves.save ();
	{
	  char sz[256];
	  sprintf (sz, "%s %s/%s", g_preferences.fetch ("more"), 
		   curves.path (), curves.file ());
	  TRACE((T_CVS_TRACE, "Executing: '%s'\n", sz));
	  system (sz);
	}
	curves.set_menu (menuMain);
	curves.restore (true);
      }
      curves.set_menu (menuMain);
      break;

    case MENU_ID (menuCVS, 'b'):
      if (curves.pane ()) {
	curves.do_file_binarytoggle ();
	curves.set_menu (menuMain);
	curves.refresh ();
      }
      break;
    case MENU_ID (menuCVS, 'c'):
      if (curves.pane () || curves.count_tagged_files ()) {
	char szPrompt[80];
	int cTagged = curves.count_tagged_files ();
	strcpy (szPrompt, "Commit ");
	if (cTagged)
	  sprintf (szPrompt + strlen (szPrompt), "(%d) ", cTagged);
	strcat (szPrompt, "comment: ");

	result = curves.win_command ()->prompt (Position (0, 0), szPrompt,
						sz, sizeof (sz), NULL);
	curves.win_command ()->erase ();

	if (result == 0) {
	  curves.show_cursor (false);

	  curves.save ();
	  {
	    char* szFiles = curves.name_tagged_files (true);
	    char* szCmd = (char*) malloc (strlen (curves.path ())
					  + strlen (szFiles) + strlen (sz) 
					  + CB_COMMAND);
	    sprintf (szCmd, "cd %s ; %s commit -m \"%s\" %s", 
		     curves.path (), g_preferences.fetch ("cvs"), sz, szFiles);
	    TRACE((T_CVS_TRACE, "Executing: '%s'\n", szCmd));
	    system (szCmd);
	    free (szCmd);
	    curves.do_update (szFiles);
	    free (szFiles);
	  }
	  curves.set_menu (menuMain);
	  curves.restore (true);
	  curves.tag (classNul, true); 	// Clear all tags
	  curves.refresh ();
	  curves.update ();
	}
      }
      curves.set_menu (menuMain);
      break;

    case MENU_ID (menuCVS, 'l'):
      if (curves.pane () || curves.count_tagged_files ()) {
	curves.save ();
	{
	  char* szFiles = curves.name_tagged_files (true);
	  char* szCmd = (char*) malloc (strlen (curves.path ()) 
					+ strlen (szFiles) + CB_COMMAND*2);
	  sprintf (szCmd, "(cd %s ; %s log %s ) 2>&1 | %s", curves.path (), 
		   g_preferences.fetch ("cvs"), szFiles, 
		   g_preferences.fetch ("more"));
	  TRACE((T_CVS_TRACE, "Executing: '%s'\n", szCmd));
	  system (szCmd);
	  free (szCmd);
	}
	curves.set_menu (menuMain);
	curves.restore (true);
      }
      curves.set_menu (menuMain);
      break;

    case MENU_ID (menuCVS, 'd'):
      if (curves.pane () || curves.count_tagged_files ()) {
	curves.save ();
	{
	  char* szFiles = curves.name_tagged_files (true);
	  char* szCmd = (char*) malloc (strlen (curves.path ()) 
					+ strlen (szFiles) + CB_COMMAND*2);
	  sprintf (szCmd, "(cd %s ; %s diff -u %s ) 2>&1 | %s",
		   curves.path (), g_preferences.fetch ("cvs"), szFiles,
		   g_preferences.fetch ("more"));
	  TRACE((T_CVS_TRACE, "Executing: '%s'\n", szCmd));
	  system (szCmd);
	  free (szCmd);
	  free (szFiles);
	}
	curves.set_menu (menuMain);
	curves.restore (true);
      }
      curves.set_menu (menuMain);
      break;

    case MENU_ID (menuCVS, 'u'):
      curves.save ();
      {
	  char* sz = (char*) malloc (strlen (curves.path ()) + CB_COMMAND*2);
	  bool fOK = false;
	  if (g_preferences.as_bool ("UseStatusHash")) {
	    sprintf (sz, "cd %s ; %s update -l -H 2>&1 | %s", curves.path (),
		     g_preferences.fetch ("cvs"), 
		     g_preferences.fetch ("more"));
	    TRACE((T_CVS_TRACE, "Executing: '%s'\n", sz));
	    fOK = (system (sz) == 0);
	  }
	  if (!fOK) {
	    sprintf (sz, "cd %s ; %s update -l 2>&1 | %s", curves.path (),
		     g_preferences.fetch ("cvs"), 
		     g_preferences.fetch ("more"));
	    TRACE((T_CVS_TRACE, "Executing: '%s'\n", sz));
	    system (sz);
	  }
	  curves.set_path (NULL); // Discard current directory's contents
	  free (sz);
      }
      curves.set_menu (menuMain);
      curves.restore (true);
      curves.explore ();
      curves.refresh ();
      break;

    case MENU_ID (menuCVS, 'a'):
      if (curves.pane () || curves.count_tagged_files ()) {
	curves.save ();
	{
				// Add text mode files
	  char* szFilesT = curves.name_files (true, flagTag | flagBinary,
					      flagTag);
	  char* szCmdT = NULL;
	  if (szFilesT) {
	    szCmdT = (char*) malloc (strlen (szFilesT) + CB_COMMAND);
	    sprintf (szCmdT, "%s add %s",
		     g_preferences.fetch ("cvs"), szFilesT);
	  }
				// Add binary mode files
	  char* szFilesB = curves.name_files (true, flagTag | flagBinary,
					      flagTag | flagBinary);
	  char* szCmdB = NULL;
	  if (szFilesB) {
	    szCmdB = (char*) malloc (strlen (szFilesB) + CB_COMMAND);
	    sprintf (szCmdB, "%s add -kb %s", g_preferences.fetch ("cvs"), 
		     szFilesB);
	  }
	  if (szCmdT || szCmdB) {
	    char* szCmd = (char*) malloc ((szCmdT ? strlen (szCmdT) : 0)
					  + (szCmdB ? strlen (szCmdB) : 0)
					  + sizeof (curves.path ())
					  + CB_COMMAND);
	    sprintf (szCmd, "(cd %s ; ", curves.path ());
	    if (szCmdT)
	      strcat (szCmd, szCmdT);
	    if (szCmdT && szCmdB)
	      strcat (szCmd, "; ");
	    if (szCmdB)
	      strcat (szCmd, szCmdB);
	    sprintf (szCmd + strlen (szCmd), ") 2>&1 | %s", 
		     g_preferences.fetch ("more"));
	    TRACE((T_CVS_TRACE, "Executing: '%s'\n", szCmd));
	    system (szCmd);
	    free (szCmd);
	    if (szFilesT) {
	      curves.do_update (szFilesT);
	      free (szFilesT);
	      free (szCmdT);
	    }
	    if (szFilesB) {
	      curves.do_update (szFilesB);
	      free (szFilesB);
	      free (szCmdB);
	    }
	  }
	}
	curves.set_menu (menuMain);
	curves.restore (true);
	curves.tag (classNul, true); 	// Clear all tags
	curves.refresh ();
	curves.update ();
      }
      curves.set_menu (menuMain);
      break;

    case MENU_ID (menuMark, 'a'):
      curves.tag (classNewAdd, true);
      curves.set_menu (menuMain);
      curves.refresh ();
      curves.update ();
      break;

    case MENU_ID (menuMark, 'c'):
      curves.tag (classNul, true);
      curves.set_menu (menuMain);
      curves.refresh ();
      curves.update ();
      break;

    case MENU_ID (menuMark, 'e'):
      curves.tag (classEdited, true);
      curves.set_menu (menuMain);
      curves.refresh ();
      curves.update ();
      break;

    case MENU_ID (menuMark, 'u'):
      curves.tag (classEdited, true);
      curves.tag (classAdded, false);
      curves.tag (classRemoved, false);
      curves.set_menu (menuMain);
      curves.refresh ();
      curves.update ();
      break;

    default:
//      curves.win_command ()->printfxy (20, 0, "-%d-", command);
      break;
    }
  }
}
