# utils.rb, copyright (c) 2006 by Vincent Fourmond: 
# Some small utility functions
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'Tioga/FigureMaker'

module CTioga

  # A module for small convenience functions.
  module Utils

    # A function that interprets a string according to a hash, and
    # returns either what was found in the hash, or the result of the
    # given block if that was not found -- or just the original string
    # itself when no block was given and no corresponding hash element
    # was found. Correspondance is done with the === operator.
    #
    # Actually, the first argument doesn't need to be a string.
    def self.interpret_arg(arg, hash)
      for key, val in hash
        if key === arg
          return val
        end
      end
      if block_given?
        return yield(arg)
      else
        return arg
      end
    end

    # Returns a frame specification from a classical array
    def self.framespec_str(a)
      return "top : #{a[2]} bottom: #{a[3]} " +
        "left : #{a[0]} right: #{a[1]}"
    end

    def self.frames_str(t)
      return framespec_str([t.frame_left, t.frame_right,
                            t.frame_top, t.frame_bottom])
    end

    # A function that takes care of converting the given parameter
    # to a float, while ensuring that it was decently taken care of.
    # Returns false if that happens not to be a float.
    def safe_float(a)
      return begin
               Float(a)
             rescue
               warn "Expected a float, but got '#{a}' instead"
               false
             end
    end

    # A function that 'applies' a margin specification (like for
    # show_plot_with_legend) to a frame spec (left,right,top,bottom).
    #
    # The margin spec has to contain all plot_*_margin things.
    def self.apply_margin_to_frame(frame, margin)
      width = frame[1] - frame[0]
      height = frame[2] - frame[3]
      dest = 
        [ 
         frame[0] + width * (margin['plot_left_margin'] || 
                             margin['left']),
         frame[1] - width * (margin['plot_right_margin']|| 
                             margin['right']),
         frame[2] - height * (margin['plot_top_margin'] || 
                              margin['top']),
         frame[3] + height * (margin['plot_bottom_margin'] || 
                              margin['bottom'])
        ]
      return dest
    end

    def self.margin_hash(a)
      if Hash === a
        return a
      else
        return {
          'left' => a[0],
          'right' => a[1],
          'top' => a[2],
          'bottom' => a[3]
        }
      end
    end


    # Converts a location into the index of the corresponding
    # coordinates...
    Locations  = {
      Tioga::FigureConstants::LEFT => 0,
      Tioga::FigureConstants::RIGHT => 1,
      Tioga::FigureConstants::TOP => 2,
      Tioga::FigureConstants::BOTTOM => 3,
    }

    def self.location_index(d)
      return Locations[d]
    end


    FrameNames = {
      :left => 0,
      :right => 1,
      :bottom => 2,
      :top => 3
    }


    # Returns true if the spec is left, right, top or bottom
    def self.side?(spec)
      return FrameNames.key?(spec.to_sym)
    end

    # Returns the position corresponding to the side:
    def self.side(spec)
      return FrameNames[spec.to_sym]
    end

  end

  # A small module to deal with versions and dates
  module Version

    # The position of the URL, used for getting the version
    SVN_URL = '$HeadURL: svn+ssh://fourmond@rubyforge.org/var/svn/sciyag/releases/ctioga-1.8/lib/utils.rb $'
    
    CTIOGA_VERSION = if SVN_URL =~ /releases\/ctioga-([^\/]+)/
                       $1
                     else
                       "SVN version"
                     end
    
    # The current version of the program.
    def self.version
      if CTIOGA_VERSION =~ /SVN/
        return "SVN, revision #{SVN_INFO['revision']}, #{SVN_INFO['date']}"
      else
        return CTIOGA_VERSION
      end
    end

    SVN_INFO = {
      'revision' => 0,
      'date' => "old"
    }
    
    def self.register_svn_info(rev_str, date_str)
      if rev_str =~ /(\d+)/
        rev = $1
        str = 'Date'
        date = date_str.gsub(/\$#{str}: (.*)\$/) { $1 }
        if SVN_INFO['revision'] < rev.to_i
          SVN_INFO['revision'] = rev.to_i
          SVN_INFO['date'] = date
        end
      end
    end

    register_svn_info('$Revision: 780 $', '$Date: 2008-03-09 14:25:14 +0100 (Sun, 09 Mar 2008) $')
  end
end
