/* $Id: cpl_type.c,v 1.12 2010/11/11 09:23:18 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2005 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: llundin $
 * $Date: 2010/11/11 09:23:18 $
 * $Revision: 1.12 $
 * $Name: cpl-5_3_0-BRANCH $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <complex.h>

#include "cpl_type.h"

#include "cpl_error.h"


/**
 * @defgroup cpl_types Type codes
 *
 * This module provides codes for the basic types (@c char, @c int, @c float,
 * etc.). These type codes may be used to indicate the type of a value stored
 * in another object, the value of a property or the pixel of an image
 * for instance. In addition, a utility function is provided to compute
 * the size, which is required to store a value of the type indicated by a
 * given type code.
 * 
 * 
 *   <table class="ec" align="center">
 *     <tr>
 *       <td class="ecl">CPL_TYPE_FLAG_ARRAY</td>
 *       <td class="ecr">
 *         Flag indicating whether a type is an array or a basic type.
 *       </td>
 *     </tr>
 *     <tr>
 *       <td class="ecl">CPL_TYPE_INVALID</td>
 *       <td class="ecr">
 *        Invalid or undetermined type. 
 *       </td>
 *     </tr>
 *     <tr>
 *       <td class="ecl">CPL_TYPE_CHAR</td>
 *       <td class="ecr">
 *         Type code corresponding to type @c char.
 *       </td>
 *     </tr>
 *     <tr>
 *       <td class="ecl">CPL_TYPE_UCHAR</td>
 *       <td class="ecr">
 *         Type code corresponding to type @c unsigned char.
 *       </td>
 *     </tr>
 *     <tr>
 *       <td class="ecl">CPL_TYPE_BOOL</td>
 *       <td class="ecr">
 *         Type code corresponding to the boolean type.
 *       </td>
 *     </tr>
 *     <tr>
 *       <td class="ecl">CPL_TYPE_INT</td>
 *       <td class="ecr">
 *         Type code corresponding to type @c int.
 *       </td>
 *     </tr>
 *     <tr>
 *       <td class="ecl">CPL_TYPE_UINT</td>
 *       <td class="ecr">
 *         Type code corresponding to type @c unsigned int.
 *       </td>
 *     </tr>
 *     <tr>
 *       <td class="ecl">CPL_TYPE_LONG</td>
 *       <td class="ecr">
 *         Type code corresponding to type @c long.
 *       </td>
 *     </tr>
 *     <tr>
 *       <td class="ecl">CPL_TYPE_ULONG</td>
 *       <td class="ecr">
 *         Type code corresponding to type @c unsigned long.
 *       </td>
 *     </tr>
 *     <tr>
 *       <td class="ecl">CPL_TYPE_FLOAT</td>
 *       <td class="ecr">
 *         Type code corresponding to type @c float.
 *       </td>
 *     </tr>
 *     <tr>
 *       <td class="ecl">CPL_TYPE_DOUBLE</td>
 *       <td class="ecr">
 *         Type code corresponding to type @c double.
 *       </td>
 *     </tr>
 *     <tr>
 *       <td class="ecl">CPL_TYPE_POINTER</td>
 *       <td class="ecr">
 *         Type code corresponding to a pointer type.
 *       </td>
 *     </tr>
 *     <tr>
 *       <td class="ecl">CPL_TYPE_STRING</td>
 *       <td class="ecr">
 *         Type code corresponding to a character array.
 *       </td>
 *     </tr>
 *   </table>
 *
 * 
 * The module 
 * @par Synopsis
 * @code
 *   #include <cpl_type.h>
 * @endcode
 */

/**@{*/

/**
 * @brief
 *   Compute the size of a type.
 *
 * @param type  Type code to be evaluated.
 *
 * @return The size of the fundamental type, or 0 in case an invalid type
 *   code was given.
 *
 * The function computes the atomic size of the type @em type. The result
 * for fundamental types like @c CPL_TYPE_FLOAT is what you would expect
 * from the C @b sizeof() operator. For arrays, i.e. types having the 
 * @c CPL_TYPE_FLAG_ARRAY set the returned size is not the size of a pointer
 * to @c CPL_TYPE_FLOAT for instance, but the size of its fundamental type,
 * i.e. the returned size is same as for the type @c CPL_TYPE_FLOAT.
 *
 * Especially for the type @c CPL_TYPE_STRING, which is explicitly defined for
 * convenience reasons, the size returned by this function is the size of
 * @c CPL_TYPE_CHAR!
 */

size_t 
cpl_type_get_sizeof(cpl_type type)
{

    size_t sz;


    if (type & CPL_TYPE_INVALID || type & CPL_TYPE_UNSPECIFIED)
        return 0;

    /*
     * We just want the size of the fundamental type, therefore the
     * array flag is removed. This also means that an explicit case
     * for CPL_TYPE_STRING is not necessary in the following switch.
     */

    type &= ~CPL_TYPE_FLAG_ARRAY;

    switch (type) {
        case CPL_TYPE_CHAR:
            sz = sizeof(char);
            break;

        case CPL_TYPE_UCHAR:
            sz = sizeof(unsigned char);
            break;

        case CPL_TYPE_BOOL:
            sz = sizeof(cpl_boolean);
            break;

        case CPL_TYPE_INT:
            sz = sizeof(int);
            break;

        case CPL_TYPE_UINT:
            sz = sizeof(unsigned int);
            break;

        case CPL_TYPE_LONG:
            sz = sizeof(long int);
            break;

        case CPL_TYPE_ULONG:
            sz = sizeof(unsigned long int);
            break;

        case CPL_TYPE_FLOAT:
            sz = sizeof(float);
            break;

        case CPL_TYPE_DOUBLE:
            sz = sizeof(double);
            break;

        case CPL_TYPE_POINTER:
            sz = sizeof(void*);
            break;

        case CPL_TYPE_FLOAT_COMPLEX:
            sz = sizeof(float complex);
            break;

        case CPL_TYPE_DOUBLE_COMPLEX:
            sz = sizeof(double complex);
            break;

        default:
            sz = 0;
            break;
    }

    return sz;

}

/**
 * @brief
 *   Get a string with the name of a type, e.g. "char", "int", "float"
 *
 * @param type  Type code to be evaluated.
 *
 * @return A pointer to a string literal with the name or empty string on error.
 *
 */

const char *
cpl_type_get_name(cpl_type type)
{

    const char * self = "";


    switch (type) {
        /* FIXME: This is a valid value for the enum... */
        case CPL_TYPE_FLAG_ARRAY:
            self = "array";
            break;

        /* FIXME: This is a valid value for the enum... */
        case CPL_TYPE_INVALID:
            self = "invalid";
            break;

        case CPL_TYPE_CHAR:
            self = "char";
            break;

        case CPL_TYPE_UCHAR:
            self = "unsigned char";
            break;

        case CPL_TYPE_BOOL:
            self = "boolean";
            break;

        case CPL_TYPE_INT:
            self = "int";
            break;

        case CPL_TYPE_UINT:
            self = "unsigned int";
            break;

        case CPL_TYPE_LONG:
            self = "long int";
            break;

        case CPL_TYPE_ULONG:
            self = "unsigned long int";
            break;

        case CPL_TYPE_FLOAT:
            self = "float";
            break;

        case CPL_TYPE_DOUBLE:
            self = "double";
            break;

        case CPL_TYPE_POINTER:
            self = "pointer";
            break;

        case CPL_TYPE_FLOAT_COMPLEX:
            self = "float complex";
            break;

        case CPL_TYPE_DOUBLE_COMPLEX:
            self = "double complex";
            break;

        case CPL_TYPE_UNSPECIFIED:
            self = "unspecified";
            break;

        case CPL_TYPE_STRING:
            self = "string";
            break;

            /* Here follows all "usable" types as arrays */

        case CPL_TYPE_CHAR | CPL_TYPE_POINTER:
            /* string: case CPL_TYPE_STRING | CPL_TYPE_FLAG_ARRAY: */
            self = "char array";
            break;

        case CPL_TYPE_UCHAR | CPL_TYPE_POINTER:
        case CPL_TYPE_UCHAR | CPL_TYPE_FLAG_ARRAY:
            self = "unsigned char array";
            break;

        case CPL_TYPE_BOOL | CPL_TYPE_POINTER:
        case CPL_TYPE_BOOL | CPL_TYPE_FLAG_ARRAY:
            self = "boolean array";
            break;

        case CPL_TYPE_INT | CPL_TYPE_POINTER:
        case CPL_TYPE_INT | CPL_TYPE_FLAG_ARRAY:
            self = "int array";
            break;

        case CPL_TYPE_UINT | CPL_TYPE_POINTER:
        case CPL_TYPE_UINT | CPL_TYPE_FLAG_ARRAY:
            self = "unsigned int array";
            break;

        case CPL_TYPE_LONG | CPL_TYPE_POINTER:
        case CPL_TYPE_LONG | CPL_TYPE_FLAG_ARRAY:
            self = "long int array";
            break;

        case CPL_TYPE_ULONG | CPL_TYPE_POINTER:
        case CPL_TYPE_ULONG | CPL_TYPE_FLAG_ARRAY:
            self = "unsigned long int array";
            break;

        case CPL_TYPE_FLOAT | CPL_TYPE_POINTER:
        case CPL_TYPE_FLOAT | CPL_TYPE_FLAG_ARRAY:
            self = "float array";
            break;

        case CPL_TYPE_DOUBLE | CPL_TYPE_POINTER:
        case CPL_TYPE_DOUBLE | CPL_TYPE_FLAG_ARRAY:
            self = "double array";
            break;

        case CPL_TYPE_FLOAT_COMPLEX | CPL_TYPE_POINTER:
        case CPL_TYPE_FLOAT_COMPLEX | CPL_TYPE_FLAG_ARRAY:
            self = "float complex array";
            break;

        case CPL_TYPE_DOUBLE_COMPLEX | CPL_TYPE_POINTER:
        case CPL_TYPE_DOUBLE_COMPLEX | CPL_TYPE_FLAG_ARRAY:
            self = "double complex array";
            break;

        case CPL_TYPE_STRING | CPL_TYPE_POINTER:
            /* char: case CPL_TYPE_STRING | CPL_TYPE_FLAG_ARRAY: */
            self = "string array";
            break;

        default:
            (void)cpl_error_set_message(cpl_func, CPL_ERROR_INVALID_TYPE,
                                        "type=0x%x", (int)type);
            break;
    }

    return self;

}

/**@}*/
