!-----------------------------------------------------------------------------!
!   CP2K: A general program to perform molecular dynamics simulations         !
!   Copyright (C) 2000 - 2010  CP2K developers group                          !
!-----------------------------------------------------------------------------!

! *****************************************************************************
!> \brief Routines to store the history of optimized wfn when the KG_GPW method is used.
!>      The method requires that the wfn are stored in molecular blocks and not as full matrix
!> \par History
!>      none
!> \author MI (01.04.2005)
! *****************************************************************************
MODULE kg_gpw_wf_history

  USE cp_dbcsr_operations,             ONLY: cp_dbcsr_deallocate_matrix,&
                                             cp_dbcsr_from_sm,&
                                             sm_from_dbcsr
  USE cp_dbcsr_types,                  ONLY: cp_dbcsr_p_type,&
                                             cp_dbcsr_type
  USE distribution_2d_types,           ONLY: distribution_2d_type
  USE kg_gpw_fm_mol_methods,           ONLY: calculate_mol_density_matrix,&
                                             mol_make_basis,&
                                             multiply_sparse_mol_mo
  USE kg_gpw_fm_mol_types,             ONLY: fm_mol_blocks_type,&
                                             get_fm_mol_block,&
                                             get_kg_fm_mol_set,&
                                             get_mol_mo_set,&
                                             kg_fm_mol_set_type,&
                                             mol_mo_set_p_type,&
                                             mol_mo_set_type
  USE kg_gpw_fm_mol_utils,             ONLY: copy_sparse2mol_block
  USE kinds,                           ONLY: dp
  USE sparse_matrix_types,             ONLY: allocate_matrix_set,&
                                             deallocate_matrix,&
                                             deallocate_matrix_set,&
                                             real_matrix_p_type,&
                                             real_matrix_type
#include "cp_common_uses.h"

  IMPLICIT NONE

  PRIVATE

  CHARACTER(len=*), PARAMETER, PRIVATE :: moduleN = 'kg_gpw_wf_history'

! *** Public subroutines ***

  PUBLIC :: kg_gpw_fm_mol_linear, kg_gpw_fm_mol_ps, kg_gpw_prev_wf

CONTAINS

! *****************************************************************************
!> \brief The linear extrapolation of the mos is calculated by using the two
!>      previous steps.
!>       set_a%...%mos =  alpha*set_a%...%mos + beta*set_b%...%mos
!>      If required the density is also calculated
!> \param alpha first multiplicative constant
!> \param set_a contains the initial mos and the output mos (inout)
!> \param beta second multiplicative constant
!> \param set_b contains the mos of a previous step that will be added (in)
!> \param matrix_p density matrix in real sparse matrix format, optional
!> \param matrix_s overlapmatrix in real sparse matrix format, optional
!> \param erro r
!> \note
!>      matrix_p and matrix_s are given as input only if the density has to be calculated
!>      i.e. when all the terms have been added to the final mos
!> \author MI
! *****************************************************************************
  SUBROUTINE kg_gpw_fm_mol_linear(alpha,set_a,beta,set_b,matrix_p_b,matrix_s_b,distribution_2d,error)

    REAL(dp), INTENT(IN), OPTIONAL           :: alpha
    TYPE(kg_fm_mol_set_type), DIMENSION(:), &
      POINTER                                :: set_a
    REAL(dp), INTENT(IN), OPTIONAL           :: beta
    TYPE(kg_fm_mol_set_type), DIMENSION(:), &
      POINTER                                :: set_b
    TYPE(cp_dbcsr_p_type), DIMENSION(:), &
      OPTIONAL, POINTER                      :: matrix_p_b, matrix_s_b
    TYPE(distribution_2d_type), POINTER      :: distribution_2d
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'kg_gpw_fm_mol_linear', &
      routineP = moduleN//':'//routineN

    INTEGER :: imol, imolecule_kind, ispin, istat, n_ao, n_mo(2), natom, &
      nmo_a, nmo_b, nmo_max, nmol, nmolecule_kind, nspins
    LOGICAL                                  :: calculate_density, failure
    REAL(dp)                                 :: my_alpha, my_beta
    REAL(dp), DIMENSION(:, :), POINTER       :: mo_coeff_a, mo_coeff_b, s_mo
    TYPE(fm_mol_blocks_type), DIMENSION(:), &
      POINTER                                :: blocks_a, blocks_b
    TYPE(fm_mol_blocks_type), POINTER        :: mol_block_a, mol_block_b
    TYPE(kg_fm_mol_set_type), POINTER        :: fm_mol_a, fm_mol_b
    TYPE(mol_mo_set_p_type), DIMENSION(:), &
      POINTER                                :: mos_a, mos_b
    TYPE(mol_mo_set_type), POINTER           :: mo_set_a, mo_set_b
    TYPE(real_matrix_p_type), DIMENSION(:), &
      POINTER                                :: matrix_p, matrix_s

!
!

    failure = .FALSE.
    calculate_density = .FALSE.
    IF(PRESENT(matrix_p_b) .AND. PRESENT(matrix_s_b)) calculate_density = .TRUE.

    NULLIFY(matrix_s)!sm->dbcsr
    IF(PRESENT(matrix_s_b)) THEN!sm->dbcsr
       CALL allocate_matrix_set( matrix_s, SIZE(matrix_s_b), error )!sm->dbcsr
       DO ispin=1,SIZE(matrix_s)!sm->dbcsr
          CALL sm_from_dbcsr(matrix_s(ispin)%matrix, matrix_s_b(ispin)%matrix, &
               distribution_2d,error)!sm->dbcsr
       ENDDO!sm->dbcsr
    ENDIF

    NULLIFY(matrix_p)!sm->dbcsr
    IF(PRESENT(matrix_p_b)) THEN!sm->dbcsr
    CALL allocate_matrix_set( matrix_p, SIZE(matrix_p_b), error )!sm->dbcsr
    DO ispin=1,SIZE(matrix_p)!sm->dbcsr
       CALL sm_from_dbcsr(matrix_p(ispin)%matrix, matrix_p_b(ispin)%matrix, &
            distribution_2d,error)!sm->dbcsr
    ENDDO!sm->dbcsr
    ENDIF

    nmolecule_kind=SIZE(set_a,1)
    CPPostcondition(SIZE(set_b,1)==nmolecule_kind,cp_failure_level,routineP,error,failure)

    NULLIFY(mo_coeff_a, mo_coeff_b,s_mo)
    my_alpha = 1.0_dp
    my_beta  = 1.0_dp
    IF (PRESENT(alpha)) my_alpha=alpha
    IF (PRESENT(beta)) my_beta=beta

    DO imolecule_kind = 1, nmolecule_kind
      NULLIFY(fm_mol_a,fm_mol_b, blocks_a, blocks_b)
      fm_mol_a => set_a(imolecule_kind)
      fm_mol_b => set_b(imolecule_kind)

      CALL get_kg_fm_mol_set(kg_fm_mol_set=fm_mol_a, &
           nmolecule_local=nmol, natom=natom,&
           n_ao = n_ao, n_mo = n_mo, fm_mol_blocks=blocks_a)
      IF(nmol>0) THEN
        CALL get_kg_fm_mol_set(kg_fm_mol_set=fm_mol_b,fm_mol_blocks=blocks_b)
        CPPostcondition(SIZE(blocks_b,1)==nmol,cp_failure_level,routineP,error,failure)
        nspins = 1
        IF(n_mo(2) /= 0) nspins = 2

        IF(calculate_density) THEN
          nmo_max = MAX(n_mo(1),n_mo(2))
          ALLOCATE(s_mo(n_ao,nmo_max), STAT = istat)
          CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
        END IF

        DO imol = 1, nmol
          NULLIFY(mol_block_a, mos_a, mol_block_b, mos_b)
          mol_block_a => blocks_a(imol)
          mol_block_b => blocks_b(imol)
          CALL get_fm_mol_block(fm_mol_block = mol_block_a,&
               mos = mos_a )
          CALL get_fm_mol_block(fm_mol_block = mol_block_b,&
               mos = mos_b)

          DO ispin=1,nspins
            NULLIFY(mo_set_a, mo_set_b)
            mo_set_a => mos_a(ispin)%mo_set
            mo_set_b => mos_b(ispin)%mo_set
  
            CALL get_mol_mo_set(mo_set=mo_set_a, mo = mo_coeff_a,&
                 nmo=nmo_a)
            CALL get_mol_mo_set(mo_set=mo_set_b, mo = mo_coeff_b,&
                 nmo=nmo_b)

            CPPostcondition(nmo_a==nmo_b,cp_failure_level,routineP,error,failure)

            IF(my_alpha == 1.0_dp) THEN
              CALL daxpy(SIZE(mo_coeff_a,1)*SIZE(mo_coeff_a,2),&
                         my_beta, mo_coeff_b(1,1),1,&
                         mo_coeff_a(1,1),1)
            ELSE
              CALL dscal(SIZE(mo_coeff_a,1)*SIZE(mo_coeff_a,2),&
                         my_alpha,mo_coeff_a(1,1),1)
              CALL daxpy(SIZE(mo_coeff_a,1)*SIZE(mo_coeff_a,2),&
                         my_beta, mo_coeff_b(1,1),1,&
                         mo_coeff_a(1,1),1)
            END IF

            IF(calculate_density) THEN

              ! Multiply Overlap matric and Coefficient matrix
              CALL  multiply_sparse_mol_mo(matrix_s(1)%matrix,mol_block_a,&
                                           natom,mo_coeff_a,nmo_a,s_mo,&
distribution_2d=distribution_2d,error=error)
              ! orthogonalize the molecular orbitals
              CALL mol_make_basis(mo_coeff_a, nmo_a, matrix_ortho=s_mo, otype="SV",error=error)

              CALL calculate_mol_density_matrix(mo_set_a, mol_block_a, &
                   natom, matrix_p(ispin)%matrix, distribution_2d=distribution_2d,error=error)
           END IF
          END DO  ! ispin

        END DO  ! imol

        IF(calculate_density) THEN
          DEALLOCATE(s_mo, STAT=istat)
          CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
        END IF

      END IF  ! nmol
    END DO  ! imolecule_kind

    IF(PRESENT(matrix_s_b))CALL deallocate_matrix_set( matrix_s, error )!sm->dbcsr

    IF(PRESENT(matrix_p_b))THEN
    DO ispin=1,SIZE(matrix_p)!sm->dbcsr
       CALL cp_dbcsr_deallocate_matrix(matrix_p_b(ispin)%matrix,error)
       ALLOCATE(matrix_p_b(ispin)%matrix)
       CALL cp_dbcsr_from_sm(matrix_p_b(ispin)%matrix, matrix_p(ispin)%matrix, error, distribution_2d)!sm->dbcsr
    ENDDO!sm->dbcsr
    CALL deallocate_matrix_set( matrix_p, error )!sm->dbcsr
    ENDIF

  END SUBROUTINE  kg_gpw_fm_mol_linear

! *****************************************************************************
!> \brief The ps extrapolation of the mos is calculated by using a
!>      number of previous steps given by the order of the extrapolation (nvec)
!>      At each i-th step in the loop to compose the final mos, the very last
!>      wfn are multiplied by the wfn of the (nlast - i)th step
!>      C(next) = C(next) - beta *  C(i)[C(i)S(i)C(last)]
!>      If required the density is also calculated
!> \param set_a contains the output mos that wiill be used as guess in the next step
!> \param beta second multiplicative constant
!> \param set_b contains the mos of the (nlast - i)th step, to be multiplied by the last mos
!> \param ov_b contains the overlap matrix of the (nlast - i)th
!> \param set_last contains the mos of the step
!> \param matrix_p density matrix in real sparse matrix format, optional
!> \param matrix_s overlapmatrix in real sparse matrix format, optional
!> \param erro r
!> \note
!>      matrix_p and matrix_s are given as input only if the density has to be calculated
!>      i.e. when all the terms have been added to the final mos
!> \author MI
! *****************************************************************************
  SUBROUTINE kg_gpw_fm_mol_ps(set_a, beta, set_b, ov_b_b, set_last, &
                              matrix_p_b, matrix_s_b,  distribution_2d, error )
    TYPE(kg_fm_mol_set_type), DIMENSION(:), &
      POINTER                                :: set_a
    REAL(dp), INTENT(IN), OPTIONAL           :: beta
    TYPE(kg_fm_mol_set_type), DIMENSION(:), &
      POINTER                                :: set_b
    TYPE(cp_dbcsr_type), POINTER             :: ov_b_b
    TYPE(kg_fm_mol_set_type), DIMENSION(:), &
      POINTER                                :: set_last
    TYPE(cp_dbcsr_p_type), DIMENSION(:), &
      OPTIONAL, POINTER                      :: matrix_p_b, matrix_s_b
    TYPE(distribution_2d_type), POINTER      :: distribution_2d
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'kg_gpw_fm_mol_ps', &
      routineP = moduleN//':'//routineN

    INTEGER :: icol, imol, imolecule_kind, irow, ispin, istat, ldcsc, ldsc, &
      n_ao, n_mo(2), natom, nmo_a, nmo_b, nmo_l, nmo_max, nmol, &
      nmolecule_kind, nspins
    LOGICAL                                  :: calculate_density, failure
    REAL(dp)                                 :: my_beta
    REAL(dp), DIMENSION(:, :), POINTER       :: mo_coeff_a, mo_coeff_b, &
                                                mo_coeff_l, mol_csc, &
                                                ov_b_mol, s_mo, sc_tmp
    TYPE(fm_mol_blocks_type), DIMENSION(:), &
      POINTER                                :: blocks_a, blocks_b, blocks_l
    TYPE(fm_mol_blocks_type), POINTER        :: mol_block_a, mol_block_b, &
                                                mol_block_l
    TYPE(kg_fm_mol_set_type), POINTER        :: fm_mol_a, fm_mol_b, fm_mol_l
    TYPE(mol_mo_set_p_type), DIMENSION(:), &
      POINTER                                :: mos_a, mos_b, mos_l
    TYPE(mol_mo_set_type), POINTER           :: mo_set_a, mo_set_b, mo_set_l
    TYPE(real_matrix_p_type), DIMENSION(:), &
      POINTER                                :: matrix_p, matrix_s
    TYPE(real_matrix_type), POINTER          :: ov_b

!
!

    NULLIFY(ov_b)!sm->dbcsr
    CALL sm_from_dbcsr(ov_b, ov_b_b, distribution_2d,error)!sm->dbcsr

    failure = .FALSE.
    calculate_density = .FALSE.
    IF(PRESENT(matrix_p_b) .AND. PRESENT(matrix_s_b)) calculate_density = .TRUE.

    NULLIFY(matrix_s)!sm->dbcsr
    IF(PRESENT(matrix_s_b)) THEN!sm->dbcsr
       CALL allocate_matrix_set( matrix_s, SIZE(matrix_s_b), error )!sm->dbcsr
       DO ispin=1,SIZE(matrix_s)!sm->dbcsr
          CALL sm_from_dbcsr(matrix_s(ispin)%matrix, matrix_s_b(ispin)%matrix, &
               distribution_2d,error)!sm->dbcsr
       ENDDO!sm->dbcsr
    ENDIF

    NULLIFY(matrix_p)!sm->dbcsr
IF(PRESENT(matrix_p_b)) THEN!sm->dbcsr
    CALL allocate_matrix_set( matrix_p, SIZE(matrix_p_b), error )!sm->dbcsr
    DO ispin=1,SIZE(matrix_p)!sm->dbcsr
       CALL sm_from_dbcsr(matrix_p(ispin)%matrix, matrix_p_b(ispin)%matrix, &
            distribution_2d,error)!sm->dbcsr
    ENDDO!sm->dbcsr
ENDIF

    nmolecule_kind=SIZE(set_a,1)
    CPPostcondition(SIZE(set_b,1)==nmolecule_kind,cp_failure_level,routineP,error,failure)
    CPPostcondition(SIZE(set_last,1)==nmolecule_kind,cp_failure_level,routineP,error,failure)

    NULLIFY( mo_coeff_a, mo_coeff_b,mo_coeff_l, mol_csc, ov_b_mol, s_mo, sc_tmp)

    my_beta  = 1.0_dp
    IF (PRESENT(beta)) my_beta=beta

    DO imolecule_kind = 1, nmolecule_kind
      NULLIFY(fm_mol_a,fm_mol_b, fm_mol_l, blocks_a, blocks_b, blocks_l)
      fm_mol_a => set_a(imolecule_kind)
      fm_mol_b => set_b(imolecule_kind)
      fm_mol_l => set_last(imolecule_kind)

      CALL get_kg_fm_mol_set(kg_fm_mol_set=fm_mol_a,&
           nmolecule_local=nmol, natom=natom,&
           n_ao = n_ao, n_mo = n_mo, fm_mol_blocks=blocks_a)
      IF(nmol>0) THEN
        CALL get_kg_fm_mol_set(kg_fm_mol_set=fm_mol_b,fm_mol_blocks=blocks_b)
        CALL get_kg_fm_mol_set(kg_fm_mol_set=fm_mol_l,fm_mol_blocks=blocks_l)
        CPPostcondition(SIZE(blocks_b,1)==nmol,cp_failure_level,routineP,error,failure)
        CPPostcondition(SIZE(blocks_l,1)==nmol,cp_failure_level,routineP,error,failure)
        nspins = 1
        IF(n_mo(2) /= 0) nspins = 2

        ! Allocate the array to store the overlap matrix of step b
        ALLOCATE(ov_b_mol(n_ao,n_ao),STAT=istat)
        CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
        ! Allocate the array CSC
        ldcsc = MAX(n_mo(1),n_mo(2))
        ALLOCATE(mol_csc(ldcsc,ldcsc), STAT= istat)
        CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
        IF(calculate_density) THEN
          nmo_max = MAX(n_mo(1),n_mo(2))
          ALLOCATE(s_mo(n_ao,nmo_max), STAT = istat)
          CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
        END IF
  
        DO imol = 1, nmol
          NULLIFY(mol_block_a, mos_a, mol_block_b, mos_b, mol_block_l, mos_l)
          mol_block_a => blocks_a(imol)
          mol_block_b => blocks_b(imol)
          mol_block_l => blocks_l(imol)
          CALL get_fm_mol_block(fm_mol_block = mol_block_a,&
               mos = mos_a, work = sc_tmp )
          CPPrecondition(ASSOCIATED(sc_tmp),cp_failure_level,routineP,error,failure)
          ldsc = SIZE(sc_tmp,1)
          CPPrecondition(ldsc>=n_ao,cp_failure_level,routineP,error,failure)
          CALL get_fm_mol_block(fm_mol_block = mol_block_b,&
               mos = mos_b)
          CALL get_fm_mol_block(fm_mol_block = mol_block_l,&
               mos = mos_l)
  
          ! Copy the right block of ov_b into the temporary space
          CALL copy_sparse2mol_block(matrix_sm=ov_b, mol_block=mol_block_b,&
               matrix_mol_block=ov_b_mol, natom=natom, nrow=n_ao, ncol=n_ao, error=error)

          DO ispin=1,nspins
            NULLIFY(mo_set_a, mo_set_b, mo_set_l)
            NULLIFY(mo_coeff_a, mo_coeff_b, mo_coeff_l)
  
            mo_set_a => mos_a(ispin)%mo_set
            mo_set_b => mos_b(ispin)%mo_set
            mo_set_l => mos_l(ispin)%mo_set

            CALL get_mol_mo_set(mo_set=mo_set_a, mo = mo_coeff_a,&
                 nmo=nmo_a)
            CALL get_mol_mo_set(mo_set=mo_set_b, mo = mo_coeff_b,&
                 nmo=nmo_b)
            CALL get_mol_mo_set(mo_set=mo_set_l, mo = mo_coeff_l,&
                 nmo=nmo_l)

            CPPostcondition(nmo_a==nmo_b,cp_failure_level,routineP,error,failure)
            CPPostcondition(nmo_a==nmo_l,cp_failure_level,routineP,error,failure)

            CALL DGEMM('N','N',n_ao,nmo_l,n_ao,1.0_dp,ov_b_mol(1,1),n_ao,&
                 mo_coeff_l(1,1),SIZE(mo_coeff_l,1),0.0_dp,sc_tmp(1,1),ldsc)
            CALL DGEMM('T','N',nmo_b,nmo_b,n_ao,1.0_dp,mo_coeff_b(1,1),n_ao,&
                 sc_tmp(1,1),ldsc,0.0_dp,mol_csc(1,1),ldcsc)
            CALL DGEMM('N','N',n_ao,nmo_b,nmo_b,1.0_dp,mo_coeff_b(1,1),n_ao,&
                 mol_csc(1,1),ldcsc,0.0_dp,sc_tmp(1,1),ldsc)
            DO icol = 1,nmo_a
              DO irow = 1,n_ao
                mo_coeff_a(irow,icol) = mo_coeff_a(irow,icol) + beta*sc_tmp(irow,icol)
              END DO
            END DO

            ! Calculate the density matrix with the completed new guess for the mos
            IF(calculate_density) THEN
              ! Multiply Overlap matric and Coefficient matrix
              CALL  multiply_sparse_mol_mo(matrix_s(1)%matrix,mol_block_a,&
                                           natom,mo_coeff_a,nmo_a,s_mo,&
                                           distribution_2d=distribution_2d,error=error)
              ! orthogonalize the molecular orbitals
              CALL mol_make_basis(mo_coeff_a, nmo_a, matrix_ortho=s_mo, otype="SV",error=error)

              CALL calculate_mol_density_matrix(mo_set_a, mol_block_a, &
                   natom, matrix_p(ispin)%matrix, distribution_2d=distribution_2d,error=error)
            END IF

          END DO  ! ispin

        END DO ! imol
        IF(calculate_density) THEN
          DEALLOCATE(s_mo, STAT=istat)
          CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
        END IF

        DEALLOCATE(mol_csc,STAT = istat)
        CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
  
        DEALLOCATE(ov_b_mol,STAT = istat)
        CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)

      END IF  ! nmol
    END DO  ! imolecule_kind

    CALL deallocate_matrix(ov_b,error=error)!sm->dbcsr
    IF(PRESENT(matrix_s_b))CALL deallocate_matrix_set( matrix_s, error )!sm->dbcsr

    IF(PRESENT(matrix_p_b))THEN
    DO ispin=1,SIZE(matrix_p)!sm->dbcsr
       CALL cp_dbcsr_deallocate_matrix(matrix_p_b(ispin)%matrix,error)
       ALLOCATE(matrix_p_b(ispin)%matrix)
       CALL cp_dbcsr_from_sm(matrix_p_b(ispin)%matrix, matrix_p(ispin)%matrix, error)!sm->dbcsr
    ENDDO!sm->dbcsr
    CALL deallocate_matrix_set( matrix_p, error )!sm->dbcsr
 ENDIF

  END SUBROUTINE  kg_gpw_fm_mol_ps

! *****************************************************************************
!> \brief Calculate the density from the previous wfn given in molecular blocks
!> \param kg_fm_mol containes the blocks of mos
!> \param matrix_p density matrix in real sparse matrix format
!> \param matrix_s overlapmatrix in real sparse matrix format
!> \param erro r
!> \author MI
! *****************************************************************************
  SUBROUTINE kg_gpw_prev_wf(kg_fm_mol_set,matrix_p_b,matrix_s_b,distribution_2d,error)

    TYPE(kg_fm_mol_set_type), DIMENSION(:), &
      POINTER                                :: kg_fm_mol_set
    TYPE(cp_dbcsr_p_type), DIMENSION(:), &
      POINTER                                :: matrix_p_b, matrix_s_b
    TYPE(distribution_2d_type), POINTER      :: distribution_2d
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(LEN=*), PARAMETER :: routineN = 'kg_gpw_prev_wf', &
      routineP = moduleN//':'//routineN

    INTEGER :: imol, imolecule_kind, ispin, istat, n_ao, nat_mol, nmo, &
      nmo_kind(2), nmo_max, nmol_local, nmolecule_kind, nspins
    LOGICAL                                  :: failure
    REAL(dp), DIMENSION(:, :), POINTER       :: mo_coeff, s_mo
    TYPE(fm_mol_blocks_type), DIMENSION(:), &
      POINTER                                :: fm_mol_blocks
    TYPE(fm_mol_blocks_type), POINTER        :: mol_block
    TYPE(kg_fm_mol_set_type), POINTER        :: fm_mol_set
    TYPE(mol_mo_set_p_type), DIMENSION(:), &
      POINTER                                :: mos
    TYPE(mol_mo_set_type), POINTER           :: mo_set
    TYPE(real_matrix_p_type), DIMENSION(:), &
      POINTER                                :: matrix_p, matrix_s

    NULLIFY(matrix_s)!sm->dbcsr
    CALL allocate_matrix_set( matrix_s, SIZE(matrix_s_b), error )!sm->dbcsr
    DO ispin=1,SIZE(matrix_s)!sm->dbcsr
       CALL sm_from_dbcsr(matrix_s(ispin)%matrix, matrix_s_b(ispin)%matrix, &
            distribution_2d,error)!sm->dbcsr
    ENDDO!sm->dbcsr

    NULLIFY(matrix_p)!sm->dbcsr
    CALL allocate_matrix_set( matrix_p, SIZE(matrix_p_b), error )!sm->dbcsr
    DO ispin=1,SIZE(matrix_p)!sm->dbcsr
       CALL sm_from_dbcsr(matrix_p(ispin)%matrix, matrix_p_b(ispin)%matrix, &
            distribution_2d,error)!sm->dbcsr
    ENDDO!sm->dbcsr

    failure = .FALSE.
    nmolecule_kind = SIZE(kg_fm_mol_set)
    NULLIFY(mo_coeff, s_mo)

    DO imolecule_kind = 1,nmolecule_kind
      NULLIFY(fm_mol_set,fm_mol_blocks)
      fm_mol_set => kg_fm_mol_set(imolecule_kind)
      CALL get_kg_fm_mol_set(kg_fm_mol_set=fm_mol_set, &
                             natom = nat_mol, &
                             n_ao = n_ao , n_mo = nmo_kind, &
                             fm_mol_blocks = fm_mol_blocks,&
                             nmolecule_local = nmol_local)
      IF(nmol_local>0) THEN
        nspins = 1
        IF(nmo_kind(2)/=0)  nspins = 2

        nmo_max = MAX(nmo_kind(1),nmo_kind(2))
        ALLOCATE(s_mo(n_ao,nmo_max), STAT = istat)
        CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)

        DO imol = 1, nmol_local
          NULLIFY(mol_block,mos)
          mol_block => fm_mol_blocks(imol)
          CALL get_fm_mol_block(fm_mol_block = mol_block,&
                                mos = mos)
          DO ispin=1,nspins
            NULLIFY(mo_set,mo_coeff)
            mo_set => mos(ispin)%mo_set
            CALL get_mol_mo_set(mo_set=mo_set, mo = mo_coeff,&
                                nmo=nmo)

            ! Multiply Overlap matric and Coefficient matrix
            CALL  multiply_sparse_mol_mo(matrix_s(1)%matrix,mol_block,&
                                         nat_mol,mo_coeff,nmo,s_mo,&
                                         distribution_2d=distribution_2d,error=error)
            ! orthogonalize the molecular orbitals
            CALL mol_make_basis(mo_coeff, nmo, matrix_ortho=s_mo, otype="SV",error=error)

            CALL calculate_mol_density_matrix(mo_set,mol_block,&
                                       nat_mol,matrix_p(ispin)%matrix,&
                                       distribution_2d=distribution_2d,error=error)
          ENDDO  ! ispin
        END DO  ! imol

        DEALLOCATE(s_mo,STAT = istat)
        CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
      END IF  ! nmol_local
    END DO  ! imolecule_kind

    CALL deallocate_matrix_set( matrix_s, error )!sm->dbcsr

    DO ispin=1,SIZE(matrix_p)!sm->dbcsr
       CALL cp_dbcsr_deallocate_matrix(matrix_p_b(ispin)%matrix,error)
       ALLOCATE(matrix_p_b(ispin)%matrix)
       CALL cp_dbcsr_from_sm(matrix_p_b(ispin)%matrix, matrix_p(ispin)%matrix, error,distribution_2d)!sm->dbcsr
    ENDDO!sm->dbcsr
    CALL deallocate_matrix_set( matrix_p, error )!sm->dbcsr

  END SUBROUTINE  kg_gpw_prev_wf

END MODULE kg_gpw_wf_history
