(in-package :contextl)

(defun ensure-layered-function
       (name
        &rest initargs
        &key (lambda-list () lambda-list-p)
        (argument-precedence-order (required-args lambda-list))
        (generic-function-class 'layered-function)
        &allow-other-keys)
  (declare (dynamic-extent initargs))
  (unless lambda-list-p
    (error "The layered function ~S must be initialized with a lambda list." name))
  (let ((gf (let ((layer-arg (gensym "LAYER-ARG-")))
              (apply #'ensure-generic-function
                     (get-layered-function-definer-name name)
		     #-allegro :generic-function-class
                     #-allegro generic-function-class
                     :argument-precedence-order
                     `(,@argument-precedence-order ,layer-arg)
                     :lambda-list
                     `(,layer-arg ,@lambda-list)
                     initargs))))
    (setf (fdefinition name)
          (compile nil `(lambda (&rest rest)
                          (declare (dynamic-extent rest)
                                   (optimize (speed 3) (debug 0) (safety 0)
                                             (compilation-speed 0)))
                          (apply (the function ,gf)
                                 (layer-context-prototype *active-context*)
                                 rest))))
    (bind-layered-function-names name)
    gf))

(defun layered-function-definer (name)
  (fdefinition (get-layered-function-definer-name name)))

(defgeneric layered-function-argument-precedence-order (function)
  (:method ((function layered-function)) (butlast (generic-function-argument-precedence-order function))))

(defgeneric layered-function-lambda-list (function)
  (:method ((function layered-function)) (rest (generic-function-lambda-list function))))

(defun lfmakunbound (name)
  (fmakunbound (get-layered-function-definer-name name))
  (fmakunbound name))

(defun ensure-layered-method
       (layered-function-name
        lambda-expression 
        &key
        #-(or allegro clisp cmu ecl mcl openmcl) 
        (method-class
         (generic-function-method-class
          (fdefinition (get-layered-function-definer-name layered-function-name))))
        (in-layer 't)
        (qualifiers ())
        (lambda-list (cadr lambda-expression))
        (specializers (required-args lambda-list (constantly (find-class 't)))))
  (let ((layer-arg (gensym "LAYER-ARG-")))
    (destructuring-bind
        (lambda (&rest args) &body body)
        lambda-expression
      (unless (eq lambda 'lambda)
        (error "Incorrect lambda expression: ~S." lambda-expression))
      (ensure-method (fdefinition (get-layered-function-definer-name layered-function-name))
                     `(lambda (,layer-arg ,@args) ,@body)
                     #-(or allegro clisp cmu ecl mcl openmcl) :method-class
                     #-(or allegro clisp cmu ecl mcl openmcl) method-class
                     :qualifiers qualifiers
                     :lambda-list `(,layer-arg ,@lambda-list)
                     :specializers (cons (find-layer-class in-layer) specializers)))))

(defgeneric layered-method-lambda-list (method)
  (:method ((method layered-method)) (rest (method-lambda-list method))))

(defgeneric layered-method-specializers (method)
  (:method ((method layered-method)) (rest (method-specializers method))))

(defgeneric layered-method-layer (method)
  (:method ((method layered-method)) (find-layer (first (method-specializers method)))))

(defmethod print-object ((object layered-method) stream)
  (print-unreadable-object (object stream :type t :identity t)
    (format stream "~A ~A ~S ~A"
            (when (method-generic-function object)
              (get-layered-function-caller-name
               (generic-function-name
                (method-generic-function object))))
            (layer-name (layered-method-layer object))
            (method-qualifiers object)
            (mapcar #'class-name (layered-method-specializers object)))))
